/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles.shapes;

import j3d.examples.appearance.texture.noise.ImprovedNoise;

import java.awt.image.BufferedImage;
import javax.media.j3d.*;
import javax.vecmath.*;
import java.io.*;
import javax.imageio.*;

/**
 * A oriented shape with a cloudy texture.  The texture is created through
 * a combination of a spherical implicit surface with a cubic density 
 * function with Perlin noise.  The contribution of the implicit
 * surface density versus noise is controled by the blend factor.  The
 * type of noise is controlled with the noise mode.
 */
// TODO: Allow the texture on the CloudPuff class to change during run time.
public class CloudPuff extends ImplicitSurface {
	protected static float NOISE_ZOOM = 23.3f;
	private static Geometry sharedGeometry;
	// TODO: Cloudpuff geometry should be shared by factory, not class?
	public static boolean shareGeometry = true;
	public static final int CLOUDS = 0;
	public static final int FLAMES = 1;
	public static final int SMOKE = 2;
	private int noiseMode;
	private double densityExponent;
	
	private static Geometry getSharedGeometry() {
		return sharedGeometry;
	}

	private static void setSharedGeometry(Geometry geometry) {
		sharedGeometry = geometry;
	}
	
	private double blend;
	private double centerX = 0, centerY = 0;
	private Color3f emissiveColor;
	private double innerRadius = 0, innerRadius2 = 0;
	private int octaves;
	private double outterRadius = 0,
		outterRadius2 = 0,
		outterRadius4 = 0,
		outterRadius6 = 0;

	public CloudPuff() {
		this((float) (5 + 7 * Math.random()));
	}

	public CloudPuff(float aRadius) {
		this(aRadius, new Color3f(1, 1, 1));
	}

	public CloudPuff(float aRadius, Color3f anAmbientColor) {
		this(aRadius, anAmbientColor, new Color3f(0.1f,0.0f,0.0f), 0.4, 6); 
	}

	public CloudPuff(
		float aRadius,
		Color3f anAmbientColor,
		Color3f anEmissiveColor,
		double aBlendFactor,
		int octaves) {
		this(aRadius, anAmbientColor, anEmissiveColor, aBlendFactor, octaves, CLOUDS);
	}
	
	public CloudPuff(
		float aRadius,
		Color3f anAmbientColor,
		Color3f anEmissiveColor,
		double aBlendFactor,
		int octaves,
		int aMode) {
		/* The number of octaves and image size are directly related.  The
		 * number of octaves determines the number of noise iterations to
		 * compute the turbulance value.  There is no reason to calculate 
		 * turbulance effects that are sub-pixel within the image so the 
		 * image size is 2 to the power of the octaves.
		 */
		super(aRadius, anAmbientColor, 1 << octaves);  
		emissiveColor = anEmissiveColor;
		blend = aBlendFactor;
		setOctaves(octaves);
		noiseMode = aMode;
		switch(aMode){
			case CLOUDS:
				densityExponent = 0.2;
				break;
			case FLAMES:
				densityExponent = 0.5;
				break;
			case SMOKE:
				densityExponent = 0.7;
				break;
			default:
				densityExponent = 0.2;
		}
		postConstructionInitialization();
	}
	
	protected int getPreferredGeometrySize(){
		// Don't worry about detailed lighting effects
		// with clouds.  Make the preferred geometry
		// a single quad.
		return 1;
	}
	
	protected Geometry createGeometry() {
		if(shareGeometry){
			if (getSharedGeometry() == null) {
				Geometry g = super.createGeometry();
				setSharedGeometry(g);
			}
			return getSharedGeometry();
		}
		else {
			return super.createGeometry();
		}
		
	}

	protected void customizeAppearance(Appearance anAppearance) {
		Material material = new Material();
		/* Using levels of gray for the ambient, diffuse and specular
		 * colors in combination with texture modulation.  The RGB
		 * of the texture will be multiplied by these colors.
		 */
		material.setAmbientColor(new Color3f(0.8f, 0.8f, 0.8f));
		material.setDiffuseColor(new Color3f(0.5f, 0.5f, 0.5f));
		material.setSpecularColor(new Color3f(0.05f, 0.05f, 0.05f));
		material.setEmissiveColor(emissiveColor);
		anAppearance.setMaterial(material);
		/*
		 * Set up the texture mode, alpha mode and alpha combine
		 * function to simplify the changing the of the transparency
		 * of this object.  Java3D will modulate (multiply) the object
		 * alpha (1 - transparency) with the alpha channel of the 
		 * texture when rendering.  The transparency of the object 
		 * then can control the overall transparency of the texture.
		 * As the object ages, simply changing the transparency will
		 * affect the entire texture.
		 */
		TextureAttributes textureAttributes = new TextureAttributes();

		/* Through experimentation with J3D 1.3.1, I found that MODULATE 
		 * resulted in very transparent clouds even with a white ambient 
		 * material color.  Using COMBINE and COMBINE_MODULATE produced 
		 * clouds that are more sensitive to ambient color producing better
		 * looking clouds.  After moving to J3D 1.3.2, this is apparently
		 * an issue again.  The resulting clouds are again too transparent
		 * for my tastes.  The article and the screen shots are from 
		 * J3D 1.3.1 DX on Windows XP. 
		 */
		
		// textureAttributes.setTextureMode(TextureAttributes.MODULATE);
		textureAttributes.setTextureMode(TextureAttributes.COMBINE);
		textureAttributes.setCombineAlphaMode(
			TextureAttributes.COMBINE_MODULATE);
		textureAttributes.setCombineAlphaFunction(
			0,
			TextureAttributes.COMBINE_SRC_ALPHA);
		anAppearance.setTextureAttributes(textureAttributes);

	}

	private double density(double x, double y, double z) {
		// Implicit function to determine density at a point
		double density = 0;
		double radius2 =
			(x - getCenterX()) * (x - getCenterX())
				+ (y - getCenterY()) * (y - getCenterY())
				+ z * z;
		if (radius2 > getOutterRadius2()) {
			density = 0.0;
		} else if (radius2 < getInnerRadius2()) {
			density = 1.0;
		} else {
			// Wyvill's standard cubic function (Wyvill, McPheeters, and Wyvill 1986)
			double radius4 = radius2 * radius2;
			double radius6 = radius4 * radius2;
			density =
				1
					- (4 / 9) * (radius6 / getOutterRadius6())
					+ (17 / 9) * (radius4 / getOutterRadius4())
					- (22 / 9) * (radius2 / getOutterRadius2());
		}

		return density;
		//return 1.0;
	}

	protected void generateImage(BufferedImage bi) {
		/*
		 * This algorithm is an adaptation of the volumetric cloud
		 * modeling system described in "Texturing & Modeling -
		 * A Procedural Approach", Third Edition, Ebert, et. al.
		 * Chapter 9 - Volumetric Cloud Modeling with Implict Functions,
		 * by David S. Ebert.  This implementation is simplified to ignore
		 * the accumulation of densities within the implicit volume and
		 * makes no attempt to emulate high-albedo illumination for
		 * truely realistic lighting effects.  Instead, an emissive
		 * color can be added (to the material) for added effect.
		 * 
		 * Some cloud rendering approaches use 'splatting' to provided 
		 * smoothed densities across multiple pixels.  This 
		 * implementation does not bother with splatting in favor of 
		 * allowing the Perlin noise (turbulance) to do the smoothing.
		 */
		double z = Math.random();
		for (int column = 0; column < getImageSize(); column++) {
			double x = column / NOISE_ZOOM;
			for (int row = 0; row < getImageSize(); row++) {
				double y = row / NOISE_ZOOM;
				double density = 0;
				density = density(x, y, 0.0);
				if (density > 0.0) {
					double noise;
					switch(noiseMode){
						case FLAMES:
							noise = ImprovedNoise.ridged(x, y, z, getOctaves(), 0.85, 2);
							break;
						default:
							noise = ImprovedNoise.turbulance(x, y, z, getOctaves());
					}
					double blended =
						getBlend() * density + (1 - getBlend()) * noise;
					blended = Math.pow(blended, densityExponent); //0.2 good for clouds, 0.5 for smoke, 0.7 for flames
					double weight = Math.min((255 * blended), 255);
					double opacity = weight * density;
					int alpha = (int) opacity;
					int red = (int) Math.round(weight * getColor().x);
					int green = (int) Math.round(weight * getColor().y);
					int blue = (int) Math.round(weight * getColor().z);
					if(DEBUG){
						red = (int) ((opacity/255)*Math.round(weight * getColor().x));
						green = (int) ((opacity/255)*Math.round(weight * getColor().y));
						blue = (int) ((opacity/255)*Math.round(weight * getColor().z)); 
					}
					bi.setRGB(
						column,
						row,
						(alpha << 24) + (red << 16) + (green << 8) + blue);
				}
			}
		}
		if(DEBUG){
			try {
				File outfile = new File("c:\\cloudpuff.png");
				ImageIO.write(bi, "png", outfile);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	public double getBlend() {
		return blend;
	}

	protected double getCenterX() {
		if (centerX == 0) {
			centerX = (getImageSize() / 2) / NOISE_ZOOM;
		}
		return centerX;
	}

	protected double getCenterY() {
		if (centerY == 0) {
			centerY = (getImageSize() / 2) / NOISE_ZOOM;
		}
		return centerY;
	}

	protected double getInnerRadius() {
		if (innerRadius == 0) {
			innerRadius = 0.04 * getImageSize() / NOISE_ZOOM;
		}
		return innerRadius;
	}

	protected double getInnerRadius2() {
		if (innerRadius2 == 0) {
			innerRadius2 = getInnerRadius() * getInnerRadius();
		}
		return innerRadius2;
	}

	protected int getOctaves() {
		return octaves;
	}

	protected double getOutterRadius() {
		if (outterRadius == 0) {
			outterRadius = (getImageSize() / 2) / NOISE_ZOOM;
		}
		return outterRadius;
	}
	protected double getOutterRadius2() {
		if (outterRadius2 == 0) {
			outterRadius2 = getOutterRadius() * getOutterRadius();
		}
		return outterRadius2;
	}

	protected double getOutterRadius4() {
		if (outterRadius4 == 0) {
			outterRadius4 = getOutterRadius2() * getOutterRadius2();
		}
		return outterRadius4;
	}

	protected double getOutterRadius6() {
		if (outterRadius6 == 0) {
			outterRadius6 = getOutterRadius4() * getOutterRadius2();
		}
		return outterRadius6;
	}

	/**
	 * The blend factor controls the contributions of the implicit surface
	 * and the noise.  The aBlendFactor must be in the range of [0..1].
	 * High blend values favor the implicit surface with a value of 1.0 
	 * resulting in a sphere (the implicit surface).  Lower values add more
	 * noise or roughness to the cloud.
	 * <p>
	 * @param aBlendFactor
	 */
	public void setBlend(double aBlendFactor) {
		blend = aBlendFactor;
	}

	protected void setOctaves(int i) {
		/* Seven or more octaves can create detailed cloud structures 
		 * in exchange for generation time.  Five to six octaves provides 
		 * reasonable clouds while keeping the computational time down 
		 * (especially when changing the image over time).
		 */
		octaves = i;
	}

}
