/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles.influences;
import j3d.examples.particles.emitters.*;
/**
 * A particle age alpha object produces a value, called the 
 * alpha value, between 0.0 and 1.0, inclusive. The alpha value
 * changes over the life time of a particle as specified by the 
 * parameters of the object. For specific parameter values at any
 * particular age, there is only one alpha value the alpha object 
 * will produce.  This object is very similar in concept to the 
 * Java3D Alpha object.  The Java3D Alpha object is based on time 
 * while the particle age alpha is based on the age of a particle.  
 * Not all features of the Java3D Alpha are needed or implemented 
 * in the particle age alpha object.
 *  
 * The particle age alpha object has the following phases (all 
 * expressed as a percentage of the age of a particle):
 * 
 * A. 	Phase Delay - How much age should pass before proceeding 
 * 		to the next phase.  Default is zero. 
 * B.	Increasing Alpha - How much of the particle age is used
 * 		to linearly increase the alpha value from 0 to 1.
 * C.	Alpha at One - How long the alpha value stays at 1.
 * D.	Decreasing Alpha - How much of the particle age is used
 * 		to linearly decrease the alpha value from 1 to 0. 
 * 
 * 
 * 1           +---------+
 *            /           \
 *           /             \
 *          /               \
 * 0_______/                 \
 * |   A  |  B |    C    |  D |
 * 
 * The duration of the phases is specified by a float value
 * expressing the percentage of a particle lifetime. 
 */
public class ParticleAgeAlpha {
	private float phaseDelay;
	private float increasingAlpha;
	private float alphaAtOne;
	private float decreasingAlpha;
	
	private float increasingAlphaEnd, alphaAtOneEnd, decreasingAlphaEnd;
	
	public ParticleAgeAlpha(){
		this(0, 0, 1, 0);
	}
	
	public ParticleAgeAlpha(float phaseDelay, float increasingAlpha, float alphaAtOne, float decreasingAlpha){
		float check = phaseDelay + increasingAlpha + alphaAtOne + decreasingAlpha;
		if((check > 1) || (check < 0)){
			throw new IllegalArgumentException("Phase percentages error");
		}
		this.phaseDelay= phaseDelay;
		this.increasingAlpha = increasingAlpha;
		this.alphaAtOne = alphaAtOne;
		this.decreasingAlpha = decreasingAlpha;
		if(check < 1){
			alphaAtOne = alphaAtOne + (1 - check);
		}

		increasingAlphaEnd = increasingAlpha + phaseDelay;
		alphaAtOneEnd = alphaAtOne + increasingAlphaEnd;
		decreasingAlphaEnd = decreasingAlpha + alphaAtOneEnd;
	}
	
	public float value(Particle aParticle){
		return value(aParticle.getLifeSpanPercentage());
	}
	
	public float value(float aPercentage){
		float answer = 0;
		
		if(aPercentage > alphaAtOneEnd){
			//System.out.println("decreasing: " + aPercentage);
			answer = 1 - (aPercentage - alphaAtOneEnd)/decreasingAlpha;
		}		
		else if((aPercentage <= alphaAtOneEnd)&&(aPercentage > increasingAlphaEnd)){
			//System.out.println("at one: " + aPercentage);
			answer = 1;
		} 		
		else if((aPercentage <= increasingAlphaEnd)&&(aPercentage > phaseDelay)){
			//System.out.println("increasing: " + aPercentage);
			answer = (aPercentage - phaseDelay)/increasingAlpha;
		} 		
		else if(aPercentage <= phaseDelay){
			//System.out.println("delayed: " + aPercentage);
			answer = 0;
		}
		
		return answer;
	}

}
