/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles.influences;
import javax.vecmath.*;
import javax.media.j3d.*;

import j3d.examples.particles.emitters.Particle;

/**
 * A simulation of Newton�s Law of Restitution for 
 * Instantaneous Collisions.  Assumes a mostly elastic impulse 
 * collision between the "ground" and the particle based on the 
 * coefficient of restitution in the world Y coordinate system.  
 * Also assumes a simple friction model to slow down the particle 
 * in the world X and Z planes. 
 */
// TODO: Find a way to eliminate the conflict with gravity for long living particles (continues to squirm)
public class BounceShape
	extends Shape3DInfluence
	implements IExternalInfluence {
	private static float COEFFICIENT_OF_RESTITUTION = 0.5f;
	private static float DRAG = 0.8f;
	private Bounds bounds;
	private Vector3f collisionNormal;
	private float drag;
	private float restitution;

	public BounceShape() {
		this(
			new BoundingBox(
				new Point3d(-500, -500, -500),
				new Point3d(500, 0, 500)));
	}

	public BounceShape(Bounds aBounds) {
		this(aBounds, DRAG, COEFFICIENT_OF_RESTITUTION);
	}

	public BounceShape(Bounds aBounds, float aDrag, float aRestitution) {
		bounds = aBounds;
		drag = aDrag;
		restitution = aRestitution;
		collisionNormal = new Vector3f(0, 1, 0);
	}

	/* (non-Javadoc)
	 * @see particles.influences.IExternalInfluence#apply(particles.emitters.Particle, float)
	 */
	public void apply(Particle aParticle, float dt) {
		Shape3D aShape = getShape3D(aParticle);
		BoundingBox bb = (BoundingBox) aShape.getCollisionBounds().clone();
		if (bb == null) {
			System.out.println(
				"Error in BounceShape: Shape factory failed to set collision bounds");
			return;
		}
		Transform3D t3d = getLocalToWorld(aParticle);
		bb.transform(t3d);
		if ((bb.intersect(bounds))) {
			/* 
			 * The shape has interpenetration with the ground.				
			 * Back up the full time interval and make the
			 * position a reflection off of the ground.  To be
			 * more precise, we should calculate the 'exact'
			 * interpenetration time (+/- some buffer), back up
			 * with the old velocity to the 'exact' time and
			 * then move the particle forward for the left over
			 * time difference with the new velocity calculated
			 * below.  Because the animation is dealing with a
			 * dt of ~50 milliseconds based on the particle 
			 * system manager, and we are dealing with our eyes
			 * and not a physical simulation, our approach here
			 * is good enough.  
			 */
			Vector3f velocity = aParticle.getWorldVelocity();
			float[] position = aParticle.getWorldPosition();

			float newX = position[0] + velocity.x * dt;
			float newY = position[1] - velocity.y * dt;
			float newZ = position[2] + velocity.z * dt;
			aParticle.setWorldPosition(newX, newY, newZ);

			if (aParticle.isRotating() && (Math.abs(velocity.y) > 0.1)) {
				/*
				 * This code block takes care of converting the collision
				 * into a change in the rotation of the shape.  This is 
				 * accomplished by calculating the force of the impact,
				 * converting the force into a torque, and using the torque
				 * to determine the change in angular velocity.
				 */
				// Get the radius of the induced torque.  
				Vector3f radius = getCollisionRadius(bb);
				// Calculate the magnitude of the impulse force.
				Vector3f w = aParticle.getEulerRotationRate();
				Vector3f total = new Vector3f();
				// linear + angular contribution to velocity
				total.cross(w, radius); // angular contribution
				total.add(velocity); // linear + angular
				/* 
				 * For this implementation, we are going to treat all particles as
				 * though they are solid spheres making the momement of inertia: 
				 * 	I = (2/5) * mass * r * r
				 */
				float r = radius.length(); // TODO: Let particles have a radius
				float mass = r*0.02f; // TODO: Let particles have mass
				float I = 0.4f * mass * r * r;
				// TODO: Let particles have a moment of inertia
				float j =
					getCollisionImpulse(
						aParticle,
						total,
						radius,
						mass,
						I,
						collisionNormal);
				/*
				 * Calculate the change in angular velocity based on the impulse
				 * collision.  First calculate the change in angular momentum and
				 * divide by the moment of interia to get the change in angular
				 * velocity:
				 * 		delta w = (radius <cross> J)/moment of inertia
				 * where J = j*collision normal
				 */
				Vector3f J = new Vector3f();
				J.scale(j, collisionNormal);
				Vector3f deltaW = new Vector3f();
				deltaW.cross(radius, J);
				deltaW.scale(1 / I);
				w.add(deltaW);
				// Finally adjust the angular velocity
				aParticle.setEulerRotationRate(w);
			} else {
				aParticle.stopRotating();
			}
			/* 
			 * Adjust the velocity due to the collision.  Because the
			 * ground is assumed to have infinite mass, is stationary
			 * and has a normal pointing straight up in the y direction, 
			 * the impulse affect in the linear velocity is simple.
			 * See Chris Hecker's March 1997 Game Developer Magazine
			 * article.  The lines below implement equation 5 while
			 * also simulating the drag of friction.
			 */
			velocity.x = DRAG * velocity.x;
			velocity.y = -COEFFICIENT_OF_RESTITUTION * velocity.y;
			velocity.z = DRAG * velocity.z;
			aParticle.setWorldVelocity(velocity);
		}
	}

	/**
	 * Calculate the impulse value of the collision of the particle 
	 * with the ground.  The impulse is based on the velocity of the 
	 * particle, coefficent of restitution of the impact, the normal 
	 * of the ground, the collision radius, the mass of the particle 
	 * and the moment of inertia of the shape.  The impulse (j) for 
	 * a collision between a particle and an infinitely massive ground is:
	 * 
	 * 	 	 					-(1 + e)(velocity <.> normal)
	 * 	j =	---------------------------------------------------------------
	 * 		 (1/mass) + ((radius <X> normal) <.> (radius <X> normal))/I
	 * 
	 * where:
	 * 		e = coefficent of restitution
	 * 		<.> is the dot product of two vectors
	 * 		<X> is the cross product of two vectors
	 * 		I = moment of inertia of the shape
	 * 
	 * See http://www.myphysicslab.com/collision.html for more details
	 */

	private float getCollisionImpulse(
		Particle aParticle,
		Vector3f aVelocity,
		Vector3f aRadius,
		float mass,
		float I,
		Vector3f aNormal) {
		float j = 0;
		Vector3f radiusXnormal = new Vector3f();
		radiusXnormal.cross(aRadius, aNormal);
		float radiusXnormal2 = radiusXnormal.dot(radiusXnormal);
		j =
			- (1 + COEFFICIENT_OF_RESTITUTION)
				* (aVelocity.dot(aNormal))
				/ ((1 / mass) + radiusXnormal2 / I);
		return j;
	}

	/**
	 * Return the collision radius based on the collision of the shape.
	 * The collision radius is defined as the vector starting at the center
	 * of mass of the shape (the world position) to the collision point on 
	 * the shape.  The collision point on the shape can be a vertex, edge or
	 * side.  If one vertex intersects this bounds, then it is used.  Otherwise
	 * the vertices are used to create an average.  For example if the collision
	 * is on an edge, then the average will produce a radius at the midpoint of
	 * edge.  If the collision is on a face, then the center of the face would
	 * be used to create the radius.  
	 * @param bb - the bounding box of the shape.  The bounding box must be
	 * transformed to world space.
	 * @param x - the world position x of the center of mass of the shape.
	 * @param y - the world position y of the center of mass of the shape.
	 * @param z - the world position z of the center of mass of the shape.
	 * @return - the radius vector.
	 */
	private Vector3f getCollisionRadius(BoundingBox bb) {
		Vector3f radius = new Vector3f();
		Point3d upper = new Point3d(); // probably reuse the same instance
		Point3d lower = new Point3d(); // probably reuse the same instance
		bb.getUpper(upper);
		bb.getLower(lower);
		int vertexCount = 0;
		Point3d aPoint = new Point3d(); // probably reuse the same instance
		// lower left front
		aPoint.x = lower.x;
		aPoint.y = lower.y;
		aPoint.z = lower.z;
		if (bounds.intersect(aPoint)) {
			radius.x = radius.x + (float) aPoint.x;
			radius.y = radius.y + (float) aPoint.y;
			radius.z = radius.z + (float) aPoint.z;
			vertexCount++;
		}
		// lower left back
		aPoint.x = lower.x;
		aPoint.y = lower.y;
		aPoint.z = upper.z;
		if (bounds.intersect(aPoint)) {
			radius.x = radius.x + (float) aPoint.x;
			radius.y = radius.y + (float) aPoint.y;
			radius.z = radius.z + (float) aPoint.z;
			vertexCount++;
		}
		// lower right front
		aPoint.x = upper.x;
		aPoint.y = lower.y;
		aPoint.z = lower.z;
		if (bounds.intersect(aPoint)) {
			radius.x = radius.x + (float) aPoint.x;
			radius.y = radius.y + (float) aPoint.y;
			radius.z = radius.z + (float) aPoint.z;
			vertexCount++;
		}
		// lower right back
		aPoint.x = upper.x;
		aPoint.y = lower.y;
		aPoint.z = upper.z;
		if (bounds.intersect(aPoint)) {
			radius.x = radius.x + (float) aPoint.x;
			radius.y = radius.y + (float) aPoint.y;
			radius.z = radius.z + (float) aPoint.z;
			vertexCount++;
		}
		// upper right front
		aPoint.x = upper.x;
		aPoint.y = upper.y;
		aPoint.z = lower.z;
		if (bounds.intersect(aPoint)) {
			radius.x = radius.x + (float) aPoint.x;
			radius.y = radius.y + (float) aPoint.y;
			radius.z = radius.z + (float) aPoint.z;
			vertexCount++;
		}
		// upper right back
		aPoint.x = upper.x;
		aPoint.y = upper.y;
		aPoint.z = upper.z;
		if (bounds.intersect(aPoint)) {
			radius.x = radius.x + (float) aPoint.x;
			radius.y = radius.y + (float) aPoint.y;
			radius.z = radius.z + (float) aPoint.z;
			vertexCount++;
		}
		// upper left front
		aPoint.x = lower.x;
		aPoint.y = upper.y;
		aPoint.z = lower.z;
		if (bounds.intersect(aPoint)) {
			radius.x = radius.x + (float) aPoint.x;
			radius.y = radius.y + (float) aPoint.y;
			radius.z = radius.z + (float) aPoint.z;
			vertexCount++;
		}
		// upper left back
		aPoint.x = lower.x;
		aPoint.y = upper.y;
		aPoint.z = upper.z;
		if (bounds.intersect(aPoint)) {
			radius.x = radius.x + (float) aPoint.x;
			radius.y = radius.y + (float) aPoint.y;
			radius.z = radius.z + (float) aPoint.z;
			vertexCount++;
		}
		if (vertexCount > 1) {
			float scale = 1.0f / (float) vertexCount;
			radius.scale(scale);
		}
		if (vertexCount == 0) {
			System.out.println("Error in BounceShape: No vertices intersect");
		}
//		radius.x = radius.x - x;
//		radius.y = radius.y - y;
//		radius.z = radius.z - z;
		return radius;
	}

	public float getRestitution() {
		return restitution;
	}

	public void initializeParticle(Particle aParticle) {
		if (isCompatible(aParticle)) {
			Shape3D aShape = getShape3D(aParticle);
			allowReadCollisionBounds(aShape);
			allowReadLocalToWorld(aShape);
		}
	}

	public void setRestitution(float f) {
		restitution = f;
	}

}
