/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles.influences;
import javax.vecmath.*;

import j3d.examples.particles.emitters.Particle;

/**
 * Represents a point in space that attracts particles to 
 * that point based on the distance between the particle 
 * and the attracting point.  Once inside the inner
 * radius, the particle is killed.  
 * 
 * Care should be taken with the velocity of the particle 
 * relative to the inner radius.  It is possible that the 
 * particle could accelerate through the inner radius during 
 * a time interval dt causing the particle to actually 
 * accelerate away from the point.  This can be avoided by
 * making the inner radius larger.
 */
public class Attract implements IExternalInfluence {
	private Point3f location;
	private Point3f particleLocation;
	private Vector3f acceleration;
	private float rate;
	private float innerRadius2;
	 
	public Attract(){
		this(500, new Point3f(0,0,0));
	}
	
	public Attract(float aRate, Point3f aLocation){
		this(aRate, aLocation, 1);
	}
	
	public Attract(float aRate, Point3d aLocation, float aRadius){
		this(aRate, new Point3f(aLocation), aRadius);
	}
	
	public Attract(float aRate, Point3f aLocation, float aRadius){
		rate = aRate;
		location = aLocation;
		particleLocation = new Point3f();
		acceleration = new Vector3f();
		innerRadius2 = aRadius*aRadius;
	}
	
	public void initializeParticle(Particle aParticle){
		// Nothing to do
	}
	
	public void apply(Particle aParticle, float dt) {
		float [] world = aParticle.getWorldPosition();
		particleLocation.set(world);
		float distance2 = location.distanceSquared(particleLocation);
		if(distance2 > innerRadius2){
			float applied = rate/distance2;
			acceleration.sub(location, particleLocation);
			acceleration.normalize();
			acceleration.scale(applied);
			aParticle.addWorldAcceleration(acceleration);
		}
		else {
			aParticle.die();
		}
	}

}
