/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles.examples;
import java.applet.Applet;
import java.awt.BorderLayout;
import java.awt.GraphicsConfiguration;

import javax.media.j3d.*;
import javax.vecmath.*;

import j3d.examples.particles.*;
import j3d.examples.particles.emitters.*;
import j3d.examples.particles.generationshapes.*;
import j3d.examples.particles.influences.*;
import j3d.examples.particles.shapes.*;

import com.sun.j3d.utils.applet.MainFrame;
import com.sun.j3d.utils.behaviors.keyboard.KeyNavigatorBehavior;
import com.sun.j3d.utils.universe.SimpleUniverse;
import com.sun.j3d.utils.geometry.*;
/*
 * An advanced example that combines most features of the
 * particle system.  The example shows a stick of dynamite
 * next to rock.  The dynamite explodes sending dust and
 * debris with a fire and smoke.  It takes the following
 * particle emitters/systems to accomplish this demo:
 * 
 * fuse - The fuse of the dynamite is a motion blurred
 *        point particle system.
 * dust - The dust is blown out away from the point
 *        of the explosion.  The dust is an implicit 
 *        object (CloudPuff).
 * debris - Small rocks are thrown and spun away from
 *          the destroyed boulder.  The rocks are 
 *          Shape3D objects created with Perlin noise.
 * fire - Flames are implemented with high velocity,
 *        but short lived CloudPuff objects as particles.
 * light - The light from the explosion and fire is done
 *         with Phantom particles with the sole purpose
 *         of moving point light sources as particles.
 * smoke - Dark CloudPuffs are used to create smoke on
 *         top of the fire.
 * 
 */
public class ExplosionExample extends Applet {

	public static void main(String[] args) {
		new MainFrame(new ExplosionExample(false), 600, 600);
	}
	private Canvas3D canvas3D;
	private boolean recording;
	
	public ExplosionExample(boolean isRecording) {
		initialize(isRecording);
	}
	
	public BranchGroup createSceneGraph(SimpleUniverse su) {
		BranchGroup objRoot = new BranchGroup();
		BoundingSphere bounds = new BoundingSphere();
		bounds.setRadius(300);
		// The ground
		XZPlane plane = new XZPlane(1, 100, true, new Color3f(0.12f,0.12f,0));
		objRoot.addChild(plane);
		
		// The dynamite with fuse and a boulder to blow up
		Appearance tntAppearance = new Appearance();
		Material tntMaterial = new Material();
		tntMaterial.setAmbientColor(0.5f,0,0);
		tntAppearance.setMaterial(tntMaterial);
		Cylinder dynamite = new Cylinder(0.7f, 2, Cylinder.GENERATE_NORMALS, tntAppearance);
		
		Rock rock = new Rock(4);
		BranchGroup dynamiteBranchGroup = new BranchGroup();
		TransformGroup dynamiteGroup = new TransformGroup();
		TransformGroup rockGroup = new TransformGroup();
		Transform3D dynamiteT3d = new Transform3D();
		Transform3D rockT3d = new Transform3D();
		dynamiteT3d.setTranslation(new Vector3f(0, 1, 0));
		rockT3d.setTranslation(new Vector3f(-1, 3, -5));
		dynamiteGroup.setTransform(dynamiteT3d);
		rockGroup.setTransform(rockT3d);
		rockGroup.addChild(rock);
		dynamiteGroup.addChild(dynamite);
		dynamiteBranchGroup.addChild(dynamiteGroup);
		dynamiteBranchGroup.addChild(rockGroup);
		objRoot.addChild(dynamiteBranchGroup);
		
		float fuseTime = 2.0f;
		DelayedGroupDetachBehavior detacher = new DelayedGroupDetachBehavior(dynamiteBranchGroup, objRoot, fuseTime+0.2f);
		detacher.setSchedulingBounds(bounds);
		objRoot.addChild(detacher);
		
		ParticleEmitter fuseEmitter =  new ParticleEmitter(
								new PointGenerationShape(Math.PI/8),
								100, 	// emission rate
								0, 	// emission rate variance
								8f, 	// velocity
								2, 		// velocity variance
								2, 		// lifetime
								0f,		// lifetime variance
								fuseTime,	// emitter lifetime in seconds
								true	// motion blur enabled
								);
		fuseEmitter.addInfluence(new FadePoint());
		fuseEmitter.addInfluence(new Gravity());
		ParticleSystem fuse = new MotionBlurredParticleSystem(fuseEmitter, new Color3f(1,1,0));
		
		TransformGroup fuseGroup = new TransformGroup();
		Transform3D fuseT3d = new Transform3D();
		fuseT3d.setTranslation(new Vector3f(0, 2, 0));
		fuseGroup.setTransform(fuseT3d);
		fuseGroup.addChild(fuse);
		objRoot.addChild(fuseGroup);

		// The rock debris thrown during the explosion
		float rate = (float) Math.PI;
		Vector3f particleRotationRate = new Vector3f(rate, rate, rate);
		Vector3f particleRotationRateVariance = new Vector3f(rate, rate, rate);
		ParticleEmitter debrisEmitter =  new ParticleEmitter(
								new PointGenerationShape(Math.PI),
								300f, 	// emission rate
								0f, 	// emission rate variance
								10f, 	// velocity
								6f, 	// velocity variance
								particleRotationRate,
								particleRotationRateVariance,
								12.0f, 	// lifetime
								0.0f,	// lifetime variance
								0.2f	// emitter lifetime in seconds
								);
		debrisEmitter.setDelayTime(fuseTime);
		debrisEmitter.addInfluence(new Gravity());
		BounceShape bounce = new BounceShape();
		bounce.setRestitution(0.3f);
		debrisEmitter.addInfluence(bounce);
		debrisEmitter.addInfluence(new FadeShape(new ParticleAgeAlpha(0,0,0.95f,0.05f)));

		IShape3DFactory debrisFactory = new RockFactory(0.6f, 0.5f);
		Shape3DParticleSystem debris = new Shape3DParticleSystem(debrisEmitter, debrisFactory);
		
		TransformGroup debrisGroup = new TransformGroup();
		Transform3D debrisT3d = new Transform3D();
		debrisT3d.setTranslation(new Vector3f(0, 1.5f, 0));
		debrisGroup.setTransform(debrisT3d);
		debrisGroup.addChild(debris);
		objRoot.addChild(debrisGroup);
		
		CloudPuff.shareGeometry = false;
		
		// The flames of the explosion
		ParticleEmitter explosionEmitter =  new ParticleEmitter(
						new DiskGenerationShape(0, 1, 1),
						25f, 	// emission rate 
						0f, 	// emission rate variance
						8f, 	// velocity
						2f, 	// velocity variance
						0.7f, 	// lifetime 
						0.4f,	// lifetime variance
						5	// emitter lifetime in seconds
						);
		explosionEmitter.setDelayTime(fuseTime);
		explosionEmitter.addInfluence(new FadeShape(new ParticleAgeAlpha(0,0,0.1f,0.9f)));
		explosionEmitter.addInfluence(new Scale(1,1, new ParticleAgeAlpha(0,0,0.4f,0.6f)));
		
		IShape3DFactory explosionFactory = new CloudPuffFactory(2, 1, new Color3f(0.7f, 0f, 0f), new Color3f(0.2f, 0.3f, 0f), 6, 1, CloudPuff.FLAMES);
		Shape3DParticleSystem explosion = new Shape3DParticleSystem(explosionEmitter, explosionFactory);
				
		TransformGroup explosionGroup = new TransformGroup();
		Transform3D explosionT3d = new Transform3D();
		explosionT3d.setTranslation(new Vector3f(0, 1, 0));
		explosionGroup.setTransform(explosionT3d);
		explosionGroup.addChild(explosion);
		objRoot.addChild(explosionGroup);
		
		// Lights for the flames
		ParticleEmitter lightEmitter =  new ParticleEmitter(
								new DiskGenerationShape(0, 1.5f, 1.5f),
								1.5f, 	// emission rate
								0.5f, 	// emission rate variance
								0f, 	// velocity
								0f, 	// velocity variance
								4f, 	// lifetime
								2f,		// lifetime variance
								5 		// emitter lifetime in seconds
								);

		lightEmitter.addInfluence(new EmitShapeLight(new Color3f(0.8f, 0.4f, 0f),1,0,2));
		lightEmitter.setDelayTime(fuseTime);
		IShape3DFactory lightFactory = new PhantomFactory(10, 3);			
		Shape3DParticleSystem lights = new Shape3DParticleSystem(lightEmitter, lightFactory);

		TransformGroup lightGroup = new TransformGroup();
		Transform3D lightT3d = new Transform3D();
		lightT3d.setTranslation(new Vector3f(0, 3f, 0));
		lightGroup.setTransform(lightT3d);
		lightGroup.addChild(lights);
		objRoot.addChild(lightGroup);

		// The dust that is blown out during the initial explosion
		ParticleEmitter blowOutEmitter =  new ParticleEmitter(
						new RadialGenerationShape(2, 0),
						200f, 	// emission rate
						0f, 	// emission rate variance
						40f, 	// velocity
						0f, 	// velocity variance
						2.0f, 	// lifetime
						0.0f,	// lifetime variance
						0.1f	// emitter lifetime in seconds
						);
		blowOutEmitter.setDelayTime(fuseTime);
		blowOutEmitter.addInfluence(new FadeShape());
		blowOutEmitter.addInfluence(new Slow(50));
		blowOutEmitter.addInfluence(new Scale(3f));
		blowOutEmitter.addInfluence(new Gravity());

		IShape3DFactory blowOutFactory = 
				new CloudPuffFactory(
						7.5f,
						0,
						new Color3f(0.3f, 0.3f, 0.3f),
						new Color3f(0.1f, 0.1f, 0.1f),
						new Color3f(0.1f, 0.1f, 0.1f),
						new Color3f(0.0f, 0.0f, 0.0f),
						5,
						0);			
		Shape3DParticleSystem blowOutSystem = new Shape3DParticleSystem(blowOutEmitter, blowOutFactory);

		TransformGroup psGroup = new TransformGroup();
		Transform3D t3d = new Transform3D();
		t3d.setTranslation(new Vector3f(0, 2f, 0));
		psGroup.setTransform(t3d);
		psGroup.addChild(blowOutSystem);
		objRoot.addChild(psGroup);
		
		// The smoke at the top of the flames
		ParticleEmitter smokeEmitter =  new ParticleEmitter(
				new PointGenerationShape(Math.PI/8),
				5f, 	// emission rate
				0f, 	// emission rate variance
				3f, 	// velocity
				1f, 	// velocity variance
				6f, 	// lifetime
				1.0f,	// lifetime variance
				5f	// emitter lifetime in seconds
				);
		smokeEmitter.setDelayTime(fuseTime + 0.8f);				
		smokeEmitter.addInfluence(new Scale(3.5f));
		smokeEmitter.addInfluence(new FadeShape());
		
		IShape3DFactory smokeFactory = 
				new CloudPuffFactory(
						9f,
						2,
						new Color3f(0.15f, 0.15f, 0.15f),
						new Color3f(0.1f, 0.1f, 0.1f),
						new Color3f(0.2f, 0.2f, 0.2f),
						new Color3f(0.2f, 0.2f, 0.2f),
						7,
						0,
						CloudPuff.SMOKE);
		Shape3DParticleSystem smoke = new Shape3DParticleSystem(smokeEmitter, smokeFactory);
		
		TransformGroup smokeGroup = new TransformGroup();
		Transform3D smokeT3d = new Transform3D();
		smokeT3d.setTranslation(new Vector3f(0, 8, 0));
		smokeGroup.setTransform(smokeT3d);
		smokeGroup.addChild(smoke);
		objRoot.addChild(smokeGroup);
		
		// Add ambient light
		AmbientLight ambient = new AmbientLight();
		ambient.setColor(new Color3f(1f, 1f, 1f));
		ambient.setInfluencingBounds(bounds);
		objRoot.addChild(ambient);

		// Add a directional light
		DirectionalLight directional = new DirectionalLight();
		directional.setDirection(1.0f, -0.4f, -1f);
		directional.setColor(new Color3f(0.5f, 0.5f, 0.5f));
		directional.setInfluencingBounds(bounds);
		objRoot.addChild(directional);

		// Add a behavior to manage the particle system animation
		ParticleSystemManager manager = ParticleSystemManager.getCurrent();
		manager.setSchedulingBounds(bounds);
		objRoot.addChild(manager);
		
		// Add an image capturing behavior
		if(recording){
			ImageCaptureBehavior capture = new ImageCaptureBehavior(canvas3D, 0, 300);
			capture.setSchedulingBounds(bounds);
			objRoot.addChild(capture);
		}
		
		// Add a keyboard navigator to move around
		TransformGroup vpTrans =
			su.getViewingPlatform().getViewPlatformTransform();
		KeyNavigatorBehavior keyNavBeh =
			new KeyNavigatorBehavior(vpTrans);
		keyNavBeh.setSchedulingBounds(bounds);
		objRoot.addChild(keyNavBeh);
		
		// Optimize the scene graph
		objRoot.compile();
		return objRoot;
	}
	private void initialize(boolean isRecording) {
		recording = isRecording;
		setLayout(new BorderLayout());
		GraphicsConfiguration config =
			SimpleUniverse.getPreferredConfiguration();
		canvas3D = new Canvas3D(config);
		add("Center", canvas3D);
		SimpleUniverse simpleU = new SimpleUniverse(canvas3D);
		
		//Position the view
		TransformGroup viewingPlatformGroup =
			simpleU.getViewingPlatform().getViewPlatformTransform();
		Transform3D t3d = new Transform3D();
		//t3d.rotX(-Math.PI/16);
		t3d.setTranslation(new Vector3f(0, 5, 30));
		viewingPlatformGroup.setTransform(t3d);

		BranchGroup scene = createSceneGraph(simpleU);
		simpleU.addBranchGraph(scene);

		canvas3D.getView().setBackClipDistance(300.0d);
		canvas3D.getView().setFrontClipDistance(0.1d);
		canvas3D.getView().setMinimumFrameCycleTime(20);
		canvas3D.getView().setTransparencySortingPolicy(View.TRANSPARENCY_SORT_GEOMETRY);
		
	}

}
