/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles.emitters;
import javax.vecmath.*;
import java.util.*;

import javax.media.j3d.*;

import j3d.examples.particles.*;
import j3d.examples.particles.influences.*;
import j3d.examples.particles.generationshapes.*;
/**
 * The particle emitter is responsible for managing the
 * lifecycle of all particles.  It supports a variety of
 * attributes that follow the center value plus variance
 * pattern.  The particle emitter depends on a generation
 * shape for the intial placement and velocity of particles.
 * The particle emitter uses influence objects to change
 * aspects of the particles.  Particle life cycle listeners
 * are notified by the particle emitter as the particles
 * live their lives.
 */
// TODO: Allow the number of live particles to be specified instead of emission rate
// TODO: Update ParticleEmitter to allow abritrary number of prior positions to support hierarchical particle systems
// TODO: Allow removal of influences from the particle system
public class ParticleEmitter {
	private static final boolean DEBUG = false;
	private static final boolean DEEP_DEBUG = false;
	private List dead;
	private float delayTime; // How much time should pass prior to starting
	private float emissionRate; // particles per second
	private float emissionRateVariance;
	private int emissions = 0;
	private IGenerationShape generationShape;
	private Collection influences;
	private List living;
	private Transform3D localToWorld;
	private int maximumParticles;
	private List monitors;
	private List newborn;
	private float particleLife;
	private float particleLifeVariance;
	private Vector3f particleRotationRate;
	private Vector3f particleRotationRateVariance;

	private IParticleSystem particleSystem;
	private float particleVelocity;
	private float particleVelocityVariance;
	private Collection pool;

	private float[] positions;

	private boolean rotatingParticles;
	private float runningTime = 0;
	private float timeToLive; // seconds to live
	protected int verticesPerParticle;
	private Transform3D worldToLocal;

	public ParticleEmitter(
		IGenerationShape generationShape,
		float emissionRate,
		float emissionRateVariance,
		float particleVelocity,
		float particleVelocityVariance,
		float particleLife,
		float particleLifeVariance,
		float emitterLife) {
		this(
			generationShape,
			emissionRate,
			emissionRateVariance,
			particleVelocity,
			particleVelocityVariance,
			particleLife,
			particleLifeVariance,
			emitterLife,
			false);
	}

	public ParticleEmitter(
		IGenerationShape generationShape,
		float emissionRate,
		float emissionRateVariance,
		float particleVelocity,
		float particleVelocityVariance,
		float particleLife,
		float particleLifeVariance,
		float emitterLife,
		boolean motionBlur) {

		generationShape.setEmitter(this);
		rotatingParticles = false;
		this.generationShape = generationShape;
		this.emissionRate = emissionRate;
		this.emissionRateVariance = emissionRateVariance;
		this.particleLife = particleLife;
		this.particleLifeVariance = particleLifeVariance;
		this.particleVelocity = particleVelocity;
		this.particleVelocityVariance = particleVelocityVariance;

		if (motionBlur) {
			verticesPerParticle = 2;
		} else {
			verticesPerParticle = 1;
		}

		maximumParticles =
			Math.round(
				(emissionRate + emissionRateVariance)
					* (particleLife + particleLifeVariance));
		if (emitterLife <= 1) {
			maximumParticles =
				Math.round((emissionRate + emissionRateVariance) * emitterLife);
		}
		positions = new float[3 * verticesPerParticle * maximumParticles];
		timeToLive = emitterLife;
	}

	public ParticleEmitter(
		IGenerationShape generationShape,
		float emissionRate,
		float emissionRateVariance,
		float particleVelocity,
		float particleVelocityVariance,
		Vector3f particleRotationRate,
		Vector3f particleRotationRateVariance,
		float particleLife,
		float particleLifeVariance,
		float emitterLife) {
		this(
			generationShape,
			emissionRate,
			emissionRateVariance,
			particleVelocity,
			particleVelocityVariance,
			particleLife,
			particleLifeVariance,
			emitterLife,
			false);

		rotatingParticles = true;
		this.particleRotationRate = particleRotationRate;
		this.particleRotationRateVariance = particleRotationRateVariance;
	}
	
	/**
	 * Add an external influence to this particle system.
	 * @param anInfluence - The influence to add.
	 */
	public void addInfluence(IExternalInfluence anInfluence) {
		getInfluences().add(anInfluence);
		if (anInfluence instanceof IParticleLifeCycleListener) {
			addMonitor((IParticleLifeCycleListener) anInfluence);
		}
	}

	/**
	 * Add a new life cycle listener to this particle system.
	 * @param aListener - The listener to be notified of changes
	 */
	public void addMonitor(IParticleLifeCycleListener aListener) {
		getMonitors().add(aListener);
	}

	protected void addWorldAcceleration(
		Particle particle,
		Vector3f anAcceleration) {
		// Should be called only by Particle
		getWorldToLocal().transform(anAcceleration);
		particle.addLocalAcceleration(anAcceleration);
	}

	protected void applyInfluences(Particle aParticle, float dt) {
		if (getInfluences().size() == 0) {
			return;
		}
		Iterator iterator = getInfluences().iterator();
		while (iterator.hasNext()) {
			IExternalInfluence anInfluence =
				(IExternalInfluence) iterator.next();
			anInfluence.apply(aParticle, dt);
		}
	}

	/**
	 * Kill this particle system.
	 */
	public void die() {
		timeToLive = 0;
	}

	private void dumpParticleCollection(String label, Collection c) {
		// For debugging the particle life cycle
		if (c.size() == 0)
			return;
		System.out.print(label);
		Iterator iterator = c.iterator();
		while (iterator.hasNext()) {
			Particle particle = (Particle) iterator.next();
			System.out.print(particle + ", ");
		}
		System.out.println("");
	}

	private void emitNewParticles(float dt) {
		int numberToEmit = 0;
		runningTime += dt;
		int idealEmissions = Math.round(runningTime * getVaryingEmissionRate());
		if (idealEmissions > emissions) {
			// Need to emit more
			numberToEmit =
				Math.min(idealEmissions - emissions, getPool().size());
		} else {
			// No need to emit during this cycle
			return;
		}

		Iterator iterator = getPool().iterator();
		for (int i = numberToEmit; i > 0; i--) {
			Particle particle = (Particle) iterator.next();
			initializeParticle(particle);
			getNewborn().add(particle);
		}
		if (DEBUG) {
			System.out.println("emitting: " + numberToEmit);
			if (DEEP_DEBUG) {
				dumpParticleCollection("emitting: ", getNewborn());
			}
		}
		getPool().removeAll(getNewborn());
		notifyAboutToEmit(getNewborn(), dt);
		getLiving().addAll(getNewborn());
		getNewborn().clear();
		emissions = emissions + numberToEmit;
		// ignoring the possibility that long running systems could exceed int
	}

	protected void endMotionBlur(Particle aParticle) {
		// Remove the motion blur from this particle to emulate
		// a particle at rest.
		savePreviousPosition(aParticle);
	}

	private List getDead() {
		if (dead == null) {
			dead = new ArrayList(Math.round(emissionRate));
		}
		return dead;
	}

	public float getDelayTime() {
		return delayTime;
	}

	protected IGenerationShape getGenerationShape() {
		return generationShape;
	}

	protected Collection getInfluences() {
		if (influences == null) {
			influences = new ArrayList();
		}
		return influences;
	}

	private List getLiving() {
		if (living == null) {
			living = new ArrayList(getMaximumParticles());
		}
		return living;
	}

	protected float[] getLocalPosition(Particle aParticle) {
		float[] position = new float[3];
		int pi = getParticlePositionIndex(aParticle);
		position[0] = positions[pi + 0];
		position[1] = positions[pi + 1];
		position[2] = positions[pi + 2];
		return position;
	}

	protected Transform3D getLocalToWorld() {
		if (localToWorld == null) {
			localToWorld = new Transform3D();
		}
		getParticleSystem().getLocalToVworld(localToWorld);
		return localToWorld;
	}

	public int getMaximumParticles() {
		return maximumParticles;
	}

	public List getMonitors() {
		if (monitors == null) {
			monitors = new ArrayList();
		}
		return monitors;
	}

	private List getNewborn() {
		if (newborn == null) {
			newborn = new ArrayList(Math.round(emissionRate));
		}
		return newborn;
	}

	protected int getParticlePositionIndex(Particle aParticle) {
		int i = aParticle.getIndice();
		int pi = 3 * verticesPerParticle * i;
		return pi;
	}

	public IParticleSystem getParticleSystem() {
		return particleSystem;
	}

	private Collection getPool() {
		if (pool == null) {
			pool = new Vector(getMaximumParticles());
			initializePool();
		}
		return pool;
	}

	/**
	 * An array of float values representing the x, y, z position
	 * of all particles (living or dead).  Can be used directly with
	 * BY_REFERENCE geometry array objects.  The first three values
	 * represent the x, y, z of the first particle and so on.
	 */
	public float[] getPositions() {
		return positions;
	}

	protected float getVaryingEmissionRate() {
		return emissionRate + random() * emissionRateVariance;
	}

	/**
	 * Get the central value plus random variance value of
	 * the particle life time.
	 */
	public float getVaryingParticleLife() {
		return particleLife + random() * particleLifeVariance;
	}

	/**
	 * Get the central value plus random variance value of
	 * the particle rotation rate.
	 */
	public Vector3f getVaryingParticleRotationRate() {
		Vector3f varyingRotation = new Vector3f();
		varyingRotation.scaleAdd(
			random(),
			particleRotationRateVariance,
			particleRotationRate);
		return varyingRotation;
	}

	/**
	 * Get the central value plus random variance value of
	 * the particle velocity.
	 */
	public float getVaryingParticleVelocity() {
		return particleVelocity + random() * particleVelocityVariance;
	}

	protected float[] getWorldPosition(Particle aParticle) {
		Point3f position = new Point3f(getLocalPosition(aParticle));
		getLocalToWorld().transform(position);
		float[] worldPosition = new float[3];
		worldPosition[0] = position.x;
		worldPosition[1] = position.y;
		worldPosition[2] = position.z;
		return worldPosition;
	}

	protected Transform3D getWorldToLocal() {
		if (worldToLocal == null) {
			worldToLocal = new Transform3D();
		}
		worldToLocal.invert(getLocalToWorld());
		return worldToLocal;
	}

	protected Vector3f getWorldVelocity(Particle particle) {
		Vector3f worldVelocity = new Vector3f(particle.getLocalVelocity());
		getLocalToWorld().transform(worldVelocity);
		return worldVelocity;
	}
	
	public void initialize(){
		// Mostly for emitters that have a delay so that 
		// it can preallocate anything it might need later.
		// Should be called after all influences have been
		// added and the emitter is ready to go.
		getPool();
	}
	protected void initializeParticle(Particle particle) {
		initializeParticleAge(particle);
		initializeParticlePosition(particle);
		initializeParticleVelocity(particle);
		initializeParticleAcceleration(particle);
		initializeParticleRotation(particle);
	}

	protected void initializeParticleAcceleration(Particle aParticle) {
		aParticle.resetAcceleration();
	}

	protected void initializeParticleAge(Particle aParticle) {
		aParticle.setLiveSpan(getVaryingParticleLife());
	}

	protected void initializeParticlePosition(Particle aParticle) {
		getGenerationShape().initializeParticlePosition(aParticle);
	}

	protected void initializeParticleRotation(Particle aParticle) {
		aParticle.setRotatable(rotatingParticles);
		if (rotatingParticles) {
			aParticle.startRotating();
			Vector3f aRotation =
				new Vector3f(
					(float) Math.PI * random(),
					(float) Math.PI * random(),
					(float) Math.PI * random());
			//aRotation = new Vector3f(0,0,0);
			aParticle.setEulerOrientation(aRotation);
			aParticle.setEulerRotationRate(getVaryingParticleRotationRate());
		}
	}

	protected void initializeParticleVelocity(Particle aParticle) {
		getGenerationShape().initializeParticleVelocity(aParticle);
	}

	private void initializePool() {
		if (DEBUG) {
			System.out.println(
			"Allocating " + getMaximumParticles() + " particles");
		}

		for (int i = 0; i < getMaximumParticles(); i++) {
			Particle particle = new Particle(i, this);
			Iterator influences = getInfluences().iterator();
			while (influences.hasNext()) {
				IExternalInfluence ei = (IExternalInfluence) influences.next();
				ei.initializeParticle(particle);
			}
			initializeParticle(particle);
			pool.add(particle);
		}
	}

	public boolean isAlive() {
		return !isDead();
	}

	public boolean isDead() {
		return timeToLive <= 0;
	}

	protected boolean isMotionBlurred() {
		return verticesPerParticle == 2;
	}

	protected void notifyAboutToDie(List aList, float dt) {
		int aSize = getMonitors().size();
		for(int i = 0; i < aSize; i++) {
		IParticleLifeCycleListener listener =
			(IParticleLifeCycleListener) getMonitors().get(i);
			listener.aboutToDie(aList, dt);
		}
	}

	protected void notifyAboutToEmit(List aList, float dt) {
		int aSize = getMonitors().size();
		for(int i = 0; i < aSize; i++) {
		IParticleLifeCycleListener listener =
			(IParticleLifeCycleListener) getMonitors().get(i);
			listener.aboutToEmit(aList, dt);
		}
	}

	protected void notifyUpdated(List aList, float dt) {
		int aSize = getMonitors().size();
		for(int i = 0; i < aSize; i++) {
			IParticleLifeCycleListener listener =
				(IParticleLifeCycleListener) getMonitors().get(i);
			listener.updated(aList, dt);
		}
	}

	public float random() {
		return 2 * (0.5f - (float) Math.random());
	}

	private void returnDeadParticlesToPool(float dt) {
		if (DEBUG) {
			System.out.println("dead: " + getDead().size());
			if (DEEP_DEBUG) {
				dumpParticleCollection("dead: ", getDead());
			}
		}

		getLiving().removeAll(getDead());
		notifyUpdated(getLiving(), dt);
		getPool().addAll(getDead());
		notifyAboutToDie(getDead(), dt);
		getDead().clear();
	}

	protected void savePreviousPosition(Particle aParticle) {
		// Save previous location for antialiasing of line
		if (isMotionBlurred()) {
			float[] local = getPositions();
			int pi = getParticlePositionIndex(aParticle);
			local[pi + 3] = local[pi + 0];
			local[pi + 4] = local[pi + 1];
			local[pi + 5] = local[pi + 2];
		}
	}

	public void setDelayTime(float dt) {
		delayTime = dt;
	}

	protected void setGenerationShape(IGenerationShape aShape) {
		generationShape = aShape;
	}

	protected void setInitialLocalPosition(
		Particle aParticle,
		float x,
		float y,
		float z) {
		// Should only be called from Particle
		int pi = getParticlePositionIndex(aParticle);
		positions[pi + 0] = x;
		positions[pi + 1] = y;
		positions[pi + 2] = z;
		if (isMotionBlurred()) {
			positions[pi + 3] = x;
			positions[pi + 4] = y;
			positions[pi + 5] = z;
		}
	}

	protected void setLocalPosition(
		Particle aParticle,
		float x,
		float y,
		float z) {
		// Should only be called from Particle
		int pi = getParticlePositionIndex(aParticle);
		positions[pi + 0] = x;
		positions[pi + 1] = y;
		positions[pi + 2] = z;
	}

	public void setParticleSystem(IParticleSystem system) {
		particleSystem = system;
	}

	protected void setWorldPosition(
		Particle aParticle,
		float x,
		float y,
		float z) {
		// Should only be called from Particle
		int pi = getParticlePositionIndex(aParticle);
		Point3f position = new Point3f(x, y, z);
		getWorldToLocal().transform(position);
		positions[pi + 0] = position.x;
		positions[pi + 1] = position.y;
		positions[pi + 2] = position.z;
		if (isMotionBlurred()) {
			positions[pi + 3] = position.x;
			positions[pi + 4] = position.y;
			positions[pi + 5] = position.z;
		}
	}

	protected void setWorldVelocity(
		Particle particle,
		Vector3f worldVelocity) {
		// Should be called only by Particle
		getWorldToLocal().transform(worldVelocity);
		particle.setLocalVelocity(worldVelocity);
	}

	/**
	 * Update the particle lives based on the time differential.  
	 * Manages the lifecycle of the particles 
	 * (emit new, collect dead, update live).
	 * Should only be called by the particle systems.
	 *  
	 * @param dt - The time differential in seconds
	 */
	public void update(float dt) {
		if(delayTime > 0){
			delayTime = delayTime - dt;
			return;
		}
		timeToLive = timeToLive - dt;
		if (DEBUG) {
			System.out.println("---------------");
			System.out.println("running time: " + (runningTime + dt));
			System.out.println("dt: " + dt);
			System.out.println("pool: " + getPool().size());
		}

		if (isAlive()) {
			emitNewParticles(dt);
		}
		returnDeadParticlesToPool(dt);
		updateLiveParticles(dt); // influences applied here
	}

	private void updateLiveParticles(float dt) {
		if (DEBUG) {
			System.out.println("living: " + getLiving().size());
			if (DEEP_DEBUG) {
				dumpParticleCollection("living: ", getLiving());
			}
		}
		Iterator iterator = getLiving().iterator();
		while (iterator.hasNext()) {
			Particle particle = (Particle) iterator.next();
			particle.update(dt);
			if (particle.isDead()) {
				initializeParticlePosition(particle);
				initializeParticleAcceleration(particle);
				if (isMotionBlurred()) {
					endMotionBlur(particle);
				}
				dead.add(particle);
			}
		}
	}

	protected void updateParticle(Particle aParticle, float dt) {
		// Should only be called from Particle
		// Apply all external influences for the time interval dt
		savePreviousPosition(aParticle);
		aParticle.resetAcceleration();
		applyInfluences(aParticle, dt);
		aParticle.move(dt);
		aParticle.rotate(dt);
	}

}
