/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/

package j3d.examples.particles.emitters;
import javax.vecmath.*;

/**
 * Represents the elementary particle of a particle system.  Maintains
 * position, velocity, acceleration and age of the particle.  
 * Assumes a cartesian coordinate representation of position, 
 * velocity, and acceleration.  This implementation also includes 
 * rotational attributes normally associated with rigid bodies.  Strictly
 * speaking particles would not have rotational attributes and could be
 * split out into a rigid body particle.  I'll leave that as an 
 * exercise to the interested student.  I would create a IParticle
 * interface and factories for the primitive particle and rigid body
 * particle so the emitter would not need to know which one was in use.
 * 
 * Note: This implementation uses Euler's approach to numeric integration.
 * This approach is simple but has the potential for significant error.
 * Physical simulation implementations should use the Runge-Kutta order
 * four method to significantly reduce the accumulated error.
 *   
 * Implies a BY_REFERENCE implementation for Java3D by delegating
 * some actions to the owning particle emitter.
 */

public class Particle {
	private Vector3f acceleration; // meters per second per second
	private Quat4f angularVelocityDirection;
	private float angularVelocityMagnitude;
	private ParticleEmitter emitter;
	private Vector3f eulerOrientation;  // Orientation as Euler angles
	private Vector3f eulerRotationRate; // Rate of rotation as Euler angles
	private int indice;
	private float lifeSpan;
	private boolean rotatable;
	private boolean rotating;
	private Quat4f orientation; // rotational position as quaternion
	private Quat4f rotationChange; // reused object for rotational calculations
	private float scale = 1.0f;
	private float timeToLive; // seconds to live
	private Vector3f velocity; // meters per second

	public Particle(int indice, ParticleEmitter emitter) {
		this.indice = indice;
		this.emitter = emitter;
	}
	
	public void addLocalAcceleration(Vector3f anAcceleration){
		Vector3f a = getLocalAcceleration();
		a.add(anAcceleration);
	}
	
	public void addWorldAcceleration(Vector3f anAcceleration){
		emitter.addWorldAcceleration(this, anAcceleration);
	}
	
	public void age(float dt){
		// dt in seconds
		timeToLive = timeToLive - dt;
	}

	/**
	 * Kill this particle before it's time.  The emitter 
	 * will treat this particle as it does any other particle 
	 * that dies of old age.
	 */
	public void die() {
		timeToLive = 0;
	}

	public Quat4f getAngularVelocityDirection() {
		return angularVelocityDirection;
	}

	public float getAngularVelocityMagnitude() {
		return angularVelocityMagnitude;
	}

	public ParticleEmitter getEmitter() {
		return emitter;
	}

	public Vector3f getEulerOrientation() {
		return eulerOrientation;
	}
	public Vector3f getEulerRotationRate() {
		return eulerRotationRate;
	}

	public int getIndice() {
		return indice;
	}

	public float getLifeSpanPercentage() {
		return (lifeSpan - timeToLive) / lifeSpan;
	}
	
	public Vector3f getLocalAcceleration() {
		if (acceleration == null) {
			acceleration = new Vector3f();
		}
		return acceleration;
	}

	public float[] getLocalPosition() {
		return emitter.getLocalPosition(this);
	}
	
	public Vector3f getLocalVelocity() {
		if (velocity == null) {
			velocity = new Vector3f();
		}
		return velocity;
	}

	/**
	 * @return The quaternion representing the the 
	 * rotational position of the particle.
	 */
	public Quat4f getOrientation() {
		if (orientation == null) {
			orientation = new Quat4f();
			// Convert the Euler rotation to a Quaternion
			// Take care of x-axis component 
			float xAngle = eulerOrientation.x / 2;
			float sine = (float) Math.sin(xAngle);
			float cosine = (float) Math.cos(xAngle);
			orientation.x = sine;
			orientation.w = cosine;

			// Take care of y-axis component 
			float yAngle = eulerOrientation.y / 2;
			sine = (float) Math.sin(yAngle);
			cosine = (float) Math.cos(yAngle);
			Quat4f yQuaternion = new Quat4f();
			yQuaternion.y = sine;
			yQuaternion.w = cosine;

			// Take care of z-axis component 
			float zAngle = eulerOrientation.z / 2;
			sine = (float) Math.sin(zAngle);
			cosine = (float) Math.cos(zAngle);
			Quat4f zQuaternion = new Quat4f();
			zQuaternion.z = sine;
			zQuaternion.w = cosine;

			orientation.mul(yQuaternion);
			orientation.mul(zQuaternion);
		}
		return orientation;
	}

	public float getScale() {
		return scale;
	}

	public float[] getWorldPosition() {
		return emitter.getWorldPosition(this);
	}

	public Vector3f getWorldVelocity() {
		return emitter.getWorldVelocity(this);
	}

	public boolean isAlive() {
		return !isDead();
	}

	public boolean isDead() {
		return timeToLive <= 0;
	}

	public boolean isRotatable() {
		return rotatable;
	}
	public boolean isRotating() {
		return rotating;
	}

	/**
	 * Move this particle based on the current position,
	 * velocity, and acceleration.  Calculations use Euler's
	 * algorithm for numerically integrating ordinary differential
	 * equations.  See Chris Hecker's excellent series game physics
	 * series in Game Developer Magazine (starting in Oct/Nov 1996).
	 * @param dt - The time interval (in seconds) since the last move.
	 */
	//	TODO: Refactor Euler's Approach out into seperate object
	public void move(float dt) {
		float[] position = getLocalPosition();
		Vector3f v = getLocalVelocity();
		Vector3f a = getLocalAcceleration();
		v.scaleAdd(dt, a, v);
		// Scale add not used with position 
		// to reduce number of objects created.
		float x = position[0] + v.x * dt;
		float y = position[1] + v.y * dt;
		float z = position[2] + v.z * dt;
		setLocalPosition(x, y, z);
	}
	
	protected void resetAcceleration(){
		Vector3f a = getLocalAcceleration();
		a.set(0,0,0);
	}

	/**
	 * Rotate this particle based on the current rotational
	 * position and angular velocity.
	 * 
	 * @param dt - The time interval (in seconds) since the last rotation.
	 */
	public void rotate(float dt) {
		if (isRotatable() && isRotating() && (angularVelocityMagnitude > 0)) {
			if (rotationChange == null) {
				// To reduce object creation
				rotationChange = new Quat4f();
			}
			/*  
			 * The time derivative of a quaternion is:
			 * 		dQ/dt = 0.5 * (angular velocity vector) * Q
			 * where Q is the quaternion.  The angular velocity vector 
			 * is represented as the direction (Quat4f angularVelocityDirection) 
			 * and magnitude (float rotationalScale) to make the following
			 * calculations easier (only quaternions can be multiplied by
			 * other quaternions).  
			 * 
			 * The new quaternion for this
			 * time interval is:
			 * 	Q(now) = Q(previous) + (dQ/dt)*dt; 
			 * 	where: 
			 * 		(dQ/dt)*dt = dt * 0.5 * (angular velocity vector) * Q(previous) 
			 * 		(dQ/dt)*dt = angularVelocityMagnitude * dt * 0.5f * angularVelocityDirection * aRotation
			 */
			Quat4f aRotation = getOrientation();  // Q(previous)
			rotationChange.mul(aRotation, getAngularVelocityDirection());
			rotationChange.scale(angularVelocityMagnitude * dt * 0.5f); // (dQ/dt)*dt;
			aRotation.add(rotationChange); // Q(now) = Q(previous) + (dQ/dt)*dt;
			aRotation.normalize(); // Otherwise affects the scale of the object in odd ways
		}
	}

	public void setAngularVelocityDirection(Quat4f aQuat4f) {
		angularVelocityDirection = aQuat4f;
	}

	public void setAngularVelocityMagnitude(float aMagnitude) {
		angularVelocityMagnitude = aMagnitude;
	}

	/**
	 * Set the rotational position of the particle by using Euler angles (in radians).  
	 * Euler angles specify rotations independently for each axis.
	 *  
	 * @param anOrientation - a Vector3f containing the axis specific orientation.
	 */
	public void setEulerOrientation(Vector3f anOrientation) {
		this.eulerOrientation = anOrientation;
	}

	public void setEulerRotationRate(Vector3f aRate) {
		/* The angular velocity vector is represented as the 
		 * direction (Quat4f angularVelocityDirection) and magnitude
		 * (float rotationalScale).  The angularVelocityDirection is 
		 * maintained as a quaternion instead of a vector so that it can
		 * more easily participate in the computations needed
		 * for rotation.  See rotate(float dt) for calculations.
		 */
		eulerRotationRate = aRate;
		setAngularVelocityDirection(new Quat4f(aRate.x, aRate.y, aRate.z, 0));
		setAngularVelocityMagnitude(aRate.length());
	}

	public void setInitialLocalPosition(float x, float y, float z) {
		emitter.setInitialLocalPosition(this, x, y, z);
	}

	protected void setLiveSpan(float lifeSpan) {
		timeToLive = lifeSpan;
		this.lifeSpan = lifeSpan;
	}

	public void setLocalPosition(float x, float y, float z) {
		emitter.setLocalPosition(this, x, y, z);
	}
	
	protected void setLocalVelocity(float vx, float vy, float vz) {
		Vector3f v = getLocalVelocity();
		v.set(vx, vy, vz);
	}
	
	public void setLocalVelocity(Vector3f aVelocity) {
		Vector3f v = getLocalVelocity();
		v.set(aVelocity);
	}

	public void setRotatable(boolean aBoolean) {
		rotatable = aBoolean;
	}

	public void setOrientation(Quat4f aQuaternion) {
		Quat4f anOrientation = getOrientation();
		anOrientation.set(aQuaternion);
	}

	public void setScale(float f) {
		scale = f;
	}

	public void setWorldPosition(float x, float y, float z) {
		emitter.setWorldPosition(this, x, y, z);
	}
	public void setWorldVelocity(Vector3f worldVelocity) {
		emitter.setWorldVelocity(this, worldVelocity);
	}

	public void startRotating() {
		rotating = true;
	}

	public void stopRotating() {
		rotating = false;
	}

	public String toString() {
		return indice + " " + isAlive();
	}

	public void update(float dt) {
		// dt in seconds
		age(dt);
		if (isAlive()) {
			emitter.updateParticle(this, dt);
		}
	}

}
