/**********************************************************
Copyright (C) 2005, Michael N. Jacobs, All Rights Reserved

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

**************************************************************/
package j3d.examples.particles;

import javax.vecmath.*;
import javax.media.j3d.*;

import j3d.examples.particles.emitters.*;

/**
 * The abstract super class for simple particle systems.
 */
public abstract class ParticleSystem extends Shape3D implements IParticleSystem, GeometryUpdater {
	private float[] colors;
	private ParticleEmitter emitter;
	private Color4f particleColor;
	private float dt;

	/**
	 * Creates a particle system with default settings.  The default initial particle
	 * color is white (opaque).
	 * @param anEmitter - A <code>ParticleEmitter</code> for this particle system.
	 */
	public ParticleSystem(ParticleEmitter anEmitter) {
		this(anEmitter, new Color4f(1,1,1,1));
	}
	
	/**
	 * Creates a particle system.
	 * @param anEmitter - A <code>ParticleEmitter</code> for this particle system.
	 * @param aColor3f - The fully opaque initial color for particles in this particle system.
	 */
	public ParticleSystem(ParticleEmitter anEmitter, Color3f aColor3f){
		this(anEmitter, new Color4f(aColor3f.x,aColor3f.y,aColor3f.z,1));
	}
	
	/**
	 * Creates a particle system.
	 * @param anEmitter - A <code>ParticleEmitter</code> for this particle system.
	 * @param aColor4f - The initial color for particles in this particle system.
	 */
	public ParticleSystem(ParticleEmitter anEmitter, Color4f aColor4f){
		this.emitter = anEmitter;
		this.particleColor = aColor4f;
		emitter.setParticleSystem(this);
		ParticleSystemManager.getCurrent().register(this);
	}
	
	/**
	 * An extension point for subclasses to create a custom appearance.
	 * @return The <code>Appearance</code> to use for the particle system.
	 */
	protected abstract Appearance createAppearance();
	/**
	 * An extension point for subclasses to create a custom geometry.
	 * @return The <code>Geometry</code> of the particle system.
	 */
	protected abstract Geometry createGeometry();
	
	/**
	 * The colors array suitable for use in a by reference implementation.
	 * @return The <code>float[]</code> containing the colors of all particles in 
	 * this particle system.  Each color consists of four elements: red, green, blue,
	 * and alpha.
	 */
	protected float[] getColors(){
		if(colors == null){
			colors = new float[4 * getVerticesPerParticle()* getMaximumParticles()];
		}
		return colors;
	}
	
	protected ParticleEmitter getEmitter(){
		return emitter;
	}
	
	protected int getMaximumParticles(){
		return emitter.getMaximumParticles();
	}
	
	/**
	 * @return The initial color for all particles in this particle system.
	 */
	protected Color4f getParticleColor(){
		return particleColor;
	}
	
	/**
	 * Get the current color of a specific particle.
	 * @param aParticle - The particle of interest.
	 * @return A <code>float[]</code> containing the red, green, blue and alpha values of the color.
	 */
	public float[] getParticleColor(Particle aParticle){
		float[] color = new float[4];
		int index = 4* getVerticesPerParticle()*aParticle.getIndice();
		float [] theColors = getColors();
		color[0] = theColors[index + 0];
		color[1] = theColors[index + 1];
		color[2] = theColors[index + 2];
		color[3] = theColors[index + 3];
		return color;
	}
	
	/**
	 * An extension point for subclasses to specify the vertex format flags.
	 * @return The logic or of the desired vertex format flags 
	 */
	protected abstract int getVertexFormat();
	/**
	 * An extension point for subclasses to specify the number of vertices
	 * that should be allocated per particle.
	 * @return The number of vertices per particle.
	 */
	protected abstract int getVerticesPerParticle();
	
	/* (non-Javadoc)
	 * @see particles.IParticleSystem#isAlive()
	 */
	public boolean isAlive() {
		return getEmitter().isAlive();
	}

	/* (non-Javadoc)
	 * @see particles.IParticleSystem#isDead()
	 */
	public boolean isDead() {
		return getEmitter().isDead();
	}
	
	/* (non-Javadoc)
	 * @see particles.IParticleSystem#nextFrame()
	 */
	public void nextFrame(float dt) {
		/* Called during animation.  Because
		 * this object is a GeometryUpdater, get
		 * the geometry and pass this object to
		 * the updateData method on the geometry.
		 */
		this.dt = dt;
		GeometryArray ga = (GeometryArray) getGeometry();
		ga.updateData(this);
	}
	
	protected void postConstructionInitialization(){
		setGeometry(createGeometry());
		setAppearance(createAppearance());
		setCapabilities();
	}

	protected void setCapabilities() {
		setCapability(Shape3D.ALLOW_GEOMETRY_READ);
		setCapability(Shape3D.ALLOW_LOCAL_TO_VWORLD_READ);
		setPickable(true);
		Geometry geometry = getGeometry();
		geometry.setCapability(GeometryArray.ALLOW_REF_DATA_READ);
		geometry.setCapability(GeometryArray.ALLOW_REF_DATA_WRITE);
	}
	
	/** 
	 * Set the color of a particle by particle index.
	 * @param anIndex - The indice of the particle.
	 * @param red - The red component of the color.
	 * @param green - The green component of the color.
	 * @param blue - The blue component of the color.
	 * @param alpha - - The alpha component of the color.
	 */
	protected void setParticleColor(int anIndex, float red, float green, float blue, float alpha){
		float [] theColors = getColors();
		theColors[anIndex + 0] = red;
		theColors[anIndex + 1] = green;
		theColors[anIndex + 2] = blue;
		theColors[anIndex + 3] = alpha;
	}
	
	/**
	 * Set the color of a particle.
	 * @param aParticle - The particle to update.
	 * @param red - The red component of the color.
	 * @param green - The green component of the color.
	 * @param blue - The blue component of the color.
	 * @param alpha - - The alpha component of the color.
	 */
	public void setParticleColor(Particle aParticle, float red, float green, float blue, float alpha){
		setParticleColor(4 * getVerticesPerParticle()* aParticle.getIndice(), red, green, blue, alpha);
	}
	
	/**
	 * Set the color of the particle.
	 * @param aParticle - The particle to update.
	 * @param aColor - A four element <code>float[]</code> containing the red, green, blue, and alpha color components.
	 */
	public void setParticleColor(Particle aParticle, float[] aColor){
		setParticleColor(4 * getVerticesPerParticle()* aParticle.getIndice(), aColor[0], aColor[1], aColor[2], aColor[3]);
	}

	/* (non-Javadoc)
	 * @see javax.media.j3d.GeometryUpdater#updateData(javax.media.j3d.Geometry)
	 */
	public void updateData(Geometry geometry) {
		getEmitter().update(dt);
	}
}
