package j3d.examples.appearance.texture.noise;

import com.sun.j3d.utils.applet.MainFrame;
import com.sun.j3d.utils.behaviors.keyboard.KeyNavigatorBehavior;
import com.sun.j3d.utils.behaviors.mouse.MouseRotate;
import com.sun.j3d.utils.geometry.Sphere;
import com.sun.j3d.utils.universe.SimpleUniverse;

import javax.media.j3d.*;
import javax.vecmath.Color3f;
import javax.vecmath.Vector3f;
import java.applet.Applet;
import java.awt.*;
import java.awt.image.BufferedImage;

/**
 * Perlin noise example that generates a texture for
 * a sphere.  The generated texture is controlled by
 * the parameter on the constructor and can be one of:
 * <p/>
 * PLAIN - Noise interpreted as color
 * TURBULANCE - Fire like pattern
 * MARBLE - Just like the stone
 * WOOD - Grain and color similar to Cherry
 * <p/>
 * No attempt has been made to spherically correct
 * the texture.  As a result, most have a seam in
 * the back and mapping effects at the poles.
 */

public class NoiseSphereMain extends Applet {

    private static final int PLAIN = 0;
    private static final int TURBULANCE = 1;
    private static final int MARBLE = 2;
    private static final int WOOD = 3;

    private int mode = PLAIN;

    public NoiseSphereMain() {
        initialize();
    }

    public NoiseSphereMain(int aMode) {
        this.mode = aMode;
        initialize();
    }

    private void initialize() {
        setLayout(new BorderLayout());
        GraphicsConfiguration config =
                SimpleUniverse.getPreferredConfiguration();
        Canvas3D canvas3D = new Canvas3D(config);
        add("Center", canvas3D);
        SimpleUniverse simpleU = new SimpleUniverse(canvas3D);
        //Position the view
        TransformGroup viewingPlatformGroup =
                simpleU.getViewingPlatform().getViewPlatformTransform();
        Transform3D t3d = new Transform3D();
        t3d.setTranslation(new Vector3f(0, 0, 20));
        viewingPlatformGroup.setTransform(t3d);

        BranchGroup scene = createSceneGraph(simpleU);
        simpleU.addBranchGraph(scene);

        // Adjust the clipping planes
        canvas3D.getView().setBackClipDistance(3000.0d);
        canvas3D.getView().setFrontClipDistance(1d);

    }

    public BranchGroup createSceneGraph(SimpleUniverse su) {
        BranchGroup objRoot = new BranchGroup();

        TransformGroup objRotate = new TransformGroup();
        objRotate.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        objRotate.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
        objRoot.addChild(objRotate);

        Sphere sphere =
                new Sphere(5,
                        Sphere.GENERATE_NORMALS | Sphere.GENERATE_TEXTURE_COORDS,
                        200,
                        getAppearance(128));
        objRotate.addChild(sphere);


        // Let the user rotate the sphere with the mouse
        MouseRotate myMouseRotate = new MouseRotate();
        myMouseRotate.setTransformGroup(objRotate);
        myMouseRotate.setSchedulingBounds(new BoundingSphere());
        objRoot.addChild(myMouseRotate);

        // Add ambient light
        BoundingSphere bounds = new BoundingSphere();
        bounds.setRadius(100);
        AmbientLight ambient = new AmbientLight();
        ambient.setColor(new Color3f(1f, 1f, 1f));
        ambient.setInfluencingBounds(bounds);
        objRoot.addChild(ambient);

        // Add a directional light
        DirectionalLight directional = new DirectionalLight();
        directional.setDirection(0.8f, -0.7f, 1f);
        directional.setColor(new Color3f(1f, 1f, 1f));
        directional.setInfluencingBounds(bounds);
        objRoot.addChild(directional);

        // Add a keyboard navigator to move around
        TransformGroup vpTrans =
                su.getViewingPlatform().getViewPlatformTransform();
        KeyNavigatorBehavior keyNavBeh = new KeyNavigatorBehavior(vpTrans);
        keyNavBeh.setSchedulingBounds(bounds);
        objRoot.addChild(keyNavBeh);

        // Optimize the scene graph
        objRoot.compile();
        return objRoot;
    }

    private Appearance getAppearance(int imageSize) {
        Appearance appearance = new Appearance();
        PolygonAttributes polyAttrib = new PolygonAttributes();
        polyAttrib.setPolygonMode(PolygonAttributes.POLYGON_FILL);

        appearance.setPolygonAttributes(polyAttrib);
        Material material = new Material();
        material.setAmbientColor(0.0f, 0.0f, 0.0f);

        appearance.setMaterial(material);

        // Create the texture for the sphere
        Texture2D texture =
                new Texture2D(Texture2D.BASE_LEVEL,
                        Texture2D.RGBA,
                        imageSize,
                        imageSize);
//texture.setMipMapMode(Texture2D.MULTI_LEVEL_MIPMAP);
        texture.setImage(0, getImage(imageSize));
//ImageComponent2D image = getImage(imageSize/2);
//texture.setImage(1, image);
//texture.setImage(2, getImage(imageSize/4));
        texture.setEnable(true);
        // Set the optional quality settings
        texture.setMagFilter(Texture2D.NICEST);
        texture.setMinFilter(Texture2D.NICEST);
        appearance.setTexture(texture);

        return appearance;
    }

    private ImageComponent2D getImage(int imageSize) {

        BufferedImage bi =
                new BufferedImage(imageSize,
                        imageSize,
                        BufferedImage.TYPE_INT_RGB);


        switch (mode) {
            case PLAIN:
                generatePlainNoiseImage(bi, imageSize);
                break;
            case TURBULANCE:
                generateTurbulanceImage(bi, imageSize);
                break;
            case MARBLE:
                generateMarbleImage(bi, imageSize);
                break;
            case WOOD:
                generateWoodImage(bi, imageSize);
                break;
            default :
                generatePlainNoiseImage(bi, imageSize);
        }

        ImageComponent2D image =
                new ImageComponent2D(ImageComponent2D.FORMAT_RGBA, bi, true, false);
        return image;
    }

    private void generatePlainNoiseImage(BufferedImage bi, int imageSize) {
        for (int x = 0; x < imageSize; x++) {
            for (int y = 0; y < imageSize; y++) {
                // A color must be in the range of 0 to 255.  The divisors
                // were arrived at through trial and error.
                double color =
                        128.0 - 127.0 * ImprovedNoise.noise(x / 7.62, y / 14.37, 0);
                int red = 0;
                int green = (int) color / 2;
                int blue = (int) color;
                bi.setRGB(x, y, color(red, green, blue));
            }
        }
    }

    private void generateTurbulanceImage(BufferedImage bi, int imageSize) {
        for (int column = 0; column < imageSize; column++) {
            for (int row = 0; row < imageSize; row++) {
                // A color must be in the range of 0 to 255.  The divisors
                // were arrived at through trial and error.
                double x = column / 27.76;
                double y = row / 49.37;
                double z = 37.8;
                double noise = ImprovedNoise.turbulance(x, y, z, 8);
                double color = Math.min(192 * noise, 192);
                int red = 255 - (int) (0.3 * color);
                int green = 192 - (int) (color);
                int blue = 0;
                bi.setRGB(column, row, color(red, green, blue));
            }
        }
    }

    private void generateMarbleImage(BufferedImage bi, int imageSize) {
        for (int column = 0; column < imageSize; column++) {
            for (int row = 0; row < imageSize; row++) {
                double x = column / 14.7;
                double y = row / 14.7;
                double z = 4.8;
                double marble = ImprovedNoise.marble(x, y, z, 8);
                double root = Math.sqrt(marble + 1) * 0.7071;
                int green = (int) (255.0 * (0.2 + 0.8 * root));
                root = Math.sqrt(root);
                int red = (int) (255.0 * (0.3 + 0.6 * root));
                int blue = (int) (255.0 * (0.6 + 0.4 * root));
                bi.setRGB(column, row, color(red, green, blue));
            }
        }
    }

    private void generateWoodImage(BufferedImage bi, int imageSize) {
        for (int column = 0; column < imageSize; column++) {
            for (int row = 0; row < imageSize; row++) {
                double x = column / 282.8;
                double y = row / 322.8;
                double z = 1.2;

                double grain = ImprovedNoise.wood(x, y, z);
                int red = 71 + (int) (164.0 * grain);
                int green = 34 + (int) (74.0 * grain);
                int blue = 34 + (int) (24.0 * grain);
                bi.setRGB(column, row, color(red, green, blue));
            }
        }
    }

    private int color(int red, int green, int blue) {
        return (red << 16) + (green << 8) + blue;
    }

    public static void main(String[] args) {
        System.out.println("The default is WOOD.  Change main() to use MARBLE, PLAIN or TURBULANCE.");
        long seed = System.currentTimeMillis();
        ImprovedNoise.reinitialize(seed);
        System.out.println("Seed: " + seed);
        new MainFrame(new NoiseSphereMain(WOOD), 500, 500);
    }

}
