package j3d.examples.appearance.texture.noise;

import com.sun.j3d.utils.applet.MainFrame;
import com.sun.j3d.utils.behaviors.keyboard.KeyNavigatorBehavior;
import com.sun.j3d.utils.geometry.GeometryInfo;
import com.sun.j3d.utils.geometry.NormalGenerator;
import com.sun.j3d.utils.universe.SimpleUniverse;

import javax.media.j3d.*;
import javax.vecmath.Color3f;
import javax.vecmath.Vector3f;
import java.applet.Applet;
import java.awt.*;

/**
 * An artifical terrain example using an indexed
 * TriangleStripArray as the geometry of the surface,
 * and fractals to generate the height field.
 * <p/>
 * Results are dependent on random numbers so a
 * a landscape will be unique each time you run this
 * example.
 * <p/>
 * Elevation color assignments are perturbed with noise
 * or a random number based on the colorIndexMode.
 * <p/>
 * For a full explanation of the algorithm used to
 * generate the terrain, refer to the September 2004
 * edition of Java Developer Journal.
 */
public class Main3 extends Applet {
    private static final int NUMBER_OF_COLORS = 27;
    private static final float SIDE_LENGTH = 350.0f; // Meters

    private static final int NONE = 0;
    private static final int RANDOM = 1;
    private static final int NOISE = 2;
    private int colorIndexMode = NONE;

    private static java.util.Random generator;

    private float roughness;
    private int divisions;
    private int lod = 0; // level of detail


    public Main3() {
        lod = 8;
        roughness = 0.45f;
    }

    public Main3(int lod, float roughness, int colorIndexMode) {
        this.roughness = roughness;
        this.lod = lod;
        this.colorIndexMode = colorIndexMode;
        initialize();
    }

    private void initialize() {
        divisions = 1 << lod;
        setLayout(new BorderLayout());
        GraphicsConfiguration config =
                SimpleUniverse.getPreferredConfiguration();
        Canvas3D canvas3D = new Canvas3D(config);
        add("Center", canvas3D);
        SimpleUniverse simpleU = new SimpleUniverse(canvas3D);
        //Position the view
        TransformGroup viewingPlatformGroup =
                simpleU.getViewingPlatform().getViewPlatformTransform();
        Transform3D t3d = new Transform3D();
        t3d.rotY(-Math.PI / 4);
        t3d.setTranslation(new Vector3f(0, 100, 0));
        viewingPlatformGroup.setTransform(t3d);

        BranchGroup scene = createSceneGraph(simpleU);
        simpleU.addBranchGraph(scene);

        // Adjust the clipping planes
        canvas3D.getView().setBackClipDistance(3000.0d);
        canvas3D.getView().setFrontClipDistance(1d);

    }

    public BranchGroup createSceneGraph(SimpleUniverse su) {

        float[][] hf = getHeightField();
        // The height field
        BranchGroup objRoot = new BranchGroup();

        GeometryInfo gi =
                new GeometryInfo(GeometryInfo.TRIANGLE_STRIP_ARRAY);
        float[] coordinates =
                new float[(divisions + 1) * (divisions + 1) * 3];

        // Convert the height field to x, y, z values
        float metersPerDivision = SIDE_LENGTH / divisions;
        for (int row = 0; row < (divisions + 1); row++) {
            for (int col = 0; col < (divisions + 1); col++) {
                // coordinate index is the column plus
                // the row times the width of a row times
                // the 3 (one for each x, y, and z).
                int ci = (col + row * (divisions + 1)) * 3;
                // x, y, z
                coordinates[ci + 0] = metersPerDivision * col;
                coordinates[ci + 1] = hf[row][col];
                coordinates[ci + 2] = -metersPerDivision * row;
            }
        }
        // The number of indices is based on the
        // number of horizontal strips (height - 1) times the
        // number of vertices per strip (width * 2).
        int[] indices = new int[divisions * (divisions + 1) * 2];
        // The secret is that the strip vertices must be ordered
        // like this: NW, SW, NE, SE for each set of four corners
        // of a quad.  A convenient way to accomplish this is to
        // organize the landscape in horizontal strips and iterate
        // across the columns calculating two vertices at a time.
        int pi = 0; // points index
        for (int row = 0; row < divisions; row++) {
            int width = row * (divisions + 1);
            for (int col = 0; col < (divisions + 1); col++) {
                int coordinateIndex = width + col;
                indices[pi] = coordinateIndex + (divisions + 1);
                indices[pi + 1] = coordinateIndex;
                pi = pi + 2;
            }
        }
        int[] stripCounts = new int[divisions];
        for (int strip = 0; strip < divisions; strip++) {
            stripCounts[strip] = (divisions + 1) * 2;
        }
        gi.setStripCounts(stripCounts);
        gi.setCoordinates(coordinates);
        gi.setCoordinateIndices(indices);
        float[] colors = getElevationColors();
        gi.setColors3(colors);
        int[] colorIndices = getElevationColorIndices(hf);
        gi.setColorIndices(colorIndices);
        NormalGenerator ng = new NormalGenerator();
        ng.generateNormals(gi);

        Geometry geometry = gi.getIndexedGeometryArray();
        Shape3D shape = new Shape3D(geometry);
        shape.setAppearance(getAppearance());
        objRoot.addChild(shape);

        // Add ambient light
        BoundingSphere bounds = new BoundingSphere();
        bounds.setRadius(10000);
        AmbientLight ambient = new AmbientLight();
        ambient.setColor(new Color3f(1f, 1f, 1f));
        ambient.setInfluencingBounds(bounds);
        objRoot.addChild(ambient);

        // Add a directional light
        DirectionalLight directional = new DirectionalLight();
        directional.setDirection(0.3f, -1f, 0.5f);
        directional.setColor(new Color3f(1f, 1f, 1f));
        directional.setInfluencingBounds(bounds);
        objRoot.addChild(directional);

        // Add a keyboard navigator to move around
        TransformGroup vpTrans =
                su.getViewingPlatform().getViewPlatformTransform();
        KeyNavigatorBehavior keyNavBeh =
                new KeyNavigatorBehavior(vpTrans);
        keyNavBeh.setSchedulingBounds(bounds);
        objRoot.addChild(keyNavBeh);

        // Optimize the scene graph
        objRoot.compile();
        return objRoot;
    }

    private float[][] getHeightField() {

        float[][] hf = new float[divisions + 1][divisions + 1];
        // Initialize the corners of the height field.  You
        // could use random() for these values.
        hf[0][0] = 0f;
        hf[0][divisions] = 0f;
        hf[divisions][divisions] = 0f;
        hf[divisions][0] = 0f;
        float rough = roughness;
        // Evaluate the fractal for each of the
        // requested levels of detail.
        for (int detail = lod; detail > 0; detail--) {
            // The length of the side for
            // this level of detail iteration.
            int side = 1 << detail;
            int half = side >> 1;
            // Evaluate each square to create the diamond
            // pattern by finding a corner of each square.
            for (int x = 0; x < divisions; x += side) {
                for (int y = 0; y < divisions; y += side) {
                    // x,y is the lower left corner of
                    // the square to evaluate...
                    diamond(hf, x, y, side, rough);
                }
            }
            if (half > 0) {
                // Evaluate each diamond to create the square
                // pattern by finding the center of each
                // diamond.
                for (int x = 0; x <= divisions; x += half) {
                    for (int y = (x + half) % side;
                         y <= divisions;
                         y += side) {
                        // x, y is the center of the
                        // diamond to evaluate ...
                        square(hf, x, y, side, rough);
                    }
                }
            }
            // Adjust the roughness factor
            // to gradually scale it down.
            rough *= roughness;
            // Fractal purists would divide the roughness
            // by 2 for each iteration.  This creates a
            // scene that is (subjectively) too rough.
            // rough = rough/2.0f;
        }

        float min = hf[0][0];
        float max = hf[0][0];
        for (int i = 0; i <= divisions; i++) {
            for (int j = 0; j <= divisions; j++) {
                if (hf[i][j] < min) {
                    min = hf[i][j];
                } else if (hf[i][j] > max) {
                    max = hf[i][j];
                }
            }
        }

        // replace the hf values with the percentage
        // of the range.
        float range = max - min;
        for (int i = 0; i <= divisions; i++) {
            for (int j = 0; j <= divisions; j++) {
                hf[i][j] = 100f * (hf[i][j] - min) / range;
            }
        }
        return hf;
    }

    private void diamond(float[][] terrain,
                         int x,
                         int y,
                         int side,
                         float roughness) {
        if (side > 1) {
            int half = side / 2;
            float sum = 0f;
            sum += terrain[x][y];
            sum += terrain[x + side][y];
            sum += terrain[x + side][y + side];
            sum += terrain[x][y + side];
            float elevation = sum / 4.0f;
            terrain[x + half][y + half] =
                    elevation + random() * roughness;
        }
    }

    private void square(float[][] terrain,
                        int x,
                        int y,
                        int side,
                        float roughness) {
        // Because x, y is the center of the diamond,
        // it is possible that corners of the diamond
        // are outside the bounds of the landscape, so
        // this method has a few boundary conditions
        // to check for this possibility.
        int half = side / 2;
        float sum = 0.0f, number = 0.0f;

        if (x - half >= 0) {
            // West corner
            sum += terrain[x - half][y];
            number += 1.0;
        }
        if (y - half >= 0) {
            // South corner
            sum += terrain[x][y - half];
            number += 1.0;
        }
        if (x + half <= divisions) {
            // East corner
            sum += terrain[x + half][y];
            number += 1.0;
        }
        if (y + half <= divisions) {
            // North corner
            sum += terrain[x][y + half];
            number += 1.0;
        }
        float elevation = sum / number;
        terrain[x][y] = elevation + random() * roughness;
    }

    private float random() {
        return 2.0f * getGenerator().nextFloat() - 1.0f;
    }

    private int[] getElevationColorIndices(float[][] hf) {
        int[] indices = new int[divisions * (divisions + 1) * 2];
        int i = 0;
        for (int row = 0; row < divisions; row++) {
            for (int col = 0; col < (divisions + 1); col++) {
                // NW
                indices[i] = getElevationColorIndex(hf[row + 1][col], row + 1, col);
                // SW
                indices[i + 1] = getElevationColorIndex(hf[row][col], row, col);
                i = i + 2;
            }
        }
        return indices;
    }

    private int getElevationColorIndex(float elevation, int row, int col) {
        // Normalize the height value to a
        // color index between 0 and NUMBER_OF_COLORS - 1
        float index = (NUMBER_OF_COLORS - 1) * ((100f - elevation) / 100f);
        float delta = 0;
        switch (colorIndexMode) {
            case NONE:
                delta = 0;
                break;
            case NOISE:
                delta = 1.5f * (float) ImprovedNoise.noise(row / 3.7, elevation / 7.4, col / 3.7);
                break;
            case RANDOM:
                delta = 1.5f * random();
                break;
            default:
                delta = 0;
        }

        int answer = Math.round(index + delta);
        // Clamp the index value
        if (answer < 0) answer = 0;
        if (answer > NUMBER_OF_COLORS - 1) answer = NUMBER_OF_COLORS - 1;
        return answer;
    }

    private float[] getElevationColors() {
        // These colors were arrived at through experimentation.
        // A color utility I found very useful is called
        // 'La boite a couleurs' by Benjamin Chartier
        //
        float[] colors = new float[3 * NUMBER_OF_COLORS];
        int i = 0;
        //
        colors[i++] = 0.72f;
        colors[i++] = 0.59f;
        colors[i++] = 0.44f;
        //
        colors[i++] = 0.64f;
        colors[i++] = 0.49f;
        colors[i++] = 0.32f;
        //
        colors[i++] = 0.51f;
        colors[i++] = 0.39f;
        colors[i++] = 0.25f;

        colors[i++] = 0.43f;
        colors[i++] = 0.33f;
        colors[i++] = 0.21f;
        //
        colors[i++] = 0.38f;
        colors[i++] = 0.29f;
        colors[i++] = 0.18f;
        //
        colors[i++] = 0.31f;
        colors[i++] = 0.25f;
        colors[i++] = 0.15f;

        colors[i++] = 0.27f;
        colors[i++] = 0.21f;
        colors[i++] = 0.13f;

        colors[i++] = 0.23f;
        colors[i++] = 0.28f;
        colors[i++] = 0.14f;

        //
        colors[i++] = 0.28f;
        colors[i++] = 0.36f;
        colors[i++] = 0.14f;
        //
        colors[i++] = 0.23f;
        colors[i++] = 0.35f;
        colors[i++] = 0.11f;
        //
        colors[i++] = 0.28f;
        colors[i++] = 0.43f;
        colors[i++] = 0.13f;

        colors[i++] = 0.30f;
        colors[i++] = 0.46f;
        colors[i++] = 0.14f;
        //
        colors[i++] = 0.33f;
        colors[i++] = 0.50f;
        colors[i++] = 0.16f;
        //
        colors[i++] = 0.35f;
        colors[i++] = 0.53f;
        colors[i++] = 0.17f;

        colors[i++] = 0.38f;
        colors[i++] = 0.58f;
        colors[i++] = 0.18f;

        colors[i++] = 0.43f;
        colors[i++] = 0.66f;
        colors[i++] = 0.20f;
        // 		sandy
        colors[i++] = 0.62f;
        colors[i++] = 0.58f;
        colors[i++] = 0.38f;
        //
        colors[i++] = 0.66f;
        colors[i++] = 0.62f;
        colors[i++] = 0.44f;
        //
        colors[i++] = 0.70f;
        colors[i++] = 0.67f;
        colors[i++] = 0.50f;
        //
        colors[i++] = 0.74f;
        colors[i++] = 0.71f;
        colors[i++] = 0.56f;
        //
        colors[i++] = 0.77f;
        colors[i++] = 0.75f;
        colors[i++] = 0.63f;
        //		blue
        colors[i++] = 0.0f;
        colors[i++] = 0.56f;
        colors[i++] = 0.57f;
        //
        colors[i++] = 0.0f;
        colors[i++] = 0.38f;
        colors[i++] = 0.54f;
        //
        colors[i++] = 0.0f;
        colors[i++] = 0.24f;
        colors[i++] = 0.35f;
        //
        colors[i++] = 0.0f;
        colors[i++] = 0.14f;
        colors[i++] = 0.20f;
        //
        colors[i++] = 0.0f;
        colors[i++] = 0.07f;
        colors[i++] = 0.10f;
        //
        colors[i++] = 0.0f;
        colors[i++] = 0.03f;
        colors[i++] = 0.04f;

        return colors;
    }

    private static java.util.Random getGenerator() {
        if (generator == null) {
            long seed = System.currentTimeMillis();
            //long seed = 1089493588870L;
            //long seed = 1089771339464L;
            //long seed = 1096734340842L;

            // The scene is reproducable with the same
            // seed so if you find one you like, keep
            // the seed.
            System.out.println("Seed: " + seed);
            generator = new java.util.Random(seed);
        }
        return generator;
    }

    private Appearance getAppearance() {
        Appearance appearance = new Appearance();
        PolygonAttributes polyAttrib = new PolygonAttributes();
        polyAttrib.setCullFace(PolygonAttributes.CULL_NONE);
        polyAttrib.setPolygonMode(PolygonAttributes.POLYGON_FILL);
        // Setting back face normal flip to true allows backfacing
        // polygons to be lit (normal is facing wrong way) but only
        // if the normal is flipped.
        polyAttrib.setBackFaceNormalFlip(true);
        appearance.setPolygonAttributes(polyAttrib);
        Material material = new Material();
        material.setAmbientColor(0f, 0f, 0f);
        // Changing the specular color reduces the shine
        // that occurs with the default setting.
        material.setSpecularColor(0.1f, 0.1f, 0.1f);
        appearance.setMaterial(material);
        return appearance;
    }

    public static void main(String[] args) {
        // Depending on the options used,
        // you may need to use the -Xms256M -Xmx1000M
        // options when running this example.
        // A roughness of 0.35 to 0.55 seems like
        // a good range.
        System.out.println("The default is to use NOISE.  Change main() to use NONE or RANDOM.");
        Frame frame =
                new MainFrame(new Main3(8, 0.45f, NOISE), 600, 600);
    }

}
