// Glenn Josefiak
// Fairfield University
// SW513
// Spring 2003

package j2d.imageproc;

import java.awt.*;
import java.awt.image.BufferedImage;
import java.awt.image.ImageObserver;
import java.awt.image.PixelGrabber;
import java.io.File;

/**
 * This class will be the basis of many image processing algorithms.
 * All image processing algorithms operate on an image and require
 * the image's height and width.  This class encapsulates that
 * functionality.  Also, all image processing operations will require
 * a performAlgorithm() method.  This method is left as abstract.
 */
public abstract class ImageProcessor
        implements ImageObserver {
	protected Image baseImage;
    protected Image newImage;
	protected int   baseImageHeight = 0;
	protected int   baseImageWidth = 0;

	/**
	 * Construct a new ImageProcessor.
	 */ 
	public ImageProcessor(){
		baseImage = null;
	}
	
	/**
	 * Set the base image reference to an image object read from
     * the specified file.
	 */ 
	public void openImage(File file){
		String fileName = file.getAbsolutePath();
		baseImage =
			Toolkit.getDefaultToolkit().getImage(fileName);
        newImage = baseImage;
		baseImageWidth = baseImage.getWidth(this);
		baseImageHeight = baseImage.getHeight(this);
	}

    /**
     * Set the base image reference.
     */
    public void setBaseImage(Image newImage){
        baseImage = newImage;
        this.newImage = baseImage;
        baseImageWidth = baseImage.getWidth(this);
        baseImageHeight = baseImage.getHeight(this);
    }

	/**
	 * Return a handle to the base image.
	 */
	public Image getBaseImage(){
		return baseImage;
	}
	
	/**
	 * Obtain the width of the base image.
	 */
	public int getBaseImageWidth(){
		return baseImageWidth;
	}
	
	/**
	 * Obtain the height of the base image.
	 */
	public int getBaseImageHeight(){
		return baseImageHeight;
	}

    /**
     * Return a handle to the processed image.
     */
    public Image getProcessedImage(){
        return newImage;
    }

    /**
     * Return newImage converted to an array of pixels.
     */
    public int[] getPixels(){
        int pixels[] = new int[baseImageWidth*baseImageHeight];

        try{
            PixelGrabber pg = new PixelGrabber(
                    baseImage, 0, 0, baseImageWidth, baseImageHeight,
                    pixels, 0, baseImageWidth);
            pg.grabPixels();
        }catch(InterruptedException ie){
            // do nothing for now
        }
        return pixels;
    }

    /**
     *  Store the array of pixels into newImage
     */
    public void setPixels(int pixels[]){
        newImage = new BufferedImage(baseImageWidth,baseImageHeight,
				BufferedImage.TYPE_INT_RGB);
		((BufferedImage)newImage).setRGB(0,0,baseImageWidth,
				baseImageHeight,pixels,0,baseImageWidth);
    }

	/**
	 * Implementation of ImageObserver.  This is used in case the
	 * image takes some time to load and the width and height are
	 * not immediately available.
	 */
 	public boolean imageUpdate(Image img, int infoflags, int x, int y, 
 		int width, int height){
 		
	    if ((infoflags & ALLBITS) != 1) {
 			baseImageWidth = width;
 			baseImageHeight = height;
      		return true;
    	} else {
      		return false;
    	}
 	}

	/**
	 * This abstract function must be overridden by any actual
	 * ImageProcessor implementation.  This is where the real
	 * image processing happens.  If the implementation generates
     * an output image, performAlgorithm is responsible for
     * updating newImage.
	 */
	public abstract void processImage() throws Exception;
}