package gui.dialogs;

import gui.run.RunButton;

import javax.swing.*;
import javax.swing.text.DefaultFormatterFactory;
import javax.swing.text.NumberFormatter;
import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.NumberFormat;
import java.text.ParseException;

/**
 * Copyright DocJava, inc.
 * User: lyon
 * Date: Sep 3, 2004
 * Time: 9:53:56 AM
 */
public class FormattedTextDialog extends JDialog
        implements PropertyChangeListener {
    //Values for the text fields
    private double amount = 100000;
    private double rate = .075;  //7.5 %
    private int numPeriods = 30;

    //Labels to identify the fields
    private JLabel amountLabel = new JLabel("loan");
    private JLabel rateLabel = new JLabel("years");
    private JLabel numPeriodsLabel = new JLabel("yrs");
    private JLabel paymentLabel = new JLabel("mnthy pmnt");

    //Fields for data entry
    private JFormattedTextField amountField=getAmountField();
    private JFormattedTextField rateField;
    private JFormattedTextField numPeriodsField;
    private JFormattedTextField paymentField;

    //Formats to format and parse numbers
    private NumberFormat amountDisplayFormat;
    private NumberFormat amountEditFormat;
    private NumberFormat percentDisplayFormat;
    private NumberFormat percentEditFormat;
    private NumberFormat paymentFormat;

    public FormattedTextDialog() {
        Container c = getContentPane();
        c.setLayout(new BorderLayout());
        setUpFormats();
        double payment = computePayment(amount,
                rate,
                numPeriods);

        NumberFormatter percentEditFormatter = getPercentFormatter();
        rateField = new JFormattedTextField(new DefaultFormatterFactory(
                new NumberFormatter(percentDisplayFormat),
                new NumberFormatter(percentDisplayFormat),
                percentEditFormatter));
        rateField.setValue(new Double(rate));
        rateField.setColumns(10);
        rateField.addPropertyChangeListener("value", this);

        numPeriodsField = new JFormattedTextField();
        numPeriodsField.setValue(new Integer(numPeriods));
        numPeriodsField.setColumns(10);
        numPeriodsField.addPropertyChangeListener("value", this);

        paymentField = new JFormattedTextField(paymentFormat);
        paymentField.setValue(new Double(payment));
        paymentField.setColumns(10);
        paymentField.setEditable(false);
        paymentField.setForeground(Color.red);

        //Tell accessibility tools about label/textfield pairs.
        amountLabel.setLabelFor(amountField);
        rateLabel.setLabelFor(rateField);
        numPeriodsLabel.setLabelFor(numPeriodsField);
        paymentLabel.setLabelFor(paymentField);

        //Lay out the labels in a panel.
        JPanel labelPane = new JPanel(new GridLayout(0, 1));
        labelPane.add(amountLabel);
        labelPane.add(rateLabel);
        labelPane.add(numPeriodsLabel);
        labelPane.add(paymentLabel);

        //Layout the text fields in a panel.
        JPanel fieldPane = getTextFieldPanel();

        //Put the panels in this panel, labels on left,
        //text fields on right.

        //setBorder(BorderFactory.createEmptyBorder(20, 20, 20, 20));
        c.add(labelPane, BorderLayout.CENTER);
        c.add(fieldPane, BorderLayout.LINE_END);
        c.add(getButtonPanel(this),BorderLayout.SOUTH);
        pack();
        setModal(true);
    }

    private NumberFormatter getPercentFormatter() {
        NumberFormatter percentEditFormatter =
                new NumberFormatter(percentEditFormat) {
                    public String valueToString(Object o)
                            throws ParseException {
                        Number number = (Number) o;
                        if (number != null) {
                            double d = number.doubleValue() * 100.0;
                            number = new Double(d);
                        }
                        return super.valueToString(number);
                    }

                    public Object stringToValue(String s)
                            throws ParseException {
                        Number number = (Number) super.stringToValue(s);
                        if (number != null) {
                            double d = number.doubleValue() / 100.0;
                            number = new Double(d);
                        }
                        return number;
                    }
                };
        return percentEditFormatter;
    }

    private JFormattedTextField getAmountField() {
        JFormattedTextField amountField = new JFormattedTextField(new DefaultFormatterFactory(
                new NumberFormatter(amountDisplayFormat),
                new NumberFormatter(amountDisplayFormat),
                new NumberFormatter(amountEditFormat)));
        amountField.setValue(new Double(amount));
        amountField.setColumns(10);
        amountField.addPropertyChangeListener("value", this);
        return amountField;
    }

    private JPanel getTextFieldPanel() {
        JPanel fieldPane = new JPanel(new GridLayout(0, 1));
        fieldPane.add(amountField);
        fieldPane.add(rateField);
        fieldPane.add(numPeriodsField);
        fieldPane.add(paymentField);
        return fieldPane;
    }

    public JPanel getButtonPanel(final JDialog parent) {
        JPanel jp = new JPanel(new FlowLayout());
        jp.add(new RunButton("OK"){
            public void run() {
                parent.setVisible(false);
            }
        });
        return jp;
    }

    /**
     * Called when a field's "value" property changes.
     */
    public void propertyChange(PropertyChangeEvent e) {
        Object source = e.getSource();
        if (source == amountField) {
            amount = ((Number) amountField.getValue()).doubleValue();
        } else if (source == rateField) {
            rate = ((Number) rateField.getValue()).doubleValue();
        } else if (source == numPeriodsField) {
            numPeriods = ((Number) numPeriodsField.getValue()).intValue();
        }

        double payment = computePayment(amount, rate, numPeriods);
        paymentField.setValue(new Double(payment));
    }

    public static void main(String[] args) {
        FormattedTextDialog ftf = new FormattedTextDialog();

        ftf.setVisible(true);
        System.out.println("done");

    }

    //Compute the monthly payment based on the loan amount,
    //APR, and length of loan.
    double computePayment(double loanAmt, double rate, int numPeriods) {
        double I, partial1, denominator, answer;

        numPeriods *= 12;        //get number of months
        if (rate > 0.001) {
            I = rate / 12.0;         //get monthly rate from annual
            partial1 = Math.pow((1 + I), (0.0 - numPeriods));
            denominator = (1 - partial1) / I;
        } else { //rate ~= 0
            denominator = numPeriods;
        }

        answer = (-1 * loanAmt) / denominator;
        return answer;
    }

    //Create and set up number formats. These objects also
    //parse numbers input by user.
    private void setUpFormats() {
        amountDisplayFormat = NumberFormat.getCurrencyInstance();
        amountDisplayFormat.setMinimumFractionDigits(0);
        amountEditFormat = NumberFormat.getNumberInstance();

        percentDisplayFormat = NumberFormat.getPercentInstance();
        percentDisplayFormat.setMinimumFractionDigits(2);
        percentEditFormat = NumberFormat.getNumberInstance();
        percentEditFormat.setMinimumFractionDigits(2);

        paymentFormat = NumberFormat.getCurrencyInstance();
    }
}
