package graphics.grapher;
import java.awt.*;
import java.util.*;


/*************************************************************************
**************************************************************************
**
**    This class is designed to bundle together all the information required
**    to draw short Strings
**
*************************************************************************/



public class TextLine {

/************
**
** Constants
**
************/
     final static int CENTER   =    0;
     final static int LEFT =        1;
     final static int RIGHT =       2;

     final static int MINIMUM_SIZE  =  5;

/*********************
**
** Protected Variables
**
*********************/
     protected double script_fraction = 0.8;
     protected double sup_offset      = 0.6;
     protected double sub_offset      = 0.7;

     protected Font font     = null;
     protected Color color   = null;
     protected String text   = null;

     protected String fontname  = "TimesRoman";
     protected int    fontsize  = 0;
     protected int    fontstyle = Font.PLAIN;

     protected int    justification = LEFT;

     protected int width   = 0;
     protected int ascent  = 0;
     protected int descent = 0;
     protected int leading = 0;

     protected boolean parse = true;

     protected Graphics lg = null;

     Vector list = new Vector(8,4);

/*********************
**
** Constructors
**
*********************/

     public TextLine() { }


     public TextLine(String s) { 
            this.text = s;
	  }

     public TextLine(String s, Font f) { 
            this(s);
            font      = f;
            if(font == null) return;
            fontname  = f.getName();
            fontstyle = f.getStyle();
            fontsize  = f.getSize();
	  }

     public TextLine(String s, Font f, Color c, int j) {
            this(s,f);
            color  = c;
            justification = j;
 	  }

     public TextLine(String s, Color c) { 
            this(s);
            color = c;
	  }

     public TextLine(Font f, Color c, int j) {
            font      = f;
            color  = c;
            justification = j;

            if(font == null) return;
            fontname  = f.getName();
            fontstyle = f.getStyle();
            fontsize  = f.getSize();
 	  }

/*****************
**
** Public Methods
**
*****************/
     public TextLine copyState() {
            return new TextLine(font,color,justification);
     }

     public void copyState(TextLine t) {
            if(t==null) return;

            font  = t.getFont();
            color = t.getColor();
            justification = t.getJustification();
       
            if(font == null) return;
            fontname  = font.getName();
            fontstyle = font.getStyle();
            fontsize  = font.getSize();

            parse = true;
     }

     public void setFont(  Font f   ) { 
            font      = f;
            fontname  = f.getName();
            fontstyle = f.getStyle();
            fontsize  = f.getSize();
            parse = true; 

     }
     public void setText(  String s ) { 
            text   = s;
            parse = true; 
     }
     public void setColor( Color c  ) { 
            color = c; 
     }
     public void setJustification( int i ) {
          switch (i) {
          case CENTER:
                    justification = CENTER;
                    break;
          case LEFT: default:
                    justification = LEFT;
                    break;
          case RIGHT:
                    justification = RIGHT;
                    break;
          }
	}

     public Font   getFont()  { 
                                return font; 
     }
     public String getText()  { 
                               return text; 
     }
     public Color  getColor() { 
                               return color; 
     }
     public int    getJustification() { 
                                return justification; 
     }

     public FontMetrics getFM(Graphics g) {
         if(g==null) return null;

         if(font==null) return g.getFontMetrics();
         else           return g.getFontMetrics(font);
     }


     public int charWidth(Graphics g, char ch) {
         FontMetrics fm;
         if(g==null) return 0;

         if(font==null) fm =  g.getFontMetrics();
         else           fm =  g.getFontMetrics(font);
         
         return fm.charWidth(ch);
     }


     public int getWidth(Graphics g) {

         parseText(g);

         return width;

     }

     public int getHeight(Graphics g) {

         parseText(g);

         return leading+ascent+descent;

     }



     public int getAscent(Graphics g) {
         if(g == null) return 0;

         parseText(g);

         return ascent;
     }

     public int getDescent(Graphics g) {
         if(g == null) return 0;

         parseText(g);

         return descent;
      
     }

     public int getLeading(Graphics g) {
         if(g == null) return 0;

         parseText(g);

         return leading;
      
     }

     public void parseText(Graphics g) {

         TextState current = new TextState();
         char ch;
         Stack state = new Stack();
         int w = 0;

         if(lg != g) parse = true;
         lg = g;

         if(!parse) return;

         parse = false;
         width   = 0;
         leading = 0;
         ascent  = 0;
         descent = 0;;

         if( text == null || g == null) return;

         list.removeAllElements();


         if(font == null)  {
                             current.f = g.getFont();
			   }
         else              {
                             current.f = font;
			   }

         state.push(current);
         list.addElement(current);
         
         g.getFontMetrics(current.f);

         for(int i=0; i<text.length(); i++) {
             ch = text.charAt(i);

             switch (ch) {

             case '/':
                      i++;
                      if(i<text.length()) current.s.append(text.charAt(i));
                      break;
/*
**                    Push the current state onto the state stack
**                    and start a new storage string
*/
	     case '{':
                      w = current.getWidth(g);
                      if(!current.isEmpty()) {
                           current = current.copyState();
                           list.addElement(current);
		      }

                      state.push(current);
                      current.x += w;
                      break;
/*
**                    Pop the state off the state stack and set the current
**                    state to the top of the state stack
*/
	     case '}':
                      w = current.x + current.getWidth(g);
                      state.pop();
                      current = ((TextState)state.peek()).copyState();
                      list.addElement(current);
                      current.x = w;
                      break;
	     case '^':
                      w = current.getWidth(g);
                      if(!current.isEmpty()) {
                           current = current.copyState();
                           list.addElement(current);
		      }
                      current.f = getScriptFont(current.f);
                      current.x += w;
                      current.y -= (int)((double)(current.getAscent(g))*sup_offset+0.5);
                      break;
 	     case '_':
                      w = current.getWidth(g);
                      if(!current.isEmpty()) {
                           current = current.copyState();
                           list.addElement(current);
		      }
                      current.f = getScriptFont(current.f);
                      current.x += w;
                      current.y += (int)((double)(current.getDescent(g))*sub_offset+0.5);
                      break;

             default: 
                      current.s.append(ch);
                      break;
	     }
	   }



         for(int i=0; i<list.size(); i++) {
            current = ((TextState)(list.elementAt(i)));

            if( !current.isEmpty() ) {
               width  += current.getWidth(g);
               ascent  = Math.max(ascent, Math.abs(current.y) + 
                                          current.getAscent(g));
               descent = Math.max(descent, Math.abs(current.y) + 
                                          current.getDescent(g));
               leading  = Math.max(leading, current.getLeading(g));
	     }
         }

         return;

     }

     public boolean isNull() {
        return (text==null);
      }


     public void draw(Graphics g, int x, int y, int j) {
         justification = j;
         draw(g,x,y);
       }

     public void draw(Graphics g, int x, int y) {
         TextState ts;
         int xoffset = x;
         int yoffset = y;

         if(g == null || text == null) return;
     
         Graphics lg = g.create();

         parseText(g);

         if(font  != null) lg.setFont(font);
         if(color != null) lg.setColor(color);

         if(justification == CENTER ) {
               xoffset = x-width/2;
         } else 
         if(justification == RIGHT ) {      
               xoffset = x-width;
	 }


         for(int i=0; i<list.size(); i++) {
              ts = ((TextState)(list.elementAt(i)));
              if(ts.f != null) lg.setFont(ts.f);
              if(ts.s != null) 
                   lg.drawString(ts.toString(),ts.x+xoffset,ts.y+yoffset);
	 }

         lg.dispose();

         lg = null;

       }


      public String getFontName()  { return fontname; }
      public int    getFontStyle() { return fontstyle; }
      public int    getFontSize()  { return fontsize; }

      public void setFontName(String s) {  fontname  = s; rebuildFont(); }
      public void setFontStyle(int i)   {  fontstyle = i; rebuildFont(); }
      public void setFontSize(int i)    {  fontsize  = i; rebuildFont(); }



      private void rebuildFont() {
         parse = true;

         if( fontsize <= 0 || fontname == null   ) {
              font = null;
	    } else {

              font = new Font(fontname, fontstyle, fontsize);
	    }
       }


        public Font getScriptFont(Font f) {
             int size;

             if(f == null) return f;

             size = f.getSize();

             if(size <= MINIMUM_SIZE) return f;

             size = (int)((double)(f.getSize())*script_fraction + 0.5);

             if(size <= MINIMUM_SIZE) return f;

             return new Font(f.getName(), f.getStyle(), size);
	   }
   }



class TextState extends Object { 
      Font f         = null;
      StringBuffer s = null;
      int x          = 0;
      int y          = 0;

      
      public TextState() {
              s = new StringBuffer();
	    }


      public TextState copyAll() {
             TextState tmp = copyState();
             if(s.length()==0) return tmp;
             for(int i=0; i<s.length(); i++) { tmp.s.append(s.charAt(i)); }
             return tmp;
	   }


      public TextState copyState() {
             TextState tmp = new TextState();
             tmp.f = f;
             tmp.x = x;
             tmp.y = y;
             return tmp;
	   }


      public String toString() {
             return s.toString();
	   }


      public boolean isEmpty() {
           return (s.length() == 0);
	 }

      public int getWidth(Graphics g) {

           if(g == null || f == null || s.length()==0 ) return 0;

           return g.getFontMetrics(f).stringWidth(s.toString());
      }

      public int getHeight(Graphics g) {
           if(g == null || f == null ) return 0;
           
           return g.getFontMetrics(f).getHeight();
	 }
      public int getAscent(Graphics g) {
           if(g == null || f == null ) return 0;
           
           return g.getFontMetrics(f).getAscent();
	 }
      public int getDescent(Graphics g) {
           if(g == null || f == null ) return 0;
           
           return g.getFontMetrics(f).getDescent();
	 }
      public int getLeading(Graphics g) {
           if(g == null || f == null ) return 0;
           
           return g.getFontMetrics(f).getLeading();
	 }
}
