package graphics.grapher;

import javax.swing.*;
import java.awt.*;
import java.net.URL;
import java.util.Vector;


public class GraphComponent extends JComponent {

    private Image imageData = null;
/*
**  The axis we are controlling
*/
    private Vector axis = new Vector(4);
/*
**  The datasets we are controlling
*/
    private Vector dataset = new Vector(10);
/*
**  The markers that have been loaded
*/
    private Markers marker;



/*
**  The border around the entire plot. Allows slopover from axis labels
**  legends etc.
*/
    private int borderTop = 10;
    private int borderBottom = 10;
    private int borderLeft = 10;
    private int borderRight = 10;


/*
**  Do you want a grid over the data window.
*/
    private boolean drawgrid = true;
    private Color gridcolor = Color.BLACK;
/*
**  Do you want grid lines along the zero's
*/
    private boolean drawzero = true;
    private Color zerocolor = Color.ORANGE;
/*
**  The rectangle where the data will be plotted
*/
    private Rectangle datarect = new Rectangle();
/*
**  Blank the screen when the update method is called? Normally
**  only modified for special effects
*/
    private boolean clearAll = true;
/*
**  Paint it All ie axis, data etc. Normally
**  only modified for special effects
*/
    private boolean paintAll = true;


    /**
     * *************** * *  Public Methods * *****************
     */


/*
**  loadDataSet( double data[], int n )  
**
**  Load a DataSet from an Array. The data is assumed to be stored
**  in the form  x,y,x,y,x,y.... A local copy of the data is made.
**
**  If everything goes well it returns 'true'
**
**  double data[]: The data to plot stored in the form x,y,x,y...
**  int   n:      The number of data points. This means that the
**                minimum length of the data array is 2*n.
*/

    public DataSet loadDataSet(double data[], int n) {
        DataSet d;
        try {
            d = new DataSet(data, n);
            dataset.addElement(d);
            d.g2d = this;
        } catch (Exception e) {
            System.out.println("Failed to load Data set ");
            e.printStackTrace();
            return null;
        }
        return d;
    }

/*
**  attachDataSet( DataSet d )  
**      Add a data set to the graph
*/

    public void attachDataSet(DataSet d) {

        if (d != null) {
            dataset.addElement(d);
            d.g2d = this;
        }
    }
/*
** deleteDataSet( DataSet d ) 
**        Delete all pointers to the local copy for the data set.
*/

    public void deleteDataSet(DataSet d) {
        if (d != null) {
            if (d.xaxis != null) d.xaxis.detachDataSet(d);
            if (d.yaxis != null) d.yaxis.detachDataSet(d);
            dataset.removeElement(d);
        }
    }
/*
** createAxis( int position )
**            create an axis
*/
    public Axis createAxis(int position) {
        Axis a;

        try {
            a = new Axis(position);
            axis.addElement(a);
        } catch (Exception e) {
            System.out.println("Failed to create Axis");
            e.printStackTrace();
            return null;
        }

        return a;
    }
/*
** attachAxis( Axis a )
**           attach an existing axis
*/
    public void attachAxis(Axis a) {

        if (a == null) return;

        try {
            axis.addElement(a);
        } catch (Exception e) {
            System.out.println("Failed to attach Axis");
            e.printStackTrace();
        }

    }

/*
** deleteAxis( Axis a )
**            delete an axis
*/
    public void deleteAxis(Axis a) {

        if (a != null) {
            a.detachAll();
            axis.removeElement(a);
        }
    }
/*
** Vector getMarkerVector(int m)
**         Given the marker index (m > 0 ) return the vector containing the
**         points to stroke the marker
*/
    public Vector getMarkerVector(int m) {

        if (marker == null) return null;


        if (marker.number > 0 && m < marker.number) {

            return marker.vert[m - 1];
        }

        return null;
    }
/*
** boolean loadMarkerFile(URL name)
**        Load the marker file given the files URL. Markers are NOT
** predefined but must be loaded from a Marker definition file
*/
    public boolean loadMarkerFile(URL name) {
        //name = new URL("file://hd/current/dewitt/diffraction%20%B9/marker.txt");
        System.out.println("Loading marker file");
        marker = new Markers(name);

        if (marker.number <= 0) return false;

        return true;

    }
/*
** paint(Graphics g)
**                  Paint it ALL.
*/
    public void paintComponent(Graphics g) {
        Rectangle r = getBounds();
        r.setSize(getSize());
        r.setLocation(getLocation());
        if (!paintAll) return;

        r.x += borderLeft;
        r.y += borderTop;
        r.width -= borderLeft + borderRight;
        r.height -= borderBottom + borderTop;
        if (!axis.isEmpty())
            r = drawAxis(g, r);
        //else
        //    drawBorder(lg, r.x, r.y);

        if (imageData == null)
            drawData(g, r);
        else
            g.drawImage(imageData, r.x, r.y, r.width, r.height, this);

       // g.dispose();
       // super.paintComponent(g);

    }

    private void drawData(Graphics lg, Rectangle r) {
        int i;
        if (!dataset.isEmpty()) {

            for (i = 0; i < dataset.size(); i++) {
                ((DataSet) dataset.elementAt(i)).draw_legend(lg, r);
            }

            lg.clipRect(r.x, r.y, r.width, r.height);

            datarect.x = r.x;
            datarect.y = r.y;
            datarect.width = r.width;
            datarect.height = r.height;

            for (i = 0; i < dataset.size(); i++) {
                ((DataSet) dataset.elementAt(i)).draw_data(lg, r);
            }
        }
    }





    /**
     * *************** * * Protected Methods * *****************
     */
/*
**  Rectangle drawAxis(Graphics g, Rectangle r)
**
**  Draw the Axis. As each axis is drawn and aligned less of the canvas
**  is avaliable to plot the data. The returned Rectangle is the canvas
**  area that the data is plotted in.
*/
    protected Rectangle drawAxis(Graphics g, Rectangle r) {
        Axis a;
        int waxis;
        int x = r.x;
        int y = r.y;
        int width = r.width;
        int height = r.height;

// Calculate the available area for the data
        for (int i = 0; i < axis.size(); i++) {
            a = ((Axis) axis.elementAt(i));

            waxis = a.getAxisWidth(g);

            switch (a.getAxisPos()) {
                case Axis.LEFT:
                    x += waxis;
                    width -= waxis;
                    break;
                case Axis.RIGHT:
                    width -= waxis;
                    break;
                case Axis.TOP:
                    y += waxis;
                    height -= waxis;
                    break;
                case Axis.BOTTOM:
                    height -= waxis;
                    break;
            }
        }

        //if (frame) drawBorder(g, x, y);

// Now draw the axis in the order specified aligning them with the final
// data area.
        for (int i = 0; i < axis.size(); i++) {
            a = ((Axis) axis.elementAt(i));

            a.data_window = new Dimension(width, height);

            switch (a.getAxisPos()) {
                case Axis.LEFT:
                    r.x += a.width;
                    r.width -= a.width;
                    a.positionAxis(r.x, r.x, y, y + height);
                    if (r.x == x) {
                        a.gridcolor = gridcolor;
                        a.drawgrid = drawgrid;
                        a.zerocolor = zerocolor;
                        a.drawzero = drawzero;
                    }

                    a.drawAxis(g);

                    a.drawgrid = false;
                    a.drawzero = false;

                    break;
                case Axis.RIGHT:
                    r.width -= a.width;
                    a.positionAxis(
                            r.x + r.width, r.x + r.width, y, y +
                            height);
                    if (r.x + r.width == x + width) {
                        a.gridcolor = gridcolor;
                        a.drawgrid = drawgrid;
                        a.zerocolor = zerocolor;
                        a.drawzero = drawzero;
                    }

                    a.drawAxis(g);

                    a.drawgrid = false;
                    a.drawzero = false;

                    break;
                case Axis.TOP:
                    r.y += a.width;
                    r.height -= a.width;
                    a.positionAxis(x, x + width, r.y, r.y);
                    if (r.y == y) {
                        a.gridcolor = gridcolor;
                        a.drawgrid = drawgrid;
                        a.zerocolor = zerocolor;
                        a.drawzero = drawzero;
                    }

                    a.drawAxis(g);

                    a.drawgrid = false;
                    a.drawzero = false;


                    break;
                case Axis.BOTTOM:
                    r.height -= a.width;
                    a.positionAxis(
                            x, x + width, r.y + r.height, r.y + r.height);
                    if (r.y + r.height == y + height) {
                        a.gridcolor = gridcolor;
                        a.drawgrid = drawgrid;
                        a.zerocolor = zerocolor;
                        a.drawzero = drawzero;
                    }

                    a.drawAxis(g);

                    a.drawgrid = false;
                    a.drawzero = false;


                    break;
            }
        }

        return r;
    }
/*
** drawFrame(Graphics g, int x, int y, int width, int height)
**    If requested draw a frame around the data area.
*/
    protected void drawBorderDep(
            Graphics g, int x, int y) {
        Dimension size = getSize();
        g.drawRect(x, y, size.width, size.height);
    }


    public Vector getDataset() {
        return dataset;
    }

    public void setDataset(Vector dataset) {
        this.dataset = dataset;
    }

    public boolean isClearAll() {
        return clearAll;
    }

    public void setClearAll(boolean clearAll) {
        this.clearAll = clearAll;
    }

    public boolean isPaintAll() {
        return paintAll;
    }

    public void setPaintAll(boolean paintAll) {
        this.paintAll = paintAll;
    }

    public Image getImageData() {
        return imageData;
    }
    public void setImageData(Image imageData) {
        this.imageData = imageData;
    }
    public Dimension getPreferredSize(){
        if (imageData == null)
         return new Dimension(200,200);
        return new Dimension(imageData.getWidth(this),imageData.getHeight(this));
    }
    public Dimension getMinimumSize(){
        return new Dimension(50,50);
    }
}

