package graphics.grapher;
import java.awt.*;
import java.util.Vector;

/*************************************************************************
**
**    Class  Contour
**                                              Version 1.0   February  1995
**
**************************************************************************
**    Copyright (C) 1996 Leigh Brookshaw
**
**    This program is free software; you can redistribute it and/or modify
**    it under the terms of the GNU General Public License as published by
**    the Free Software Foundation; either version 2 of the License, or
**    (at your option) any later version.
**
**    This program is distributed in the hope that it will be useful,
**    but WITHOUT ANY WARRANTY; without even the implied warranty of
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**    GNU General Public License for more details.
**
**    You should have received a copy of the GNU General Public License
**    along with this program; if not, write to the Free Software
*    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**************************************************************************
**
** This class extends the interactive graphics class to incorporate
** contours.
**
*************************************************************************/


public class Contour extends G2Dint {

/*************
**
** Constants
**
*************/
	
/*
**   The minimum length of a curve before it gets a label
*/
	 static final int MINCELLS = 30;
/*
**   Default number of contour levels
*/
	 static final int NLEVELS = 12;
	 

/**********************
**
** Protected Variables
**
***********************/
	 
/*
**   Dimensions of the grid
*/
     protected int nx;
     protected int ny;
/*
**   Vector array containing the Contour curves.
**   Each index in the array contains curves at a given
**   contour level
*/
     protected Vector curves[];

/*
**   autoLevels
**              If set we calculate the contour levels based on
**   the data minimum and maximum
*/
     protected boolean autoLevels;
/*
**   logLevels
**             If true the contour levels are calculated in
**              logarithmic intervals
*/
     protected boolean logLevels;
/*
**   gridLimits
**             If true the limits of the plot are the limits of the
**   data grid not the limits of the contours!
*/
     protected boolean gridLimits;
/*
**   levels
**          The array of contour levels
*/
     protected double levels[];
/*
**   labels
**         The label for each contour level
*/
     protected TextLine labels[];
/*
**   Font to use in drawing Labels
*/
     protected Font labelfont;
/*
**   Color to use in drawing Labels
*/
     protected Color labelcolor;

/*
**   LabelLevels
**            Which levels will get labels. if equal to 1 every level
**   gets a label. Equal 2 every second level etc. if it equals 0
**   no labels are displayed.   
*/
     protected int labelLevels;
/*
**   DrawLabels
**              If false labels are not drawn
*/
     protected boolean drawlabels;
/*
**   autoLabels
**              If true we will calculate the labels for each
**   contour level
*/
     protected boolean autoLabels;
/*
**   grid
**         The data grid a 2D array stored in linear form.
**   It is assumed that [0,0] is the bottom left corner
**   and the data is stored by row.
*/
     protected double grid[];

/*
**   The limits of the data grid
*/
     protected double xmin;
     protected double xmax;
     protected double ymin;
     protected double ymax;
/*
**   The limits of the grid values
*/
     protected double zmin;
     protected double zmax;

/*
**   Parameters for `nice' levels (Not implemented yet)
*/ 
     private double level_start;
     private double level_exponent;
     private double level_step;
     
/****************
**
** Constructors
**
****************/

     public Contour() {

           grid = null;
           xmin = 0.0;
           xmax = 0.0;
           ymin = 0.0;
           ymax = 0.0;
           zmin = 0.0;
           zmax = 0.0;

           nx = 0;
           ny = 0;

           levels = new double[NLEVELS];
           labels = new TextLine[NLEVELS];

           autoLevels  = true;
           logLevels   = false;
           gridLimits  = false;
           autoLabels  = true;
           labelfont   = new Font("Helvetica",Font.PLAIN,12);
           labelcolor  = Color.blue;
           labelLevels = 1;
           drawlabels = true;

           curves = null;


           level_start    = 0.0;
           level_step     = 0.0;
           level_exponent = 0.0;

     }

/***********
**
** Methods
**
***********/

     
/*
** setRange(double, double, double, double)
**      Set the range of the grid
*/


     public void setRange(double xmin,double xmax,double ymin,double ymax) {

            if( xmin >= xmax || ymin >= ymax ) return;

            this.xmin = xmin;
            this.xmax = xmax;
            this.ymin = ymin;
            this.ymax = ymax;
     }

/*
**   getRange()
**              Return the range of the grid
*/
     public double[] getRange() {
            double d[] = new double[4];
            d[0] = xmin;
            d[1] = xmax;
            d[2] = ymin;
            d[3] = ymax;

            return d;
      }
/*
**   getDim()
**           return the dimensions of the grid
*/
     public int[] getDim() {
            int i[] = new int[2];
         
            i[0] = nx;
            i[1] = ny;

            return i;
      }

/*
**   getGrid()
**              Return the grid
*/
     public double[] getGrid() { return grid; }

/*
**   setLevels(double levels[], int nl)
**                Manually set the contour levels.
*/
     public void setLevels(double levels[], int nl) {
           int i;
           if( levels == null || nl <= 0 ) return;

           detachCurves();
           curves = null;

           autoLevels = false;

           this.levels = new double[nl];
           
           System.arraycopy(levels,0,this.levels,0,nl);

           labels = new TextLine[nl];
           for(i=0; i<labels.length; i++) {
             labels[i] = new TextLine( String.valueOf( levels[i] ) );
           } 
     }
/*
**  setLabels(TextLine labels[], int nl)
**            Manually set the Contour labels.
*/
     public void setLabels(TextLine labels[], int nl) {
           if( labels == null || nl <= 0 ) return;

           autoLabels = false;
           this.labels = new TextLine[nl];
           
           System.arraycopy(labels,0,this.labels,0,nl);
     }

/*
**   setLabelFont(Font f)
**              Set the font to be used with All the labels
*/
     public void setLabelFont(Font f) {
           labelfont = f;
     }

/*
**   setLabelColor(Color c)
**               Set the Color to be used with all the labels.
*/
     public void setLabelColor(Color c) {
           labelcolor = c;
     }
     
/*
**   setGrid(double grid[],int nx,int ny)
**      Set the grid to be contoured.
*/
     public void setGrid(double grid[],int nx,int ny) {
        this.grid = grid;
        this.nx   = nx;
        this.ny   = ny;

        zrange();
        calcLevels();

     }
    
/*
**   getLevels()
**              Return the contour levels.
*/
     public double[] getLevels() { return levels; }
/*
**   setLimitsToGrid(boolean b)
**          If true the limits of the plot will be the grid limits.
**          If false the limits of the plot will be the contours.
*/
     
     public void setLimitsToGrid(boolean b) { gridLimits = b; }
/*
**   setLabelLevels(int i)
**           Set the levels that are to have labels.
**           If 0 no labels are drawn
**           if 1 every level gets a label
**           If 2 every 2nd level gets a label
**           etc.
*/
     public void setLabelLevels(int i) { 
          if(i<=0) labelLevels = 0;
          else     labelLevels = i;
	}

/*
**   setLogLevels(boolean b)
**       If true contour levels are calculated on a log scale
*/
     public void setLogLevels(boolean b) {
               logLevels = b;

               if( zmin <= 0.0 || zmax <= 0.0 ) logLevels = false;
	     }

/*
**   setNLevels(int l)
**       Set the number of contour levels.
*/
     public void setNLevels(int l) {
         if(l <= 0) return;
         
         levels = new double[l];
         
         calcLevels();

         detachCurves();
         curves = null;
     }

/******************
**
** Private Methods
**
*******************/

/*
**  calcLevels()
**              Calculate the contour levels
*/
     private void calcLevels() {
       int i;
       int l;
       
       if(!autoLevels) return;

       if(levels == null) levels = new double[NLEVELS];
       labels = new TextLine[levels.length];
       // Nice label steps not implemented yet
       //levelStep();

       
       if( logLevels ) {
          double inc = Math.log(zmax-zmin)/
                      (double)(levels.length+1);
          try {
             for(i=0; i<levels.length; i++) levels[i] = zmin + 
                          Math.pow(Math.E,(double)(i+1)*inc);
	   } catch (Exception e) {
             System.out.println("Error calculateing Log levels!");
             System.out.println("... calculating linear levels instead");
             logLevels = false;
             calcLevels();
           }
	} else {
          double inc = (zmax-zmin)/(double)(levels.length+1);   
          for(i=0; i<levels.length; i++) levels[i] = zmin + (double)(i+1)*inc;
        }
     }
/*
**  calcLabels()
**              Calculate the labels
*/
     private void calcLabels() {
        int i;
        if( !autoLabels ) return;

        labels = new TextLine[levels.length];


        for(i=0; i<labels.length; i++) {
             labels[i] = new TextLine( String.valueOf(levels[i] ) );
        } 
     }

/*
**   zrange()
**           Calculate the range of the grid
*/

     private void zrange() {
        int i;
     
        zmin = grid[0];
        zmax = grid[1];
        for( i=0; i<grid.length; i++) {

             zmin = Math.min(zmin,grid[i]);
             zmax = Math.max(zmax,grid[i]);

        }

        System.out.println("Data range: zmin="+zmin+", zmax="+zmax);

        if(zmin == zmax) {
           System.out.println("Cannot produce contours of a constant surface!");
	 }

        if(zmin <= 0 || zmax <= 0) logLevels = false;


      }
/*
**   paintFirst(Graphics g, Rectangle r)
**        before anything is painted calculate the contours.
*/

      public void paintFirst(Graphics g, Rectangle r) {

         //System.out.println("paintFirst called");


         if( curves == null ) { 
                                  calculateCurves();
                                  calcLabels();
				}


         if(gridLimits && !userlimits ) {
           if( xaxis != null ) {
                if(xaxis.minimum > xmin ) xaxis.minimum = xmin;
                if(xaxis.maximum < xmax ) xaxis.maximum = xmax;
	      }

           if( yaxis != null ) {
                if(yaxis.minimum > ymin ) yaxis.minimum = ymin;
                if(yaxis.maximum < ymax ) yaxis.maximum = ymax;
	      }
         } else 
         if( getDataset().isEmpty() ) {
           if( xaxis != null ) {
                xaxis.minimum = xmin;
                xaxis.maximum = xmax;
	      }

           if( yaxis != null ) {
                yaxis.minimum = ymin;
                yaxis.maximum = ymax;
	      }
         }


      }
/*
**    attachCurves()
**        Attach all the curves to the graph and to the axes
*/
      private void attachCurves() {
         int i;
         if(curves == null) return;

         for(i=0; i<curves.length; i++) attachCurves(curves[i]);

      }
/*
**    attachCurves(Vector v)
**        Attach all the curves from a given level to the graph and to the axes
*/

      private void attachCurves(Vector v) {
         int j;
         if(v == null) return;
         for(j=0; j<v.size(); j++) {
               attachDataSet((DataSet)(v.elementAt(j)));
               if(xaxis != null) 
                        xaxis.attachDataSet((DataSet)(v.elementAt(j)));
               if(yaxis != null) 
                        yaxis.attachDataSet((DataSet)(v.elementAt(j)));
	     }
      }
/*
**    detachCurves()
**                 Detach All the curves from the graph and the axes.
*/
      private void detachCurves() {
         int i;
         if(curves == null) return;

         for(i=0; i<curves.length; i++) detachCurves(curves[i]);
      }
/*
**    detachCurves()
**                 Detach all the curves from a given level from 
**                 the graph and the axes.
*/
      private void detachCurves(Vector v) {
         int j;
         if(v == null) return;
         for(j=0; j<v.size(); j++) {
               deleteDataSet((DataSet)(v.elementAt(j)));
               if(xaxis != null) 
                        xaxis.detachDataSet((DataSet)(v.elementAt(j)));
               if(yaxis != null) 
                        yaxis.detachDataSet((DataSet)(v.elementAt(j)));
	     }
      }

/*
**    paintLast(Graphics g, Rectangle rect)
**          Last thing to be done is to draw the contour labels if required.
*/
      public void paintLast(Graphics g, Rectangle rect) {
              int i, j;
              int points;
              int index;
              Vector v;
              DataSet ds;
              double point[] = new double[2];
              int x;
              int y;
              Color current = g.getColor();
              Rectangle r = new Rectangle();

              if( xaxis == null || yaxis == null || labels == null || 
                  labelLevels == 0 || !drawlabels ) return;

              for(i=0; i<levels.length; i++) {
                 if( labels[i] != null && !labels[i].isNull() &&
                     i%labelLevels == 0 ) {
                    labels[i].setFont(labelfont);
                    labels[i].setColor(labelcolor); 
                    v = curves[i];
                    for(j=0; j<v.size(); j++) {
                        ds =  (DataSet)(v.elementAt(j));
                        points = ds.dataPoints();
                        index = (int)(Math.random()*(double)MINCELLS);
                        while ( points > MINCELLS ) {
                             point = ds.getPoint(index);
                             x = xaxis.getInteger(point[0]);
                             y = yaxis.getInteger(point[1]);

                             r.width  = labels[i].getWidth(g);
                             r.height = labels[i].getAscent(g);
                             r.x = x - r.width/2;
                             r.y = y - r.height/2;

                             g.setColor(getBackground());
                             g.fillRect(r.x, r.y, r.width, r.height);

                             g.setColor(current);
                             
                             labels[i].draw(g, r.x, r.y+r.height, 
                                            TextLine.LEFT);

                             points -= MINCELLS;
                             index += MINCELLS;
                        }                        
		      }
		  }
	       }
	    }
/*
**   calculateCurves()
**        Calculate the contours and attach them to the graph and axes.
*/

     protected void calculateCurves() {
          int i;
          int j;
          double data[];
          double xscale = (xmax-xmin)/(double)(nx-1);
          double yscale = (ymax-ymin)/(double)(ny-1);
          
          IsoCurve isocurve = new IsoCurve(grid,nx,ny);


          if( curves != null) {
                               detachCurves();
                               curves = null;
			     }

          curves = new Vector[levels.length];
          
          for(i=0; i<levels.length; i++) {
              isocurve.setValue(levels[i]);

              curves[i] = new Vector();

              while( (data = isocurve.getCurve()) != null ) {
                for(j=0; j<data.length; ) {
                   data[j] = xmin + data[j]*xscale;
                   j++;
                   data[j] = ymin + data[j]*yscale;
                   j++;
		 }

                try {
                  curves[i].addElement(new DataSet(data, data.length/2));
                } catch (Exception e) {
                  System.out.println("Error loading contour into DataSet!");
                  System.out.println("...Contour Level "+levels[i]);
                }
              }

              attachCurves(curves[i]);

              //repaint();
          }


	}
	

// Following methods not implemented yet. They will be used to calculate
// 'Nice' contour steps/levels.
	

       private void levelStep() {
          double min;
          double max;

          if( logLevels ) {
               min = Math.log(zmin);
               max = Math.log(zmax);
	      } else {
               min = zmin;
               max = zmax;
	      }




        if (Math.abs(min) > Math.abs(max) ) 
         level_exponent = ((int)Math.floor(log10(Math.abs(min))/3.0) )*3;
        else
         level_exponent = ((int)Math.floor(log10(Math.abs(max))/3.0) )*3;

        level_step = RoundUp( (min-max)/levels.length);
        level_start = Math.floor( min/level_step )*level_step;
      }

      private double RoundUp( double val ) {
          int exponent;
          int i;

          exponent = (int)(Math.floor( log10(val) ) );

          if( exponent < 0 ) {
             for(i=exponent; i<0; i++) { val *= 10.0; }
          } else {
             for(i=0; i<exponent; i++) { val /= 10.0; }
          }

          if( val > 5.0 )     val = 10.0;
          else
          if( val > 2.0 )     val = 5.0;
          else
          if( val > 1.0 )     val = 2.0;
          else
                              val = 1.0;

          if( exponent < 0 ) {
             for(i=exponent; i<0; i++) { val /= 10.0; }
          } else {
             for(i=0; i<exponent; i++) { val *= 10.0; }
          }
          
          return val;

      }                  


      private double log10( double a ) throws ArithmeticException {
           return Math.log(a)/2.30258509299404568401;
      }


   }
