package classUtils.pack.util;

import classUtils.putils.ClassPathBean;

import java.io.*;
import java.util.*;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.regex.Pattern;

/**
 * A package explorer implementation.
 * <p>
 * WARNING: Sealed Jars aren't supported yet.
 * 
 * @version 1.0
 * @author Cristiano Sadun
 */
public class SimpleClassPackageExplorer implements ClassPackageExplorer {
	
	private String classPath;
	private Pattern [] dirPatterns;
	private StateInfoSupport sis;
	private CPoolReader cpr;
	private boolean errorOccurred;
	
	/*
	 * A map (package name -> List(File files containing classes in that package)).
	 */
	private Map filesPerPackage;
	
	/* 
	 * A map (package name -> Integer ).
	 */
	private Map packageStatus;
	
	/*
	 * A map (package name -> List(class names).
	 */
	private Map classesPerPackage;

    private static FileFilter classFileFilter = new ExtensionFileFilter("class");
	private static FileFilter dirFileFilter = new DirectoryFileFilter();
	
	/**
	 * Create a SimpleClassPackageExplorer on the system class path.
	 */
	public SimpleClassPackageExplorer() {
		this(System.getProperty("java.class.path"));
	}
	
	/**
	 * Create a SimpleClassPackageExplorer on the given class path.
	 * 
	 * @param  classPath string to iterate on
	 */	
	public SimpleClassPackageExplorer(String classPath) {
		this(classPath, new String[] { ".*" });
        ClassPathBean cpb = ClassPathBean.restore();
        cpb.addClassPath(classPath);
		this.cpr=new CPoolReader(cpb);
	}

	/**
	 * Create a SimpleClassPackageExplorer on the given class path.
	 * 
	 * @param  classPath string to iterate on
	 * @param classDirs an array of regular expression for the names of subdirectories to search for;
	 *         these patterns are not considered for JARs.
	 */	
	public SimpleClassPackageExplorer(String classPath, String[] classDirs) {
		this.classPath=classPath;
		dirPatterns=new Pattern[classDirs.length];
		for(int i=0;i<classDirs.length;i++) {
			dirPatterns[i]=Pattern.compile(classDirs[i]);
		}
	}

	/**
	 * @see classUtils.pack.util.ClassPackageExplorer#listPackage(String)
	 */
	public String[] listPackage(String packageName) {
		return listPackage(packageName, IN_DIRECTORY+IN_JAR+IN_JAR_SEALED);
	}

	/**
	 * @see classUtils.pack.util.ClassPackageExplorer#listPackageNames()
	 */
	public String[] listPackageNames() {
		return listPackageNames(false);
	}

	/**
	 * Method buildInfo.
	 */
	private void buildInfo() {
		sis = new StateInfoSupport();
		filesPerPackage = new HashMap();
		packageStatus = new HashMap();
		classesPerPackage = new HashMap();
		errorOccurred=false;
		
		for(ClassPathIterator i=new ClassPathIterator(classPath);i.hasNext();) {
			File entry=i.nextEntryFile();
			if (i.isJar())
				scanJarFile(entry);
			else 
				scanDirectory(entry);
		}
	}

	/**
	 * Method scanDirectory.
	 * @param entry
	 */
	private void scanDirectory(File entry) {
		// Is the entry matching one of the patterns?
		boolean matchedOne=false;
		for(int i=0;i<dirPatterns.length;i++) {
			if (dirPatterns[i].matcher(entry.getName()).matches()) {
				matchedOne=true;
				break;
			}
		}
		if (!matchedOne) return; // Skip the entry
		
		// Fetch all the .class files
		File [] classFiles = entry.listFiles(classFileFilter);
		for(int i=0;i<classFiles.length;i++) {
			try {
				//System.out.println("processing "+classFiles[i]);
				processFile(entry, false, new BufferedInputStream(new FileInputStream(classFiles[i])));
			} catch (IOException e) {
				sis.addEntry("Could not open/process class file "+classFiles[i].getAbsolutePath());
				errorOccurred=true;
			}
		}
		
		// Fetch all the subdirectories
		File [] subDirs = entry.listFiles(dirFileFilter);
		
		// Recurse
		for(int i=0;i<subDirs.length;i++) {
			scanDirectory(subDirs[i]);
		}
	}

	/**
	 * Method scanJarFile.
	 * @param entry
	 */
	private void scanJarFile(File entry) {
		try {
			JarFile jf = new JarFile(entry);
			Enumeration e = jf.entries();
			while(e.hasMoreElements()) {
				JarEntry jarEntry = (JarEntry)e.nextElement();
				if (jarEntry.isDirectory()) continue;
				String jarEntryName=jarEntry.getName();
				if (jarEntryName.endsWith(".class")) {
					//System.out.println("processing "+jarEntry.getName());
					processFile(entry, true, new BufferedInputStream(jf.getInputStream(jarEntry)));
				}
			}
		} catch (IOException e) {
			sis.addEntry("Scanning Jar file", "Could not open/process jar file \""+entry+"\"", e);
			errorOccurred=true;
		}
	}

	private void processFile(File entry, boolean isJar, InputStream classDataInputStream) throws IOException {
		
		CPoolReader.ClassFile c = cpr.readClassData(classDataInputStream);
		
		String className = c.getCPClassName(true);
		String pkgName = getPkgName(className);
		
		List files = (List)filesPerPackage.get(pkgName);
		if (files==null) {
			files=new ArrayList();
			filesPerPackage.put(pkgName, files);
		}
		files.add(entry);
		
		Integer status = (Integer)packageStatus.get(pkgName);
		if (status==null) {
			status=new Integer(0);
			packageStatus.put(pkgName, status);
		}


        new Integer(status.intValue() | (isJar ? IN_JAR : IN_DIRECTORY));
        packageStatus.put(pkgName, status);
		
		List classes = (List)classesPerPackage.get(pkgName);
		if (classes==null) {
			classes=new ArrayList();
			classesPerPackage.put(pkgName, classes);
		}
		classes.add(className);
	}

	/**
	 * Method getPkgName.
	 * @param className
	 * @return String
	 */
	private String getPkgName(String className) {
		int i=className.lastIndexOf('.');
		if (i==-1) return className;
		return className.substring(0,i);
	}


	/**
	 * Returns the classPath.
	 * @return String
	 */
	public String getClassPath() {
		return classPath;
	}

	/**
	 * @see classUtils.pack.util.ClassPackageExplorer#getPackageFiles(String)
	 */
	public synchronized File[] getPackageFiles(String packageName) {
		List l = (List)filesPerPackage.get(packageName);
		if (l==null) return new File[0];
		File [] files = new File[l.size()];
		l.toArray(files);
		return files;
	}

	/**
	 * @see classUtils.pack.util.ClassPackageExplorer#getStatus(String)
	 */
	public synchronized int getStatus(String packageName) {
		Integer i = (Integer)packageStatus.get(packageName);
		if (i==null) return -1;
		return i.intValue();
	}

	/**
	 * @see classUtils.pack.util.ClassPackageExplorer#listPackage(String, int)
	 */
	public synchronized String[] listPackage(String packageName, int status) {
		if (status != IN_DIRECTORY + IN_JAR + IN_JAR_SEALED) 
			throw new UnsupportedOperationException("Disaggregation by status not supported yet");
		if (classesPerPackage==null) buildInfo();
		
		List l;
		if ((l=(List)classesPerPackage.get(packageName))==null) return new String[0];
		String [] classes = new String[l.size()];
		l.toArray(classes);
		return classes;
	}

	/**
	 * @see classUtils.pack.util.ClassPackageExplorer#listPackageNames(boolean)
	 */
	public synchronized String[] listPackageNames(boolean rescan) {
		if (rescan || classesPerPackage==null) buildInfo();
		String [] names = new String[classesPerPackage.keySet().size()];
		classesPerPackage.keySet().toArray(names);
		return names;
	}
	
	/**
	 * Sets the classPath.
	 * @param classPath The classPath to set
	 */
	public synchronized void setClassPath(String classPath) {
			this.classPath = classPath;
			buildInfo();
	}

	/**
	 * @see classUtils.pack.util.ClassPackageExplorer#getErrorLog()
	 */
	public synchronized String getErrorLog() {
		if (sis==null) buildInfo();
		return sis.getStateDescription(null);
	}

	/**
	 * A test method
	 */
	public static void main(String args[]) throws Exception {
		//ClassPackageExplorer explorer = new SimpleClassPackageExplorer("c:\\projects\\objectmapper\\classes");
		SimpleClassPackageExplorer explorer = new SimpleClassPackageExplorer("c:\\projects\\jutil\\org.sadun.util.jar");
		//ClassPackageExplorer explorer = new SimpleClassPackageExplorer();
		String [] pkgs = explorer.listPackageNames();
		System.out.println(ObjectLister.getInstance().list(pkgs));
		if (pkgs.length > 0) 
			System.out.println(ObjectLister.getInstance().list(explorer.listPackage(pkgs[0])));
		System.out.println(explorer.sis.getStateDescription(null));
	}

	/**
	 * @see classUtils.pack.util.ClassPackageExplorer#hasErrorOccurred()
	 */
	public synchronized boolean hasErrorOccurred() {
		if (sis==null) buildInfo();
		return errorOccurred;
	}

}
