/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */

package addbk.print.labels;
import javax.swing.*;
import javax.swing.border.LineBorder;
import javax.swing.event.TableModelEvent;
import javax.swing.table.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.print.PageFormat;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;
import java.text.SimpleDateFormat;

public class DymoLabelTable extends JFrame
	implements Printable {

	protected JTable m_table;
	protected LabelTableData m_data;
	protected JLabel m_title;

	protected int m_maxNumPage = 1;

	public DymoLabelTable() {
		super("Label Table");
        setSize(252, 79);

		UIManager.put("Table.focusCellHighlightBorder",
			new LineBorder(Color.black, 1));

		m_data = new LabelTableData();

		m_title = new JLabel(m_data.getTitle(), SwingConstants.CENTER);
		m_title.setFont(new Font("Helvetica",Font.PLAIN,24));
		getContentPane().add(m_title, BorderLayout.NORTH);

		m_table = new JTable();
		m_table.setAutoCreateColumnsFromModel(false);
		m_table.setModel(m_data);

		for (int k = 0; k < m_data.getColumnCount(); k++) {
			DefaultTableCellRenderer renderer = new
				ColoredTableCellRenderer();
			renderer.setHorizontalAlignment(
				LabelTableData.colArray[k].m_alignment);
			TableColumn column = new TableColumn(k,
				LabelTableData.colArray[k].m_width, renderer, null);
			column.setHeaderRenderer(createDefaultRenderer());
			m_table.addColumn(column);
		}

		JTableHeader header = m_table.getTableHeader();
		header.setUpdateTableInRealTime(true);
		header.addMouseListener(new ColumnListener(this));
		header.setReorderingAllowed(true);

		m_table.getColumnModel().addColumnModelListener(m_data);

		setJMenuBar(createMenuBar());

		JScrollPane ps = new JScrollPane();
		ps.getViewport().setBackground(m_table.getBackground());
		ps.getViewport().add(m_table);
		getContentPane().add(ps, BorderLayout.CENTER);
	}

	protected TableCellRenderer createDefaultRenderer() {
		DefaultTableCellRenderer label = new DefaultTableCellRenderer() {
			public Component getTableCellRendererComponent(JTable table, Object value,
							 boolean isSelected, boolean hasFocus, int row, int column) {
				if (table != null) {
					JTableHeader header = table.getTableHeader();
					if (header != null) {
						setForeground(header.getForeground());
						setBackground(header.getBackground());
						setFont(header.getFont());
					}
				}

				setText((value == null) ? "" : value.toString());
				setBorder(UIManager.getBorder("TableHeader.cellBorder"));
				return this;
			}
		};
		label.setHorizontalAlignment(JLabel.CENTER);
		return label;
	}

	protected JMenuBar createMenuBar() {
		JMenuBar menuBar = new JMenuBar();

		JMenu mFile = new JMenu("File");
		mFile.setMnemonic('f');

		JMenuItem mData = new JMenuItem("Retrieve Data...");
		mData.setMnemonic('r');
		ActionListener lstData = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				retrieveData();
			}
		};
		mData.addActionListener(lstData);
		mFile.add(mData);
		mFile.addSeparator();

		// NEW
		JMenuItem mPrint = new JMenuItem("Print...");
		mPrint.setMnemonic('p');
		ActionListener lstPrint = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				Thread runner = new Thread() {
					public void run() {
						printData();
					}
				};
				runner.start();
			}
		};
		mPrint.addActionListener(lstPrint);
		mFile.add(mPrint);

		JMenuItem mPreviewP = new JMenuItem("Print Preview: PORTRAIT");
		mPreviewP.setMnemonic('o');
		ActionListener lstPreview = new ActionListener() 
		{ 
			public void actionPerformed(ActionEvent e)
			{
                            Thread runner = new Thread() {
                              public void run() {
				setCursor(Cursor.getPredefinedCursor(
					Cursor.WAIT_CURSOR));
				new PrintPreview(DymoLabelTable.this,
					m_title.getText()+" preview",
                                        PageFormat.PORTRAIT);
				setCursor(Cursor.getPredefinedCursor(
					Cursor.DEFAULT_CURSOR));
                              }
                            };
                            runner.start();
			}
		};
		mPreviewP.addActionListener(lstPreview);
		mFile.add(mPreviewP);

		JMenuItem mPreviewL = new JMenuItem("Print Preview: LANDSCAPE");
		mPreviewL.setMnemonic('l');
		lstPreview = new ActionListener() 
		{ 
			public void actionPerformed(ActionEvent e)
			{
                            Thread runner = new Thread() {
                              public void run() {
				setCursor(Cursor.getPredefinedCursor(
					Cursor.WAIT_CURSOR));
				new PrintPreview(DymoLabelTable.this,
					m_title.getText()+" preview",
                                        PageFormat.LANDSCAPE);
				setCursor(Cursor.getPredefinedCursor(
					Cursor.DEFAULT_CURSOR));
                              }
                            };
                            runner.start();
			}
		};
		mPreviewL.addActionListener(lstPreview);
		mFile.add(mPreviewL);
		mFile.addSeparator();

		JMenuItem mExit = new JMenuItem("Exit");
		mExit.setMnemonic('x');
		ActionListener lstExit = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				System.exit(0);
			}
		};
		mExit.addActionListener(lstExit);
		mFile.add(mExit);
		menuBar.add(mFile);

		JMenu mView = new JMenu("View");
		mView.setMnemonic('v');
		TableColumnModel model = m_table.getColumnModel();
		for (int k = 0; k < m_data.getColumnCount(); k++) {
			JCheckBoxMenuItem item = new JCheckBoxMenuItem(
				LabelTableData.colArray[k].m_title);
			item.setSelected(true);
			TableColumn column = model.getColumn(k);
			item.addActionListener(new ColumnKeeper(column,
				LabelTableData.colArray[k]));
			mView.add(item);
		}
		menuBar.add(mView);

		return menuBar;
	}

	// NEW
	public void printData() {
		try {
			PrinterJob prnJob = PrinterJob.getPrinterJob();
			prnJob.setPrintable(this);
			if (!prnJob.printDialog())
				return;
			m_maxNumPage = 1;
			prnJob.print();
		}
		catch (PrinterException e) {
			e.printStackTrace();
		}
	}

	// NEW
	public int print(Graphics pg, PageFormat pageFormat,
	 int pageIndex) throws PrinterException {
		if (pageIndex >= m_maxNumPage)
			return NO_SUCH_PAGE;

		pg.translate((int)pageFormat.getImageableX(),
			(int)pageFormat.getImageableY());
		int wPage = 0;
		int hPage = 0;
		if (pageFormat.getOrientation() == PageFormat.PORTRAIT) {
			wPage = (int)pageFormat.getImageableWidth();
			hPage = (int)pageFormat.getImageableHeight();
		}
		else {
			wPage = (int)pageFormat.getImageableWidth();
			wPage += wPage/2;
			hPage = (int)pageFormat.getImageableHeight();
			pg.setClip(0,0,wPage,hPage);
		}

		int y = 0;
		pg.setFont(m_title.getFont());
		pg.setColor(Color.black);
		FontMetrics fm = pg.getFontMetrics();
		y += fm.getAscent();
		pg.drawString(m_title.getText(), 0, y);
		y += 20; // space between title and table headers

		Font headerFont = m_table.getFont().deriveFont(Font.BOLD);
		pg.setFont(headerFont);
		fm = pg.getFontMetrics();

		TableColumnModel colModel = m_table.getColumnModel();
		int nColumns = colModel.getColumnCount();
		int x[] = new int[nColumns];
		x[0] = 0;

		int h = fm.getAscent();
		y += h; // add ascent of header font because of baseline
						// positioning (see figure 2.10)

		int nRow, nCol;
		for (nCol=0; nCol<nColumns; nCol++) {
			TableColumn tk = colModel.getColumn(nCol);
			int width = tk.getWidth();
			if (x[nCol] + width > wPage) {
				nColumns = nCol;
				break;
			}
			if (nCol+1<nColumns)
				x[nCol+1] = x[nCol] + width;
			String title = (String)tk.getIdentifier();
			pg.drawString(title, x[nCol], y);
		}

		pg.setFont(m_table.getFont());
		fm = pg.getFontMetrics();

		int header = y;
		h = fm.getHeight();
		int rowH = Math.max((int)(h*1.5), 10);
		int rowPerPage = (hPage-header)/rowH;
		m_maxNumPage = Math.max((int)Math.ceil(m_table.getRowCount()/
			(double)rowPerPage), 1);

        int iniRow = pageIndex*rowPerPage;
		int endRow = Math.min(m_table.getRowCount(),
			iniRow+rowPerPage);

		for (nRow=iniRow; nRow<endRow; nRow++) {
			y += h;
			for (nCol=0; nCol<nColumns; nCol++) {
				int col = m_table.getColumnModel().getColumn(nCol).getModelIndex();
				Object obj = m_data.getValueAt(nRow, col);
				String str = obj.toString();
					pg.setColor(Color.black);
					pg.drawString(str, x[nCol], y);
			}
		}

		System.gc();
		return PAGE_EXISTS;
	}

	public void retrieveData() {
		Runnable updater = new Runnable() {
			public void run() {
				SimpleDateFormat frm = new SimpleDateFormat("MM/dd/yyyy");
				String currentDate = frm.format(m_data.m_date);
				String result = (String)JOptionPane.showInputDialog(DymoLabelTable.this,
					"Please enter date in form mm/dd/yyyy:", "Input",
					JOptionPane.INFORMATION_MESSAGE, null, null,
					currentDate);
				if (result==null)
					return;

				java.util.Date date = null;
				try {
					date = frm.parse(result);
				}
				catch (java.text.ParseException ex) {
					date = null;
				}

				if (date == null) {
					JOptionPane.showMessageDialog(DymoLabelTable.this,
						result+" is not a valid date",
						"Warning", JOptionPane.WARNING_MESSAGE);
					return;
				}

				setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
				try {
					m_data.retrieveData(date);
				}
				catch (Exception ex) {
					JOptionPane.showMessageDialog(DymoLabelTable.this,
						"Error retrieving data:\n"+ex.getMessage(),
						"Error", JOptionPane.ERROR_MESSAGE);
				}
				setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
				m_title.setText(m_data.getTitle());
				m_table.tableChanged(new TableModelEvent(m_data));
			}
		};
		SwingUtilities.invokeLater(updater);
	}

	public static void main(String argv[]) {
		DymoLabelTable frame = new DymoLabelTable();
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		frame.setVisible(true);
	}

    public 	class ColumnKeeper implements ActionListener {
		protected TableColumn m_column;
		protected ColumnData	m_colData;

		public ColumnKeeper(TableColumn column, ColumnData	colData) {
			m_column = column;
			m_colData = colData;
		}

		public void actionPerformed(ActionEvent e) {
			JLabel renderer = (JLabel)m_column.getHeaderRenderer();
			renderer.setIcon(null);
			JCheckBoxMenuItem item = (JCheckBoxMenuItem)e.getSource();
			TableColumnModel model = m_table.getColumnModel();
			if (item.isSelected()) {
				model.addColumn(m_column);
			}
			else {
				model.removeColumn(m_column);
			}
			m_table.tableChanged(new TableModelEvent(m_data));
			m_table.repaint();
		}
	}
}


