package sound.audioDigitizer;

import gui.In;
import gui.run.RunButton;
import sound.OscopePanel;
import sound.Utils;
import sound.ulaw.UlawCodec;

import javax.sound.sampled.*;
import javax.swing.*;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import java.awt.*;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Vector;

public class CapturePlayBackOscope extends JPanel {
    private final int bufSize = 16384;
    private Capture capture = new Capture();
    private Playback playback = new Playback();
    private double audioData[] = null;
    private OscopePanel dp = new OscopePanel();
    private AudioInputStream audioInputStream;

    public CapturePlayBackOscope() {
        setLayout(new BorderLayout());
        double d[] = {0.0, 0.0};
        dp.setData(d);
        add(dp, BorderLayout.CENTER);
        add(getButtonPanel(), BorderLayout.SOUTH);
    }

    public JPanel getButtonPanel() {
        JPanel bp = new JPanel();
        bp.setLayout(new FlowLayout());

        bp.add(new RunButton("play") {
            public void run() {
                playback.start();
            }
        });

        bp.add(new RunButton("record") {
            public void run() {
                capture.start();
            }
        });
        bp.add(new RunButton("stop") {
            public void run() {
                capture.stop();
            }
        });


        bp.add(new RunButton("echo ulaw code") {
            public void run() {
                UlawCodec ulc = new UlawCodec(audioData);
                ulc.echo();
            }
        });
        bp.add(new RunButton("quit") {
            public void run() {
                System.exit(0);
            }
        });

        return bp;
    }


    /**
     * This is a method that alters its parameters.
     *
     * @param audioBytes
     */
    public byte[] createWaveForm() {
        AudioFormat format = audioInputStream.getFormat();
        byte[] audioBytes = new byte[
                (int) (audioInputStream.getFrameLength()
                * format.getFrameSize())];
        try {
            audioInputStream.read(audioBytes);
        } catch (IOException e) {
            e.printStackTrace();
        }
        int nlengthInSamples = audioBytes.length;
        audioData = new double[nlengthInSamples];
        for (int i = 0; i < audioBytes.length; i++) {
            audioData[i] = ((0xFF & audioBytes[i]) - 128) / 128.0;
        }
        dp.setData(audioData);
        dp.repaint(500);
        return audioBytes;
    }

    /**
     * Write data to the OutputChannel.
     */
    public class Playback implements Runnable {
        SourceDataLine line;
        Thread thread;

        public void start() {
            thread = new Thread(this);
            thread.setName("Playback");
            thread.start();
        }

        public void stop() {
            thread = null;
        }

        private void shutDown(String message) {
            if (thread != null)
                thread = null;
        }

        public void run() {
            // make sure we have something to play
            if (audioInputStream == null) {
                shutDown("No loaded audio to play back");
                return;
            }
            // reset to the beginnning of the stream
            try {
                audioInputStream.reset();
            } catch (Exception e) {
                shutDown("Unable to reset the stream\n" + e);
                return;
            }

            // get an AudioInputStream of the desired format for playback
            AudioFormat format = Utils.get8khzMono8Format();
            AudioInputStream playbackInputStream = AudioSystem.getAudioInputStream(
                    format, audioInputStream);
            DataLine.Info info = new DataLine.Info(SourceDataLine.class,
                    format);
            try {
                line = (SourceDataLine) AudioSystem.getLine(info);
                line.open(format, bufSize);
            } catch (LineUnavailableException ex) {
                shutDown("Unable to open the line: " + ex);
                return;
            }
            // play back the captured audio data

            int frameSizeInBytes = format.getFrameSize();
            int bufferLengthInFrames = line.getBufferSize() / 8;
            int bufferLengthInBytes = bufferLengthInFrames *
                    frameSizeInBytes;
            byte[] data = new byte[bufferLengthInBytes];
            int numBytesRead = 0;

            // start the source data line
            line.start();
            while (thread != null) {
                try {
                    if ((numBytesRead = playbackInputStream.read(data)) ==
                            -1) {
                        break;
                    }
                    int numBytesRemaining = numBytesRead;
                    while (numBytesRemaining > 0) {
                        numBytesRemaining -=
                                line.write(data, 0, numBytesRemaining);
                    }
                } catch (Exception e) {
                    shutDown("Error during playback: " + e);
                    break;
                }
            }
            // we reached the end of the stream.  let the data play out, then
            // stop and close the line.
            if (thread != null) {
                line.drain();
            }
            line.stop();
            line.close();
            line = null;
            shutDown(null);
        }
    } // End class Playback

    /**
     * Reads data from the input channel and writes to the output stream
     */
    class Capture implements Runnable {
        TargetDataLine targetDataLine;
        Thread thread;

        public void start() {

            thread = new Thread(this);
            thread.setName("Capture");
            thread.start();
        }

        public void stop() {
            thread = null;
        }


        public void run() {
            audioInputStream = null;

            // define the required attributes for our line,
            // and make sure a compatible line is supported.

            AudioFormat format = Utils.get8khzMono8Format();
            DataLine.Info info = new DataLine.Info(TargetDataLine.class,
                    format);
            if (!AudioSystem.isLineSupported(info)) {
                In.message("Line matching " + info + " not supported.");
                return;
            }

            // get and open the target data line for capture.

            try {
                targetDataLine =
                        (TargetDataLine) AudioSystem.getLine(info);
                targetDataLine.open(format,
                        targetDataLine.getBufferSize());
            } catch (LineUnavailableException ex) {
                In.message("Unable to open the line: " + ex);
                return;
            } catch (SecurityException ex) {
                In.message(ex.toString());
                return;
            } catch (Exception ex) {
                In.message(ex.toString());
                return;
            }

            // play back the captured audio data
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            int frameSizeInBytes = format.getFrameSize();
            int bufferLengthInFrames = targetDataLine.getBufferSize() / 8;
            int bufferLengthInBytes = bufferLengthInFrames *
                    frameSizeInBytes;
            byte[] data = new byte[bufferLengthInBytes];
            int numBytesRead;
            targetDataLine.start();
            while (thread != null) {
                if ((numBytesRead =
                        targetDataLine.read(data, 0, bufferLengthInBytes)) ==
                        -1) {
                    break;
                }
                out.write(data, 0, numBytesRead);
            }

            // we reached the end of the stream.  stop and close the line.
            targetDataLine.stop();
            targetDataLine.close();
            targetDataLine = null;

            // stop and close the output stream
            try {
                out.flush();
                out.close();
            } catch (IOException ex) {
                ex.printStackTrace();
            }

            // load bytes into the audio input stream for playback

            getAudioInputStream(out, format, frameSizeInBytes);
            try {
                audioInputStream.reset();
            } catch (Exception ex) {
                ex.printStackTrace();
                return;
            }
            createWaveForm();
        }

        private void getAudioInputStream(ByteArrayOutputStream out,
                                         AudioFormat format,
                                         int frameSizeInBytes) {
            byte audioBytes[] = out.toByteArray();
            ByteArrayInputStream bais = new ByteArrayInputStream(
                    audioBytes);
            audioInputStream = new AudioInputStream(bais, format,
                    audioBytes.length / frameSizeInBytes);
        }
    } // End class Capture

    /**
     * Controls for the AudioFormat.
     */
    public static class FormatControls extends JPanel {
        Vector groups = new Vector();
        JToggleButton linrB, ulawB, alawB, rate8B, rate11B, rate16B, rate22B, rate44B;
        JToggleButton size8B, size16B, signB, unsignB, litB, bigB, monoB, sterB;

        public FormatControls() {
            setLayout(new GridLayout(0, 1));
            CompoundBorder cb = new CompoundBorder();
            setBorder(new CompoundBorder(cb, new EmptyBorder(8, 5, 5, 5)));
            JPanel p1 = new JPanel();
            ButtonGroup encodingGroup = new ButtonGroup();
            linrB = addToggleButton(p1, encodingGroup, "linear", true);
            groups.addElement(encodingGroup);
            ButtonGroup sampleRateGroup = new ButtonGroup();
            rate8B = addToggleButton(p1, sampleRateGroup, "8000", false);
            rate11B =
                    addToggleButton(p1, sampleRateGroup, "11025", false);
            rate16B =
                    addToggleButton(p1, sampleRateGroup, "16000", false);
            rate22B =
                    addToggleButton(p1, sampleRateGroup, "22050", false);
            rate44B = addToggleButton(p1, sampleRateGroup, "44100", true);
            groups.addElement(sampleRateGroup);
            JPanel p3 = new JPanel();
            ButtonGroup sampleSizeInBitsGroup = new ButtonGroup();
            size8B =
                    addToggleButton(p3,
                            sampleSizeInBitsGroup,
                            "8",
                            false);
            size16B =
                    addToggleButton(p3,
                            sampleSizeInBitsGroup,
                            "16",
                            true);
            add(p3);
            groups.addElement(sampleSizeInBitsGroup);
            JPanel p4 = new JPanel();
            ButtonGroup signGroup = new ButtonGroup();
            signB = addToggleButton(p4, signGroup, "signed", true);
            unsignB = addToggleButton(p4, signGroup, "unsigned", false);
            add(p4);
            groups.addElement(signGroup);
            JPanel p5 = new JPanel();
            ButtonGroup endianGroup = new ButtonGroup();
            litB =
                    addToggleButton(p5,
                            endianGroup,
                            "little endian",
                            false);
            bigB = addToggleButton(p5, endianGroup, "big endian", true);
            add(p5);
            groups.addElement(endianGroup);
            JPanel p6 = new JPanel();
            ButtonGroup channelsGroup = new ButtonGroup();
            monoB = addToggleButton(p6, channelsGroup, "mono", false);
            sterB = addToggleButton(p6, channelsGroup, "stereo", true);
            add(p6);
            groups.addElement(channelsGroup);
        }

        private JToggleButton addToggleButton(JPanel p, ButtonGroup g,
                                              String name, boolean state) {
            JToggleButton b = new JToggleButton(name, state);
            p.add(b);
            g.add(b);
            return b;
        }

        public AudioFormat getFormat() {
            Vector v = new Vector(groups.size());
            for (int i = 0; i < groups.size(); i++) {
                ButtonGroup g = (ButtonGroup) groups.get(i);
                for (Enumeration e = g.getElements();
                     e.hasMoreElements();) {
                    AbstractButton b = (AbstractButton) e.nextElement();
                    if (b.isSelected()) {
                        v.add(b.getText());
                        break;
                    }
                }
            }
            AudioFormat.Encoding encoding = AudioFormat.Encoding.ULAW;
            String encString = (String) v.get(0);
            float rate = Float.valueOf((String) v.get(1)).floatValue();
            int sampleSize = Integer.valueOf((String) v.get(2)).intValue();
            String signedString = (String) v.get(3);
            boolean bigEndian = ((String) v.get(4)).startsWith("big");
            int channels = (v.get(5)).equals("mono") ? 1 : 2;
            if (encString.equals("linear")) {
                if (signedString.equals("signed")) {
                    encoding = AudioFormat.Encoding.PCM_SIGNED;
                } else {
                    encoding = AudioFormat.Encoding.PCM_UNSIGNED;
                }
            }
            return new AudioFormat(encoding,
                    rate,
                    sampleSize,
                    channels,
                    (sampleSize / 8) * channels,
                    rate,
                    bigEndian);
        }
    } // End class FormatControls

    public static void main(String[] args) {
        CapturePlayBackOscope cpbp = new CapturePlayBackOscope();
        gui.ClosableJFrame cf = new gui.ClosableJFrame();
        Container c = cf.getContentPane();
        c.add(cpbp);
        cf.setSize(400, 400);
        cf.setVisible(true);
    }
}