package sound;

/*
 * Open Source Software by http://www.Docjava.com
 * programmer: D. Lyon
 * e-mail: lyon@docjava.com
 * Date: Apr 29, 2002
 * Time: 12:35:09 PM
 */

import gui.ClosableJFrame;
import sound.soundDemo.CapturePlaybackPanel;
import sound.ulaw.UlawCodec;

import javax.sound.midi.MidiChannel;
import javax.sound.midi.MidiSystem;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.Synthesizer;
import javax.sound.sampled.*;
import java.awt.*;

public class Utils {
    private static MidiChannel channels[] = null;
    private static Synthesizer synth = null;

    public static void play(int ia[], int vel, int dur) {
        for (int i = 0; i < ia.length; i++)
            play(ia[i], vel, dur);
    }

    public static void playThread(final int ia[],
                                  final int vel,
                                  final int dur) {
        Runnable r = new Runnable() {
            public void run() {
                for (int i = 0; i < ia.length; i++)
                    play(ia[i], vel, dur);
            }
        };
        Thread t = new Thread(r);
        t.start();
    }

    public static void play(int ia[], int dur) {
        for (int i = 0; i < ia.length; i++)
            play(ia[i], 127, dur);
    }

    public static int[] transpose(int ia[], int bias) {
        int ans[] = new int[ia.length];
        for (int i = 0; i < ans.length; i++)
            ans[i] = ia[i] + bias;
        return ans;
    }

    public static void print(int ia[]) {
        for (int i = 0; i < ia.length; i++)
            System.out.println(ia[i]);
    }

    public static void play(int nn) {
        play(nn, 127, 20);
    }

    public static void play(int nn, int dur) {
        play(nn, 127, dur);
    }

    public static void play(int nn, int vel, int dur) {
        play(getSynthesizer(), nn, vel, dur);
    }

    public static void play(Synthesizer synth, int nNoteNumber,
                            int nVelocity, int nDuration) {
        noteOn(nNoteNumber, nVelocity);
        if (nDuration < 0) return;
        sleep(nDuration);
        noteOff(nNoteNumber);
        System.out.println("synthesizer=" + synth);
    }

    public static void noteOff(int nNoteNumber) {
        channels[0].noteOff(nNoteNumber);
    }

    public static void noteOn(int nNoteNumber, int nVelocity) {
        channels[0].noteOn(nNoteNumber, nVelocity);
    }

    private static void sleep(int nDuration) {
        try {
            Thread.sleep(nDuration);
        } catch (InterruptedException e) {
        }
    }

    public static Synthesizer getSynthesizer() {
        Synthesizer synth = initSynth();
        openSynth(synth);
        channels = initChannel(synth);
        return synth;
    }

    private static Synthesizer initSynth() {
        if (synth != null) return synth;
        try {
            synth = MidiSystem.getSynthesizer();
        } catch (MidiUnavailableException e) {
        }
        return synth;
    }

    private static void openSynth(Synthesizer synth) {
        try {
            synth.open();
        } catch (MidiUnavailableException e) {
            e.printStackTrace();
            System.exit(1);
        }
    }
    public static void main(String[] args) {
        ClosableJFrame cf = new ClosableJFrame();
        Container c = cf.getContentPane();
        c.setLayout(new BorderLayout());
        CapturePlaybackPanel cpp = new CapturePlaybackPanel();
        c.add(cpp,BorderLayout.CENTER);
        cf.setSize(600,400);
        cf.setVisible(true);
        //Vector graphicLineData = cpp.getGraphicLineData();
    }

    private static MidiChannel[] initChannel(Synthesizer synth) {
        MidiChannel[] channels = synth.getChannels();
        return channels;
    }

    public static double[] getDoubleData(AudioFormat format,
                                         byte[] audioBytes) {
        int data16[] = get16BitAudioData(format, audioBytes);
        double d[] = new double[data16.length];
        for (int i = 0; i < d.length; i++)
            d[i] = data16[i] / 32768.0;
        return d;
    }

    public static int[] get16BitAudioData(AudioFormat format,
                                          byte[] audioBytes) {
        int[] audioData = new int[audioBytes.length];
        if (format.getSampleSizeInBits() == 16) {
            int nlengthInSamples = audioBytes.length / 2;
            audioData = new int[nlengthInSamples];
            if (format.isBigEndian()) {
                for (int i = 0; i < nlengthInSamples; i++) {
                    /* First byte is MSB (high order) */
                    int MSB = (int) audioBytes[2 * i];
                    /* Second byte is LSB (low order) */
                    int LSB = (int) audioBytes[2 * i + 1];
                    audioData[i] = MSB << 8 | (255 & LSB);
                }
            } else {
                for (int i = 0; i < nlengthInSamples; i++) {
                    /* First byte is LSB (low order) */
                    int LSB = (int) audioBytes[2 * i];
                    /* Second byte is MSB (high order) */
                    int MSB = (int) audioBytes[2 * i + 1];
                    audioData[i] = MSB << 8 | (255 & LSB);
                }
            }
        } else if (format.getSampleSizeInBits() == 8) {
            int nlengthInSamples = audioBytes.length;
            audioData = new int[nlengthInSamples];
            if (format.getEncoding().toString().startsWith("PCM_SIGN")) {
                for (int i = 0; i < audioBytes.length; i++) {
                    audioData[i] = audioBytes[i];
                }
            } else {
                for (int i = 0; i < audioBytes.length; i++) {
                    audioData[i] = audioBytes[i] - 128;
                }
            }
        }
        return audioData;
    }

    public static void echo(double[] linearSamples) {
        int ECHO_NUMBER = 4;
        double DECAY = 0.5;
        echo(ECHO_NUMBER, DECAY, linearSamples);
    }

    public static void echo(int numberOfEchos, double decay, double[] linearSamples) {
        int n = linearSamples.length;
        int numTimes = numberOfEchos + 1;
        double currDecay = 1.0;
        short sample, newSample;
        double[] newSamples = new double[n * numTimes];
        for (int j = 0; j < numTimes; j++) {
            for (int i = 0; i < n; i++) // copy the sound's bytes
                newSamples[i + (n * j)] =
                        linearSamples[i] * currDecay;
            currDecay *= decay;
        }
        linearSamples = newSamples;
        UlawCodec ulc = new UlawCodec(linearSamples);
        ulc.play();
    }

    /**
     *
     * @return  little endian 8khz linear 8bit format. Unsigned.
     */
    public static AudioFormat get8khzMono8Format() {
        float sampleRate = 8000.0F;
//8000,11025,16000,22050,44100
        int sampleSizeInBits = 8;
//8,16
        int channels = 1;
//1,2
        // use signed if 16 bit, otherwise use
        // unsigned
        boolean signed = false;
//true,false
        boolean bigEndian = true;
//true,false
        return new AudioFormat(sampleRate,
                sampleSizeInBits,
                channels,
                signed,
                bigEndian);
    }//end get8khzMono8Format

    /**
     * Converts a byte array into an array of samples (a bouble).
     * Each consecutive bytes in the byte array are converted into a double,
     * and becomes the next element in the double array. The size of the
     * returned array is (length/bytesPerDouble).
     * Currently, only 1 byte (8-bit) or 2 bytes (16-bit)
     * samples are supported.
     *
     * @param byteArray      a byte array
     * @param offset         which byte to start from
     * @param length         how many bytes to convert
     * @param bytesPerSample the number of bytes per sample
     * @param signedData     whether the data is signed
     * @return a double array, or <code>null</code> if byteArray is of zero
     *         length
     * @throws ArrayIndexOutOfBoundsException
     *
     */
    public static final double[] bytesToSamples(byte[] byteArray,
                                                int offset,
                                                int length,
                                                int bytesPerSample,
                                                boolean signedData)
            throws ArrayIndexOutOfBoundsException {

        if (0 < length && (offset + length) <= byteArray.length) {
            int doubleLength = length / bytesPerSample;
            double[] doubleArray = new double[doubleLength];

            if (bytesPerSample == 2) {
                if (!signedData) {
                    for (int i = offset, j = 0; j < doubleLength; j++) {
                        int temp = (int) byteArray[i++];
                        temp = (temp << 8);
                        temp |= (int) (0x000000FF & byteArray[i++]);
                        doubleArray[j] = (double) temp;
                    }
                } else {
                    for (int i = offset, j = 0; j < doubleLength; j++) {
                        short temp = (short) (byteArray[i++] << 8);
                        temp |= (short) (0x00FF & byteArray[i++]);
                        doubleArray[j] = (double) temp;
                    }
                }
            } else if (bytesPerSample == 1) {
                for (int i = offset; i < doubleLength; i++) {
                    doubleArray[i] = (double) byteArray[i];
                }
            } else {
                throw new Error
                        ("Unsupported bytes per sample: " + bytesPerSample);
            }

            return doubleArray;

        } else {
            throw new ArrayIndexOutOfBoundsException
                    ("offset: " + offset + ", length: " + length
                    + ", array length: " + byteArray.length);
        }
    }

    /**
     * Converts a little-endian byte array into an array of samples (double).
     * Each consecutive bytes of a float are converted into a double, and
     * becomes the next element in the double array. The number of bytes
     * in the double is specified as an argument. The size of
     * the returned array is (data.length/bytesPerSample).
     *
     * @param data           a byte array
     * @param offset         which byte to start from
     * @param length         how many bytes to convert
     * @param bytesPerSample the number of bytes per sample
     * @param signed         whether the data is signed
     * @return a double array, or <code>null</code> if byteArray is of zero
     *         length
     * @throws ArrayIndexOutOfBoundsException
     *
     */
    public static final double[] littleEndianBytesToSamples(byte[] data,
                                                            int offset,
                                                            int length,
                                                            int bytesPerSample,
                                                            boolean signed)
            throws ArrayIndexOutOfBoundsException {

        if (0 < length && (offset + length) <= data.length) {
            double[] doubleArray = new double[length / bytesPerSample];

            if (bytesPerSample == 2) {
                if (signed) {
                    for (int i = offset, j = 0; i < length; j++) {
                        short temp = (short) ((0x000000FF & data[i++]) |
                                (data[i++] << 8));
                        doubleArray[j] = (double) temp;
                    }
                } else {
                    for (int i = offset, j = 0; i < length; j++) {
                        int temp = (int) ((0x000000FF & data[i++]) |
                                (data[i++] << 8));
                        doubleArray[j] = (double) temp;
                    }
                }
            } else if (bytesPerSample == 1) {
                for (int i = 0; i < doubleArray.length; i++) {
                    doubleArray[i] = data[i];
                }
            } else {
                throw new Error
                        ("Unsupported bytesPerSample: " + bytesPerSample);
            }

            return doubleArray;

        } else {
            throw new ArrayIndexOutOfBoundsException
                    ("offset: " + offset + ", length: " + length
                    + ", array length: " + data.length);
        }
    }

    public static LineListener getLineListener() {
        return new LineListener() {
            public void update(LineEvent e) {
                System.out.println("event:" + e);
            }
        };
    }

    public static Port getMicrophonePort() {
        if (AudioSystem.isLineSupported(Port.Info.MICROPHONE)) {
            try {
                return (Port) AudioSystem.getLine(Port.Info.MICROPHONE);
            } catch (LineUnavailableException e) {
                e.printStackTrace();

            }
        }
        return null;
    }

    public static void listMixersAndExit() {
        System.out.println("Available Mixers:");
        Mixer.Info[] aInfos = AudioSystem.getMixerInfo();
        for (int i = 0; i < aInfos.length; i++) {
            System.out.println(aInfos[i].getName());
        }
        if (aInfos.length == 0) {
            System.out.println("[No mixers available]");
        }
        System.exit(0);
    }

    public static void listSupportedTargetTypes() {
        String strMessage = "Supported target types:";
        AudioFileFormat.Type[] aTypes = AudioSystem.getAudioFileTypes();
        for (int i = 0; i < aTypes.length; i++) {
            strMessage += " " + aTypes[i].getExtension();
        }
        System.out.println(strMessage);
    }

    public static void printMicrophoneInfo() {
        Port p = getMicrophonePort();
        Line.Info inf = p.getLineInfo();
        System.out.println(inf);
        p.addLineListener(getLineListener());
    }
}
