package sound;

import gui.ClosableJFrame;
import gui.In;
import gui.run.RunJob;
import math.fourierTransforms.DFT;
import sound.ulaw.UlawCodec;

import javax.sound.sampled.*;
import java.awt.*;
import java.io.ByteArrayInputStream;
import java.io.IOException;

public class Oscillator {
    private double audioData[];
    private double waveTable[];
    private static int sampleRate = 8000;
    private double frequency;
    private double lambda;
    private double samplesPerCycle;

    // private double delta_freq;

    private final static double twopi = Math.PI * 2;
    private double frequencyOfModulation;
    private double modulationIndex;

    public Oscillator(double frequency_, int length) {
        frequency = frequency_;
        audioData = new double[length];

        //the period of the wave form is
        lambda = 1 / frequency;

        //The number of samples per period is
        samplesPerCycle = sampleRate * lambda;

        //delta_freq = 1 / samplesPerCycle;
        waveTable =
                new double[(int) Math.round(samplesPerCycle)];
    }

    /**
     * Plays the system beep. I only put this here because I did not know
     * where else to put it. After all, it uses the AWT tool kit, and how
     * does that make any sense?
     */
    public static void beep() {
        final Toolkit tk = Toolkit.getDefaultToolkit();
        tk.beep();
    }

    public static void main(String arg[]) {
        beep();
    }

    public static void testPlayTone() {
        while (true)
            playTone(220, In.getInt("enter duration"));
    }

    private static void playTone(int f, int dur) {
        Oscillator osc = new Oscillator(f, dur);
        UlawCodec ulc = new UlawCodec(osc.getSineWave());
        ulc.play();
    }

    public static void asynchronousToneTest() {
        RunJob rj1 = new RunJob(.25, true, 8) {
            public void run() {
                playTone(440);
            }
        };
        RunJob rj2 = new RunJob(2, true, 2) {
            public void run() {
                playTone(230);
            }
        };
        rj1.start();
        rj2.start();
    }

    public static void testPlayAndDisplayTone() {
        playAndDisplayTone(440);
        playAndDisplayTone(220);
    }

    public static void playAndDisplayTone(int frequency) {
        Oscillator osc = new Oscillator(frequency, 8000);
        double d [] = osc.getSineWave();
        UlawCodec ulc = new UlawCodec(d);
        ulc.displayInternalData();
        System.out.println("freq=" + ulc.getFrequency());
    }

    public static void playTone(int frequency) {
        Oscillator osc = new Oscillator(frequency, 8000);
        double d [] = osc.getSineWave();
        UlawCodec ulc = new UlawCodec(d);
        ulc.play();
    }

    public double actualFrequency() {
        return (double) sampleRate / (double) waveTable.length;
    }

    private double[] AudioDataFromTable() {
        int k = 0;
        for (int i = 0; i < audioData.length; i++) {
            audioData[i] = waveTable[k];
            k++;
            if (k >= waveTable.length)
                k = 0;
        }
        //debug();
        return audioData;
    }

    private void debug() {
        System.out.println(
                "\nlambda=" +
                lambda +
                "\nfrequency = " +
                frequency +
                "\nwaveTable.length = " +
                waveTable.length +
                "\nsampleRate = " +
                sampleRate +
                "\naudioData.length = " +
                audioData.length +
                "\nactual frequency = " +
                actualFrequency());
    }

    public double[] getSineWave() {
        for (int i = 0; i < waveTable.length; i++)
            waveTable[i] =
                    0.98 * Math.sin(twopi * i / waveTable.length);
        return AudioDataFromTable();
    }

    /**
     * Get a tone.
     *
     * @param frequency1 the first tone
     * @param frequency2 the second tone
     * @param duration   the duration, in ms
     * @return an array that ranges from -1 to 1
     */
    public static double[] getSineWave(
            int frequency1, int frequency2, long duration) {
        int n = (int) (sampleRate * duration / 1000);
        double d[] = new double[n];
        double eps = twopi/sampleRate;
        int i=0;
        for (double t=0; (t < twopi )&&(i<d.length); t = t + eps) {
            final double f1 = Math.sin(frequency1 * t);
            final double f2 = Math.sin(frequency2 * t);
            d[i] = (f1 + f2)/2.1;
            i++;
        }
        return d;
    }

    public double[] getSquareWave() {
        getSawWave();
        for (int i = 0; i < waveTable.length; i++)
            if (waveTable[i] > 0)
                waveTable[i] = .98;
            else
                waveTable[i] = -.98;
        return AudioDataFromTable();
    }

    public double[] getSawWave() {
        double v = -0.99;
        double dv = 2.0 / (double) waveTable.length;
        for (int i = 0; i < waveTable.length; i++) {
            waveTable[i] = v;
            v += dv;
        }
        System.out.println("Sawwave ends at:" + (v - dv));
        return AudioDataFromTable();
    }

    public double[] getTriangleWave() {
        int sign;
        double v = 0;
        double dv = 3.d / (double) waveTable.length;
        for (int i = 0; i < waveTable.length; i++) {
            waveTable[i] = v;
            sign =
                    sign(Math.cos(twopi * i / waveTable.length));
            v = v + sign * dv;
        }
        return AudioDataFromTable();
    }

    public int sign(double d) {
        if (d < 0) return -1;
        return 1;
    }

    public double getDuration() {
        return
                (
                (double) audioData.length /
                (double) sampleRate);
    }

    public int getSampleRate() {
        return sampleRate;
    }

    public double getFrequency() {
        return frequency;
    }

    public void setModulationIndex(double I) {
        modulationIndex = I;
    }

    public void setModulationFrequency(double fm) {
        frequencyOfModulation = fm;
    }

    public double[] getFM() {
        double r1 = twopi * frequency / sampleRate;
        double r2 = twopi * frequencyOfModulation / sampleRate;
        for (int n = 0; n < audioData.length; n++)
            audioData[n] =
                    Math.sin(
                            r1 * n +
                    +modulationIndex * Math.sin(r2 * n));
        return audioData;
    }

    public double[] getAM() {
        double r1 = twopi * frequency / sampleRate;
        double r2 = twopi * frequencyOfModulation / sampleRate;
        for (int n = 0; n < audioData.length; n++)
            audioData[n] =
                    Math.cos(r1 * n) * Math.cos(r2 * n);
        return audioData;
    }

    public static void testAudioDft() {
        Oscillator o = new Oscillator(440, 64);
        double d[] = o.getSineWave();
        DFT f = new DFT(d.length);
        f.dft(d);
        double psd[] = f.getPowerSpectralDensity();
        OscopePanel op = new OscopePanel(psd);
        ClosableJFrame cjf = new ClosableJFrame("psd");
        Container c = cjf.getContentPane();
        c.setLayout(new BorderLayout());
        c.add(op, BorderLayout.CENTER);
        cjf.setSize(400, 400);
        cjf.setVisible(true);
    }

    public static double[] mult(double d[], double a) {
        double c[] = new double[d.length];
        for (int i = 0; i < d.length; i++)
            c[i] = d[i] * a;
        return c;
    }

    public static double[] add(double a[], double b[]) {
        double c[] = new double[a.length];
        for (int i = 0; i < a.length; i++)
            c[i] = a[i] + b[i];
        return c;
    }

    public static void playLinearData(double[] d)
            throws LineUnavailableException, IOException {
        ByteArrayInputStream bis = new ByteArrayInputStream(
                UlawCodec.getLinearPCMData(d));
        //PCM_SIGNED, 8000.0 Hz, 16 bit, mono, big-endian, audio data
        int frameRate = 8000,
                frameSize = 2,
                sampleRate = 8000,
                sampleSizeInBits = 16,
                channels = 1;
        boolean bigEndian = true;
        /*
        sampleSizeInBits
        */

        AudioFormat af = new AudioFormat(
                AudioFormat.Encoding.PCM_SIGNED,
                sampleRate, sampleSizeInBits,
                channels, frameSize, frameRate, bigEndian);


        AudioInputStream ais = new AudioInputStream(bis,
                af,
                bis.available() / af.getFrameSize());
        DataLine.Info dataLineInfo =
                new DataLine.Info(SourceDataLine.class,  af);
        SourceDataLine sourceDataLine = (SourceDataLine)
                AudioSystem.getLine(dataLineInfo);
        sourceDataLine.open(af);
        sourceDataLine.start();
        byte playBuffer[] = new byte[16384];
        int cnt;
        while ((
                cnt = ais.read(playBuffer, 0,
                        playBuffer.length))
                != -1) {
            //Keep looping until the input read
            // method returns -1 for empty stream.
            if (cnt > 0) {
                //Write data to the internal buffer of
                // the data line where it will be
                // delivered to the speakers in real
                // time
                sourceDataLine.write(playBuffer, 0, cnt);
            }//end if
        }//end while

        //Block and wait for internal buffer of the
        // SourceDataLine to become empty.
        sourceDataLine.drain();
        sourceDataLine.stop();
        sourceDataLine.close();
    }
}