package net.server.servlets;

/**
 * The HtmlPage class encapsulates the functionality
 * required to prepare an HTML script to display in
 * a browser. This class provides methods to construct
 * various HTML elements such as forms, list boxes,
 * anchor tags and headlines.
 *
 * @author Robert Lysik
 * @version 1.00
 */
public class HtmlPage {
    private String strTitle;
    private String strBody;

    /**
     * This is the default constructor for the HtmlPage class.
     * An empty title tag is provided for the title string, as
     * well as the opening tag for the body section of the
     * HTML page.
     */
    public HtmlPage() {
        strTitle = new String("<head><title></title></head>");
        strBody = new String("<body>");
    }

    /**
     * This constructor takes a String as an argument which
     * is used to initialize the title of the page.
     *
     * @param String the title of the page
     */
    public HtmlPage(String title) {
        strTitle = new String("<head><title>"
                + title
                + "</title></head>");

        strBody = new String("<body>");
    }

    /**
     * This function inserts a line break into the HTML
     * script.
     */
    public void addBreak() {
        strBody += "<br>\r\n";
    }

    /**
     * This function inserts a headline tag with the
     * value provided as a parameter to the function,
     * as well as the text to be displayed.
     *
     * @param int the headline type
     * @param String the headline text
     */
    public void addHeadline(int headlineType,
                            String headlineText) {
        strBody += "<h" + headlineType + ">";
        strBody += headlineText;
        strBody += "</h" + headlineType + ">\r\n";
    }

    /**
     * This function inserts a hidden input control
     * tag into the HTML script.
     *
     * @param String name of hidden tag
     * @param String value of hidden tag
     */
    public void addHidden(String name,
                          String value) {
        strBody += "<input type = " +
                quote("hidden") +
                " name = " +
                quote(name) +
                " value = " +
                quote(value) +
                ">\r\n";
    }

    /**
     * This function creates an input element for a form. Input
     * elements, such as text boxes, radio buttons, and checkboxes
     * allow the user of an HTML page to enter information which
     * may then be processed by another page, or by script within
     * the existing page. These elements must be enclosed within
     * form tags in order for correct processing.
     *
     * @param String the type of form input, i.e. text, radio, checkbox
     * @param String the name of the input element
     * @param String the value associated with this element
     * @param String the size of the element
     */
    public void addInput(String type,
                         String name,
                         String value,
                         String size) {
        strBody += "<input type = " +
                quote(type) +
                " name = " +
                quote(name) +
                " value = " +
                quote(value) +
                " size = " +
                quote(size) +
                ">\r\n";
    }

    /**
     * This function adds submit button to a form in an HTML page.
     *
     * @param String the text to be displayed on the submit button
     */
    public void addSubmit(String value) {
        strBody += "<input type = " +
                quote("submit") +
                " value = " +
                quote(value) +
                ">\r\n";
    }

    /**
     * This function adds a normal text string to the
     * HTML script.
     *
     * @param String the text to be displayed
     */
    public void addText(String text) {
        strBody += text + "\r\n";
    }

    /**
     * This function adds a closing form HTML tag.
     */
    public void endForm() {
        strBody += "</form>\r\n";
    }

    /**
     * This function returns the HTML script to generate the
     * page as it is currently defined. The title and body
     * elements of the HTML page are enclosed in opening and closing
     * HTML tags.
     */
    public String getHtml() {
        String html = new String();

        html = "<html>"
                + strTitle
                + strBody
                + "</body>"
                + "</html>";

        return html;
    }

    /**
     * This function creates an input element which is called a
     * select list. This is a drop down list box, if the size
     * of the list box is set to one, or a normal list box if
     * the size is greater than one. This element must be
     * encapsulated within the FORM section of an HTML page in
     * order for correct processing to occur.
     *
     * @param String the name of the select list
     * @param String the size of the select list. If set to
     *        one, this becomes a drop down list box.
     * @param String the values associated with the list box
     * @param String the text displayed in the list box.
     */
    public String getSelect(String name,
                            String size,
                            String values[],
                            String text[]) {
        String select = new String();

        // The opening tag for the select list includes
        // the name of the list and its size.
        select += "<select name = " +
                quote(name) +
                " size = " +
                quote(size) +
                ">\r\n";

        // Each element of the list box is then added, prefixed
        // by the OPTION tag.
        for (int index = 0; index < values.length; index++) {
            select += "<option value = " +
                    quote(values[index]) +
                    ">" +
                    text[index] +
                    "\r\n";
        }

        // The closing tag for the select list is then added
        select += "</select>\r\n";

        return select;
    }

    /**
     * This function returns a string which is surrounded
     * by quotes.
     *
     * @param String the string to be encapsulated in quotes
     */
    public String quote(String s) {
        return '\"' + s + '\"';
    }

    /**
     * This function inserts the opening tag for a form into
     * the HTML script.
     *
     * @param String the method of form submission, either
     *        post or get.
     * @param String the action attribute of the form.
     */
    public void startForm(String method, String action) {
        strBody += "<form method = " +
                quote(method) +
                " action = " +
                quote(action) +
                ">\r\n";
    }
}
