/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */
package net.rmi.rjs.jobs;

import gui.ClosableJFrame;
import j2d.ShortImageBean;

import java.awt.*;
import java.io.Serializable;

public class FractalLogic {
    private final MandleBrotDimensions mandleBrotDimensions =
            new MandleBrotDimensions();
    private static final MandelTables mandelTables = new MandelTables();
    private Answers ans = new Answers();
    private int fromPoint;
    private int toPoint;

    public FractalLogic() {

    }

    public static void main(String[] args) {
        FractalLogic fl = new FractalLogic();
        fl.testMandelbrot();
    }

    public int getColor(float pixelr, float pixeli) {

        float zr = 0;
        float zi = 0;
        float tmp = 0;
        int iter = 0;
        //z*z+c, in complex basically;
        do {
            iter++;
            tmp = zr;
            zr = zr * zr - zi * zi;
            zi = 2 * tmp * zi;
            zr = zr + pixelr;
            zi = zi + pixeli;
        } while (((zr * zr + zi * zi) <= 300) && (iter <= mandelTables.maxIter));
        if (iter > mandelTables.maxIter)
            return -1;
        else
            return iter;
    }

    public void testMandelbrot() {
        final Dimension dim = new Dimension(400, 400);
        ClosableJFrame cjf = new ClosableJFrame("mandlebrot") {
            public void paint(Graphics g) {
                Dimension d = getSize();
                Image i = getMandelbrot2(d.width, d.height);
//Image i = getMandelbrot(dim, 3);
                g.drawImage(i, 0, 0, null);
            }

        };
        cjf.setSize(dim.width, dim.height);
        cjf.setVisible(true);
    }
     
    /* Author= Francisco Castellanos
     * Date = 06/09/05
     * 1. This method takes a ShortImageBean object as paramter and an integer.
     * 2. It generates of computable jobs which will process Image.
     * 3. Each computable job will process a section of the image. The number of jobs
     * is equal to the integer passed as parameter.
     * 4. It returns an array of computable jobs.
     */
    
    public ComputeJob[] TopDownPartition(final ShortImageBean sib, int noParts) {

        ComputeJob[] cjs = new ComputeJob[noParts];
        final short[][] r = sib.getR();

        int height = r[0].length;
        final int width = r.length - 1;

        if (noParts <= 0) {
            return null;
        }

        int newHeight = height / noParts;

        fromPoint = 0;
        toPoint = 0;
        //break images (array) = All same width as original image
        //This will process the image top-down.
        for (int i = 1; i <= noParts; i++) {
            fromPoint = toPoint;
            //next portion of image goes from previous height
            //to next equal height image or to end of image (max height)
            toPoint = (i * newHeight < height) ? (i * newHeight) : (height - 1);
            //System.out.println("Array row "+ (i - 1) + " from= " + fromPoint + " to= " + toPoint);
            final int fromPy = fromPoint;
            final int toPy = toPoint;
            final int fromPx = 0;
            final int toPx = width;


            final short[][] rr = new short[r.length][];
            final short[][] gg = new short[r.length][];
            final short[][] bb = new short[r.length][];

            for (int j = 0; j < r.length; j++) {
                rr[j] = (short[]) r[j].clone();
                gg[j] = (short[]) r[j].clone();
                bb[j] = (short[]) r[j].clone();
            }

            final int px = 0;
            final int py = i - 1;

            cjs[i - 1] = new ComputeJob() {
                public Serializable getAnswer() {
                    final ShortImageBean s = new FractalShortImageBean(px, py, fromPx, toPx, fromPy, toPy);
                    s.setR(rr);
                    s.setG(gg);
                    s.setB(bb);
                    //System.out.println("Processing piece Image fp=" + fromPy + "tp=" + toPy);
                    for (int y = fromPy; y < toPy; y++) {
                        for (int x = fromPx; x < toPx; x++) {
                            mandelbrot(x, y, s.getR(), s.getG(), s.getB());
                        }

                    }
                    return s;
                }
            };
        }
        return cjs;
    }

    public ShortImageBean[] TopDownPartition2(ShortImageBean sib, int noParts) {

        short[][] r = sib.getR();
        short[][] g = sib.getG();
        short[][] b = sib.getB();
        int height = r[0].length;

        ShortImageBean[] sibs = new ShortImageBean[noParts];

        if (noParts <= 0) {
            return null;
        }

        int newHeight = height / noParts;

        fromPoint = 0;
        toPoint = 0;
        //break images (array) = All same width as original image
        //This will process the image top-down.
        for (int i = 1; i <= noParts; i++) {
            fromPoint = toPoint;
            //next portion of image goes from previous height
            //to next equal height image or to end of image (max height)
            toPoint = (i * newHeight < height) ? (i * newHeight) : (height - 1);
            //System.out.println("Array row "+ (i - 1) + " from= " + fromPoint + " to= " + toPoint);
            ShortImageBean s = new ShortImageBean();
            short[][] rr = new short[toPoint - fromPoint][];
            short[][] gg = new short[toPoint - fromPoint][];
            short[][] bb = new short[toPoint - fromPoint][];
            int indx = 0;
            for (int y = fromPoint; y < toPoint; y++) {
                rr[indx] = r[y];
                gg[indx] = g[y];
                bb[indx] = b[y];
                indx++;
            }
            s.setR(rr);
            s.setG(gg);
            s.setB(bb);
            sibs[i - 1] = s;
        }
        return sibs;
    }

    public Image getMandelbrot(int w, int h) {
        ShortImageBean sib = new ShortImageBean(w, h);
        mandelbrot(sib.getR(), sib.getG(), sib.getB());
        return sib.getImage();
    }

    //Francisco Castellanos = Testing purpose: Process image passing points.
    public Image getMandelbrot2(int w, int h) {
        ShortImageBean sib = new ShortImageBean(w, h);
        //mandelbrot(sib.getR(), sib.getG(), sib.getB());
        short[][] r = sib.getR();
        short[][] g = sib.getG();
        short[][] b = sib.getB();

        int height = r[0].length;
        int width = r.length;

        for (int y = 0; y < height; y++)
            for (int x = 0; x < width; x++) {

                mandelbrot(x, y, r, g, b);
            }

        return sib.getImage();
    }

    /* Author =Francisco Castellanos
     * Date = 06/09/05 
     * Overloaded the mandelbrot method to process a single point.
     */
    public void mandelbrot(int x, int y, short[][] r, short[][] g, short[][] b) {
        int height = r[0].length;
        int width = r.length;
        int Clr;
        float pixelr, pixeli;

        pixelr
                = mandleBrotDimensions.getxMin() +
                (float) x / width *
                (
                mandleBrotDimensions.getxMax() -
                mandleBrotDimensions.getxMin());
        pixeli
                = mandleBrotDimensions.getyMin() +
                (float) y / height *
                (
                mandleBrotDimensions.getyMax() -
                mandleBrotDimensions.getyMin());
        Clr = getColor(pixelr, pixeli);
        if (Clr == -1) {
            r[x][y] = 255;
            g[x][y] = 128;
            b[x][y] = 0;
        } else {
            r[x][y] = mandelTables.colorR[Clr % mandelTables.maxIter];
            g[x][y] = mandelTables.colorG[Clr % mandelTables.maxIter];
            b[x][y] = mandelTables.colorB[Clr % mandelTables.maxIter];
        }
    }

    public void mandelbrot(int x, int y, int h, int w, short[][] r, short[][] g, short[][] b) {
        int height = h;
        int width = w;
        int Clr;
        float pixelr, pixeli;

        pixelr
                = mandleBrotDimensions.getxMin() +
                (float) x / width *
                (
                mandleBrotDimensions.getxMax() -
                mandleBrotDimensions.getxMin());
        pixeli
                = mandleBrotDimensions.getyMin() +
                (float) y / height *
                (
                mandleBrotDimensions.getyMax() -
                mandleBrotDimensions.getyMin());
        Clr = getColor(pixelr, pixeli);
        if (Clr == -1) {
            r[x][y] = 255;
            g[x][y] = 128;
            b[x][y] = 0;
        } else {
            r[x][y] = mandelTables.colorR[Clr % mandelTables.maxIter];
            g[x][y] = mandelTables.colorG[Clr % mandelTables.maxIter];
            b[x][y] = mandelTables.colorB[Clr % mandelTables.maxIter];
        }
    }

    public void mandelbrot(short[][] r, short[][] g, short[][] b) {
        int height = r[0].length;
        int width = r.length;
        int Clr;
        float pixelr, pixeli;
        for (int y = 0; y < height; y++)
            for (int x = 0; x < width; x++) {
                pixelr
                        = mandleBrotDimensions.getxMin() +
                        (float) x / width *
                        (
                        mandleBrotDimensions.getxMax() -
                        mandleBrotDimensions.getxMin());
                pixeli
                        = mandleBrotDimensions.getyMin() +
                        (float) y / height *
                        (
                        mandleBrotDimensions.getyMax() -
                        mandleBrotDimensions.getyMin());
                Clr = getColor(pixelr, pixeli);
                if (Clr == -1) {
                    r[x][y] = 255;
                    g[x][y] = 128;
                    b[x][y] = 0;
                } else {
                    r[x][y] = mandelTables.colorR[Clr % mandelTables.maxIter];
                    g[x][y] = mandelTables.colorG[Clr % mandelTables.maxIter];
                    b[x][y] = mandelTables.colorB[Clr % mandelTables.maxIter];
                }
            }
    }

    public int getMaxIter() {
        return mandelTables.maxIter;
    }

    public void setMaxIter(int maxIter) {
        mandelTables.maxIter = maxIter;
    }

    public Answers getAns() {
        return ans;
    }
}

/*
 * 
 * @author Francisco Castellanos
 *
 * The classes below where created with the purpose of testing and to 
 * create a frame to display the image processed.
 */

class FractalLogicTest {

    public static void main(String[] args) {

        FlImageFrame frame = new FlImageFrame();
        frame.setSize(400, 400);
        frame.setVisible(true);

    }
}

class FlImageFrame extends ClosableJFrame {

    public FlImageFrame() {

        FlImagePanel imp = new FlImagePanel();

        addComponent(imp);
    }
}

class FlImagePanel extends javax.swing.JPanel {

    private Image im;
    private FractalLogic fl;

    public FlImagePanel() {
        fl = new FractalLogic();
        im = fl.getMandelbrot(400, 400);
        ;
    }

    public void paintComponent(Graphics g) {
        System.out.println(im.getHeight(null));
        System.out.println(im.getWidth(null));
        g.drawImage(im, 0, 0, null);
    }
}