/*
 * The org.opensourcephysics.numerics package contains numerical methods
 * for the book Simulations in Physics.
 * Copyright (c) 2002  H. Gould, J. Tobochnik, and W. Christian.
 */
package math.numerics;

/**
 * ParsedFunction defines a function of a single varianble using a String.
 *
 * This function is immutable.  That is, once an instance is created with a particular
 * function string, the function cannot be changed.  Because immutable
 * objects cannot change, they are thread safe and can be freely shared in a Java
 * program.
 *
 * @author Wolfgang Christian
 */
public final class ParsedFunction implements Function{

  private final String fStr;
  private final Function function;

  /**
   * Constructs a function x with from the given string.
   *
   * @param fStr the function
   */
  public ParsedFunction(String fStr) throws ParserException {
    this(fStr, "x");
  }

  /**
   * Constructs a ParsedFunction from the given string and independent variable.
   *
   * @param _fStr the function
   * @param var the independent variable
   */
  public ParsedFunction(String _fStr, String var) throws ParserException {
    fStr=_fStr;
    SuryonoParser parser=null;
    parser = new SuryonoParser (fStr, var);
    function=parser;
  }

  /**
   * Evaluates the function, f.
   *
   * @param x the value of the independent variable
   *
   * @return the value of the function
   */
  public double evaluate(double x) {
    return function.evaluate(x);
  }


  /**
   * Represents the function as a string.
   *
   * @return the string
   */
  public String toString() {
      return "f(x) = "+fStr;
  }
}