/***************************************************************************
 // @(#)vsFFT1D.java
 //
 // Perform a non recursive in place FFT using doubles.
 //
 // @version 1.0, 4 JULY 1997
 // @author  D. Lyon, V. Silva
 ****************************************************************************/
package math.fourierTransforms;

import math.MathUtils;
import utils.Timer;

/**
 * A 1D FFT for integral power of two FFTs
 */
public class FFTSimple1d {
    private final int FORWARD_FFT = -1;
    private final int REVERSE_FFT = 1;
    private int direction = FORWARD_FFT;
    private static final float twoPI = (float) (2 * Math.PI);
    // These arrays hold the complex 1D FFT data.
    private float realData[] = null;
    private float imaginaryData[] = null;

    public static float[] getPsd(float audioWaveForm[]) {
        float i[] = new float[audioWaveForm.length];
        FFTSimple1d f = new FFTSimple1d();
        f.computeForwardFFT(audioWaveForm, i);
        float[] magnitudeSpectrum = f.getMagnitudeSpectrum();
        return magnitudeSpectrum;
    }

    public void visuallyTest() {
        System.out.println("Testing FFT engine in 1D class...");
        int N = 8;
        float[] in_r = new float[N];
        float[] in_i = new float[N];
        float[] holder_r = new float[N];
        float[] holder_i = new float[N];

        // Create a test ramp signal.
        for (int i = 0; i < N; i++) {
            in_r[i] = (float) i / N;
            in_i[i] = (float) 0;
        }
        computeForwardFFT(in_r, in_i);

        // Copy to new array because IFFT will destroy the FFT results.
        for (int i = 0; i < N; i++) {
            holder_r[i] = in_r[i];
            holder_i[i] = in_i[i];
        }
        for (int i = 0; i < N; i++) {
            in_r[i] = in_r[i];
            in_i[i] = in_i[i];
        }
        computeBackwardFFT(in_r, in_i);
        float[] mag = getMagnitudeSpectrum(in_r, in_i);
        System.out.println("i    x1[i]    re[i]             im[i]           tv[i]");
        for (int i = 0; i < N; i++) {
            System.out.println(i +
                    "\t" + i +
                    "\t" + holder_r[i] +
                    "\t\t" +
                    holder_i[i] +
                    "\t\t" + mag[i]);
        }
    }

    public void timeFFT(int n) {
        // Time a 256K FFT
        float realPart[] = new float[n];
        float imaginaryPart[] = new float[n];

        // Create a test ramp signal.

        synthesizeRamp(realPart, imaginaryPart);
        //MathUtils.print(realPart);

        utils.Timer t1 = new utils.Timer();
        runExperiment(t1, realPart, imaginaryPart);
    }

    public static void print(Object o[]) {
        for (int i = 0; i < o.length; i++)
            System.out.println(o[i]);
    }

    /**
     * Destroy the input data with a linear ramp.
     *
     * @param realPart      input datas real component
     * @param imaginaryPart input datas' imaginary component.
     */
    public static final void synthesizeRamp(float[] realPart, float[] imaginaryPart) {
        int n = realPart.length;
        for (int i = 0; i < n; i++) {
            realPart[i] = (float) i / n;
            imaginaryPart[i] = (float) 0;
        }
    }

    private void runExperiment(Timer t1,
                               float[] realPart,
                               float[] imaginaryPart) {
        t1.start();
        computeForwardFFT(realPart, imaginaryPart);
        t1.stop();
        t1.print("did an fft-radix 2 of length" + realPart.length +
                "using float in:");
    }

    private static final void window(float r[], float w[]) {
        for (int i = 0; i < r.length; i++)
            r[i] = r[i] * w[i];
    }

    /**
     * 1D FFT utility functions.
     */
    public void swap(int i, int numBits) {
        float tempr;
        int j = bitr(i, numBits);
        tempr = realData[j];
        realData[j] = realData[i];
        realData[i] = tempr;
        tempr = imaginaryData[j];
        imaginaryData[j] = imaginaryData[i];
        imaginaryData[i] = tempr;
    }

    // swap Zi with Zj
    private void swapInt(int i, int j) {
        float tempr;
        int ti;
        int tj;
        ti = i - 1;
        tj = j - 1;
        // System.out.print("swapInt " + ti+","+tj+"\t");
        // System.out.println(Integer.toBinaryString(ti)+","+Integer.toBinaryString(tj));
        // System.out.println(Integer.toBinaryString(ti)+"bitr="+Integer.toBinaryString(bitr(ti)));
        tempr = realData[tj];
        realData[tj] = realData[ti];
        realData[ti] = tempr;
        tempr = imaginaryData[tj];
        imaginaryData[tj] = imaginaryData[ti];
        imaginaryData[ti] = tempr;
    }

    /**
     * get the location of the maximum partial
     *
     * @param realPart
     * @param imaginaryPart
     * @return location of max(realPart[i],imaginaryPart[i])
     */
    public static final int getMaxPSDLocation(float realPart[],
                                              float imaginaryPart[]) {
        float max;
        max = -0.99e30f;
        int location = 0;
        for (int i = 0; i < realPart.length; i++)
            if (getMagnitude(realPart[i], imaginaryPart[i]) > max) {
                max = realPart[i];
                location = i;
            }
        return location;
    }

    /**
     * Compute the sum of the squares of a complex number
     *
     * @param r    real part
     * @param imag imaginary part
     * @return r*r + imag * imag.
     */
    public static final float getMagnitude(float r, float imag) {
        return r * r + imag * imag;
    }

    float getMaxValue(float in[]) {
        float max;
        max = -0.99e30f;
        for (int i = 0; i < in.length; i++)
            if (in[i] > max)
                max = in[i];
        return max;
    }

    private void bitReverse2() {
        /* bit reversal */
        int n = realData.length;
        int j = 1;
        int k;
        for (int i = 1; i < n; i++) {
            if (i < j) {
                swapInt(i, j);
            } //if
            k = n / 2;
            while (k >= 1 && k < j) {
                j = j - k;
                k = k / 2;
            }
            j = j + k;
        } // for
    }

    private int bitr(int j, int numBits) {
        int ans = 0;
        for (int i = 0; i < numBits; i++) {
            ans = (ans << 1) + (j & 1);
            j = j >> 1;
        }
        return ans;
    }

    public void computeForwardFFT(float[] in_r, float[] in_i) {
        direction = FORWARD_FFT;
        fft(in_r, in_i);
    }

    public void computeBackwardFFT(float[] in_r, float[] in_i) {
        direction = REVERSE_FFT;
        fft(in_r, in_i);
    }

    /**
     * FFT engine.
     */
    public void fft(float _realPart[], float _imaginaryPart[]) {
        int id;
        // radix 2 number if sample, 1D of course.
        int localN;
        float wtemp, Wjk_r, Wjk_i, Wj_r, Wj_i;
        float theta, tempr, tempi;
        int numBits = MathUtils.getLogBase2(_realPart.length);
        // Truncate input data to a power of two
        int length = 1 << numBits; // length = 2**nu
        int n = length;

        // Copy passed references to variables to be used within
        // transforms.fft routines & utilities
        realData = _realPart;
        imaginaryData = _imaginaryPart;
        bitReverse2();
        for (int m = 1; m <= numBits; m++) {
            // localN = 2^m;
            localN = 1 << m;
            Wjk_r = 1;
            Wjk_i = 0;
            theta = twoPI / localN;
            Wj_r = (float) Math.cos(theta);
            Wj_i = (float) (direction * Math.sin(theta));
            int nby2 = localN / 2;
            for (int j = 0; j < nby2; j++) {
                // This is the FFT innermost loop
                // Any optimizations that can be made here will yield
                // great rewards.
                for (int k = j; k < n; k += localN) {
                    id = k + nby2;
                    tempr = Wjk_r * realData[id] - Wjk_i * imaginaryData[id];
                    tempi = Wjk_r * imaginaryData[id] + Wjk_i * realData[id];

                    // Zid = Zi -C
                    realData[id] = realData[k] - tempr;
                    imaginaryData[id] = imaginaryData[k] - tempi;
                    realData[k] += tempr;
                    imaginaryData[k] += tempi;
                }

                // (eq 6.23) and (eq 6.24)

                wtemp = Wjk_r;
                Wjk_r = Wj_r * Wjk_r - Wj_i * Wjk_i;
                Wjk_i = Wj_r * Wjk_i + Wj_i * wtemp;
            }
        }
    }

    /**
     * @return gets the real component from FFT.
     */
    public float[] getRealData() {
        return realData;
    }

    /**
     * @return gets the imaginary component from FFT.
     */
    public float[] getImaginaryData() {
        return imaginaryData;
    }

    public float[] getMagnitudeSpectrum() {
        float magnitudeSpectrum[] =
                getMagnitudeSpectrum(realData, imaginaryData);
        double s = 0.0;
        for (int i = 0; i < magnitudeSpectrum.length; i++)
            s = s + magnitudeSpectrum[i];
        double v = s;
        scale(magnitudeSpectrum, (float) (1.0 / v));
        return magnitudeSpectrum;
    }
      public static void scale(final float[] a,
                             final float k) {
        if (a == null) return;
        for (int i = 0; i < a.length; i++) {
            a[i] = a[i] * k;
        }
    }
    /**
     * Compute the power spectral density of the input
     * arrays
     *
     * @param in_r real part of an fft
     * @param in_i imaginary part of an fft
     * @return the psd.
     */
    public float[] getMagnitudeSpectrum(float in_r[], float in_i[]) {
        int N = in_r.length;
        float[] mag = new float[N];
        for (int i = 0; i < in_r.length; i++)
            mag[i] = (float) Math.sqrt(in_r[i] * in_r[i] + in_i[i] * in_i[i]);
        return (mag);
    }

    public static void testFFT() {
        FFTSimple1d f = new FFTSimple1d();
        for (int i = 0; i < 10; i++)
            f.timeFFT(1024 * 1024);
    }

    public static void main(String args[]) {
        testFFT();
    }
}
