package j3d.utils.sceneGraphBuilderUtils;

import j3d.utils.propertyEditors.PropertyEditorFactory;

import javax.media.j3d.*;
import javax.swing.*;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;

public class ETransformGroup extends TransformGroup {

    public static final int X = 0, Y = 1, Z = 2;

    public ETransformGroup(String name) {
        setCapability(ALLOW_CHILDREN_READ);
        setUserData(name);
    }

    public ETransformGroup(String name, Node node) {
        this(name);
        addChild(node);
    }

    public ETransformGroup(String name, Transform3D t) {
        this(name);
        setTransform(t);
    }

    private static ETransformGroup addRotationControl(String name, Node node,
                                                      int axis, JComponent pane) {
        ETransformGroup tg = new ETransformGroup(name);
        tg.addChild(node);
        TransformGroupWrapper tgw = new TransformGroupWrapper(tg);
        pane.add(rotationControlPanel(axis, tgw), 0);
        return tg;
    }

    public static ETransformGroup addCombinedControls(Node node, JComponent pane) {
        ETransformGroup tg = new ETransformGroup("CombinedControls", node);
        TransformGroupWrapper tgw = new TransformGroupWrapper(tg);
        includeRotationControls(pane, tgw);
        includeTranslationControls(pane, tgw);
        includeResetButton(pane, tgw);
        return tg;
    }

    public static
    ETransformGroup addHierarchicalControls(Node node, Box box) {
        ETransformGroup tg = new ETransformGroup("TranslationControls", node);
        TransformGroupWrapper tgw = new TransformGroupWrapper(tg);
        includeTranslationControls(box, tgw);
        tg = addRotationControl("rotate-Z", tg, Z, box);
        tg = addRotationControl("rotate-Y", tg, Y, box);
        tg = addRotationControl("rotate-X", tg, X, box);
        return tg;
    }

    private static JButton alphaPauseButton(final Alpha alpha) {
        final JButton b = new JButton("Pause");
        b.setBackground(Color.red);
        b.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent ae) {
                if (alpha.isPaused()) {
                    alpha.resume();
                    b.setText("Pause");
                    b.setBackground(Color.red);
                } else {
                    alpha.pause();
                    b.setText("Resume");
                    b.setBackground(Color.green);
                }
            }
        });
        return b;
    }

    public static ETransformGroup controlledSubgraph(Box pane,
                                                     String title,
                                                     Node[] children,
                                                     boolean hierarchical) {
        BranchGroup scene = SGBuilder.newBranchGroup(title);
        for (int i = 0; i < children.length; i++) scene.addChild(children[i]);
        return hierarchical ? addHierarchicalControls(scene, pane) :
                addCombinedControls(scene, pane);
    }

    private static char getAxis(int axis) {
        return axis == X ? 'X' :
                axis == Y ? 'Y' :
                'Z';
    }

    private static ArrayList getSliders(Container container) {
        ArrayList sliders = new ArrayList();
        int n = container.getComponentCount();
        for (int i = 0; i < n; i++) {
            Component c = container.getComponent(i);
            if (c instanceof JSlider)
                sliders.add(c);
            else if (c instanceof Container) sliders.addAll(getSliders((Container) c));
        }
        return sliders;
    }

    private static void includeResetButton(JComponent pane,
                                           final TransformGroupWrapper tgw) {
        JButton b = new JButton("Reset");
        final ArrayList sliders = getSliders(pane);
        b.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                tgw.reset();
                for (int i = 0; i < sliders.size(); i++)
                    ((JSlider) sliders.get(i)).setValue(0);
            }
        });
        Box box = Box.createHorizontalBox();
        pane.add(box);
        box.add(b);
        box.add(tgw.getTextArea());
    }

    private static void includeRotationControls(JComponent pane,
                                                TransformGroupWrapper tgw) {
        pane.add(rotationControlPanel(X, tgw, false));
        pane.add(rotationControlPanel(Y, tgw, false));
        pane.add(rotationControlPanel(Z, tgw, false));
    }

    private static void includeTranslationControls(JComponent pane,
                                                   TransformGroupWrapper tgw) {
        Box hb = Box.createHorizontalBox();
        pane.add(hb);
        Box vbWest = Box.createVerticalBox();
        hb.add(vbWest);
        vbWest.add(PropertyEditorFactory.sliderPE("X", -20, 20, tgw, false));
        vbWest.add(PropertyEditorFactory.sliderPE("Y", -20, 20, tgw, false));
        vbWest.add(PropertyEditorFactory.sliderPE("Z", -20, 20, tgw, false));
        JPanel vbEast = new JPanel(new GridLayout(0, 1));
        hb.add(vbEast);
        vbEast.add(new JPanel());
        vbEast.add(tgw.getTextArea());
        vbEast.add(new JPanel());
    }

    public static TransformGroup rotate(String name, Node node,
                                        int axis, double radians) {
        if (name == null) name = "Rotation Group";
        Transform3D t = new Transform3D();
        rotate(t, axis, radians);
        TransformGroup tg = new ETransformGroup(name, t);
        tg.addChild(node);
        // How to get the angle of rotation?
        return tg;
    }

    public static void rotate(Transform3D t, int axis, double radians) {
        if (axis == X)
            t.rotX(radians);
        else if (axis == Y)
            t.rotY(radians);
        else if (axis == Z) t.rotZ(radians);
    }

    private static JPanel rotationControlPanel(int axis,
                                               TransformGroupWrapper tgw) {
        return rotationControlPanel(axis, tgw, true);
    }

    private static JPanel rotationControlPanel(int axis,
                                               TransformGroupWrapper tgw,
                                               boolean includeTransformMatrix) {
        JPanel jp =
                PropertyEditorFactory.sliderPE("Rot" + getAxis(axis), 0, 360, tgw, false);
        if (includeTransformMatrix) jp.add(BorderLayout.EAST, tgw.getTextArea());
        return jp;
    }

    public static ETransformGroup spinTransform(Node node, int times, long period,
                                                Vector3d axis, JComponent controls) {
        ETransformGroup spinTransformGroup =
                new ETransformGroup("SpinTransformGroup", node);
        spinTransformGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        Alpha alpha = new Alpha(times, period);
        if (controls != null) controls.add(alphaPauseButton(alpha));
        RotationInterpolator rotator =
                new RotationInterpolator(alpha, spinTransformGroup);
        if (axis != null) {
            Transform3D t3d = new Transform3D();
            t3d.setTranslation(axis);
            rotator.setTransformAxis(t3d);
        }
        rotator.setSchedulingBounds(new BoundingSphere(new Point3d(), 150));
        spinTransformGroup.addChild(rotator);
        return spinTransformGroup;
    }

    public static ETransformGroup translate(String name, Node node,
                                            float x, float y, float z) {
        Transform3D t = new Transform3D();
        t.setTranslation(new Vector3f(x, y, z));
        ETransformGroup tg = new ETransformGroup(name, t);
        tg.addChild(node);
        return tg;
    }

}