package classUtils.resolver.model;

//import java.util.Vector;
import classUtils.resolver.interfaces.Auditor;

import java.util.ArrayList;
import java.util.Map;
import java.util.TreeMap;
import java.util.jar.JarFile;

/**
 * @author Douglas Tillman, VTE Consulting, 12/2002
 * @version 2.0
 * http://home.attbi.com/~doug.tillman/index.html
 *
 * Controller class responsible for Auditing the system
 * .jar files and registering any duplicate jar file names
 * and duplicate exploded contents Strings for later reporting.
 * 
 * The results are retained as a master list and a duplicate list - both stored
 * in TreeMap structures.  I opted to keep them separate rather than having
 * a single data structure with an isDuplicate boolean flag on the JarFile object
 * because as the audits get larger, it seemed more efficient if one wanted to
 * process just the duplicates, to not have to iterate over the entire inventory.
 * This may change in a later version depending on the feedback to the utility.
 */
public class AuditMaster implements Auditor {

	private Map unqualMasterJarList;
	private Map masterJarList;
	private Map duplicateJarList;
	private Map masterClassList;
	private Map duplicateClassList;
	private Map unqualClassNameList;
	private ArrayList jarClassNames;

	public AuditMaster() {
		
		unqualMasterJarList = new TreeMap();
		masterJarList = new TreeMap();
		duplicateJarList = new TreeMap();
		masterClassList = new TreeMap();
		duplicateClassList = new TreeMap();
		unqualClassNameList = new TreeMap();
		ArrayList jarClassNames = new ArrayList();
		
	}
	
	/**
	 * Add the JarFile object to the audit for evaluation and 
	 * addition to the appropriate TreeMap and consequently the 
	 * corresponding pane in the GUI splitpane widget.
	 * @param The current JarFile being evaluated.
	 */
	public void addJarFileToAudit(JarFile jar){
		
		AuditableJarFile j = (AuditableJarFile)jar;
		
		//Check if the file name is already in the unqual master audit list (Map)
		//and if not, put it in. The unqual list holds just the file name but
		//the key in the masterJarList and the duplicatesJarList has to be a
		//unique file path or the map entry will be overwritten.
		if(!(unqualMasterJarList.containsKey(j.getFile().getName()))) {
			unqualMasterJarList.put(j.getFile().getName(), j);
			masterJarList.put(j.getFile().getAbsolutePath(), j);			
		} else {
			//if already in the Map then put the file in the duplicates Map.
			duplicateJarList.put(j.getFile().getAbsolutePath(), j);
		}
		
		//Obtain an ArrayList of all the classes referenced in the .jar file
		jarClassNames = j.getExplodedContents();
		
		//Iterate through the ArrayList
		for(int x = 0; x < jarClassNames.size(); x++) {
			
		  if(jarClassNames.get(x) != null) {
					
			String qualClassName = ((String)jarClassNames.get(x)).trim();
		
			int posLastSlash = qualClassName.lastIndexOf('/');
			String unqualClassName = (qualClassName.substring(
										posLastSlash, qualClassName.length())).trim();			

			//Similar to above for .jar files - check if class is already in the Master
			//class list and if not, put it in.  Otherwise, add it to the duplicates
			//Map.
			if(!(unqualClassNameList.containsKey(unqualClassName))){
				unqualClassNameList.put(unqualClassName, j);
				masterClassList.put(qualClassName, j);
			} else {
				duplicateClassList.put(qualClassName, j);
			}
		  }//end check for null
		}	//end for loop			
	}

	//Standard getter methods to return the various Maps of the JarFile objects
	//and any duplicate .jar's.
	public Map getMasterJarList(){
		return masterJarList;
	}
	public Map getDuplicateJarList(){
		return duplicateJarList;
	}
	public Map getMasterClassList(){
		return masterClassList;
	}
	public Map getDuplicateClassList(){
		return duplicateClassList;
	}
	

}
