package classUtils.javassist.sample.vector;

import classUtils.javassist.*;
import classUtils.javassist.preproc.Assistant;

import java.io.IOException;

/**
 * This is a Javassist program which produce a new class representing
 * vectors of a given type.  For example,
 *
 * <ul>import java.util.Vector by sample.vector.VectorAssistant(int)</ul>
 *
 * <p>requests the Javassist preprocessor to substitute the following
 * lines for the original import declaration:
 *
 * <ul><pre>
 * import java.util.Vector;
 * import sample.vector.intVector;
 * </pre></ul>
 *
 * <p>The Javassist preprocessor calls <code>VectorAssistant.assist()</code>
 * and produces class <code>intVector</code> equivalent to:
 *
 * <ul><pre>
 * package sample.vector;
 *
 * public class intVector extends Vector {
 *   pubilc void add(int value) {
 *     addElement(new Integer(value));
 *   }
 *
 *   public int at(int index) {
 *     return elementAt(index).intValue();
 *   }
 * }
 * </pre></ul>
 *
 * <p><code>VectorAssistant.assist()</code> uses
 * <code>sample.vector.Sample</code> and <code>sample.vector.Sample2</code>
 * as a template to produce the methods <code>add()</code> and
 * <code>at()</code>.
 */
public class VectorAssistant implements Assistant {
    public final String packageName = "sample.vector.";

    /**
     * Calls <code>makeSubclass()</code> and produces a new vector class.
     * This method is called by a <code>javassist.preproc.Compiler</code>.
     *
     * @see classUtils.javassist.preproc.Compiler
     */
    public CompileTimeClass[] assist(ClassPool pool, String vec, String[] args)
            throws CannotCompileException {
        if (args.length != 1)
            throw new CannotCompileException(
                    "VectorAssistant receives a single argument.");

        try {
            CompileTimeClass subclass;
            CompileTimeClass elementType = pool.get(args[0]);
            if (elementType.isPrimitive())
                subclass = makeSubclass2(pool, elementType);
            else
                subclass = makeSubclass(pool, elementType);

            CompileTimeClass[] results = {subclass, pool.get(vec)};
            return results;
        } catch (NotFoundException e) {
            throw new CannotCompileException(e);
        } catch (IOException e) {
            throw new CannotCompileException(e);
        }
    }

    /**
     * Produces a new vector class.  This method does not work if
     * the element type is a primitive type.
     *
     * @param type	the type of elements
     */
    public CompileTimeClass makeSubclass(ClassPool pool, CompileTimeClass type)
            throws CannotCompileException, NotFoundException, IOException {
        CompileTimeClass vec = pool.makeClass(makeClassName(type));
        vec.setSuperclass(pool.get("java.util.Vector"));

        CompileTimeClass c = pool.get("sample.vector.Sample");
        CtMethod addmethod = c.getDeclaredMethod("add");
        CtMethod atmethod = c.getDeclaredMethod("at");

        ClassMap map = new ClassMap();
        map.put("sample.vector.X", type.getName());

        vec.addMethod(CtNewMethod.copy(addmethod, "add", vec, map));
        vec.addMethod(CtNewMethod.copy(atmethod, "at", vec, map));
        pool.writeFile(vec.getName());
        return vec;
    }

    /**
     * Produces a new vector class.  This uses wrapped methods so that
     * the element type can be a primitive type.
     *
     * @param type	the type of elements
     */
    public CompileTimeClass makeSubclass2(ClassPool pool, CompileTimeClass type)
            throws CannotCompileException, NotFoundException, IOException {
        CompileTimeClass vec = pool.makeClass(makeClassName(type));
        vec.setSuperclass(pool.get("java.util.Vector"));

        CompileTimeClass c = pool.get("sample.vector.Sample2");
        CtMethod addmethod = c.getDeclaredMethod("add");
        CtMethod atmethod = c.getDeclaredMethod("at");

        CompileTimeClass[] args1 = {type};
        CompileTimeClass[] args2 = {CompileTimeClass.intType};
        CtMethod m
                = CtNewMethod.wrapped(CompileTimeClass.voidType, "add", args1,
                        null, addmethod, null, vec);
        vec.addMethod(m);
        m = CtNewMethod.wrapped(type, "at", args2,
                null, atmethod, null, vec);
        vec.addMethod(m);
        pool.writeFile(vec.getName());
        return vec;
    }

    private String makeClassName(CompileTimeClass type) {
        return packageName + type.getSimpleName() + "Vector";
    }
}
