/*
 *  SwitchTest.java
 *  Shows how to use the StringSwitch
 *  class to use switch based on strings.
 *  Runs performance tests.
 *
 *  @author Thomas Rowland
 *  @version november 5, 2002
 */

package bookExamples.ch05ControlStructs;

import utils.Timer;

import java.util.HashMap;
import java.util.Map;

public class SwitchTest extends StringSwitch {

    static SwitchTest st;
    static Timer timer;
    static int COUNT;
    static final int ITER = 20;

    // StringSwitch constants
    public final int QUIT = 0;
    public final int DIR = 1;
    public final int RUN = 2;
    public final int STOP = 3;
    public final int DEBUG = 4;
    public final int FWD = 5;
    public static final int REV = 6;

    /** Hashmap for storing String/int pairs.
     (local testing. see <fast>)  */
    static Map hashMap = new HashMap();

    /** Creates a new instance of SwitchTest */
    public SwitchTest() {
        /* Add the String/int pairs to the hashmap.
           key=String, value=static final int */
        add("quit", 0);
        add("dir", 1);
        add("run", 2);
        add("stop", 3);
        add("debug", 4);
        add("forward", 5);
        add("reverse", 6);
        initTypes();
    }

    /** Tests StringSwitch */
    public static void main(String[] args) {
        COUNT = 1000000;
        System.out.println("COUNT = " + COUNT + "\n");


        getSwitchTest();
        int id = st.getIdForString("TIFF_BIG_ENDIAN");
        System.out.println("id=" + id);

        timer = new Timer();

        ifTestBottom();

        ifTestTop();

        ifTestSwitch();

        ifTestInt();

        ifTestFast();

    }//main

    private static void getSwitchTest() {
        st = new SwitchTest();

        /* (for local testing. see <fast>)  */
        st.addKeyValue("quit", 0);
        st.addKeyValue("dir", 1);
        st.addKeyValue("run", 2);
        st.addKeyValue("stop", 3);
        st.addKeyValue("debug", 4);
        st.addKeyValue("forward", 5);
        st.addKeyValue("reverse", 6);
    }

    private static void ifTestBottom() {
        /*  Test If-Then where match is at bottom */
        System.out.println("\nTesting if-then (Bottom)");
        for (int i = 0; i < ITER; i++) {
            timer.clear();
            timer.start();
            for (int j = 0; j < COUNT; j++) {
                st.doIf("reverse");
            }
            timer.stop();
            System.out.println(timer.getElapsedTime());
        }
    }

    private static void ifTestFast() {
        /*  Test <fast> Switch with a (String) where hashmap is local.
              The purpose of this is to determine the amount
              of contribution that the method calls on the StringSwitch object
              have on performance. */
        System.out.println("\nTesting Switch with a (String) <fast>");
        for (int i = 0; i < ITER; i++) {
            timer.clear();
            timer.start();
            for (int j = 0; j < COUNT; j++)
                st.fastSwitch("reverse");
            timer.stop();
            System.out.println(timer.getElapsedTime());
        }
    }

    private static void ifTestInt() {
        /*  Test Switch with an (int) */
        System.out.println("\nTesting Switch with an (int)");
        for (int i = 0; i < ITER; i++) {
            timer.clear();
            timer.start();
            for (int j = 0; j < COUNT; j++) {
                st.doSwitch2(REV);
            }
            timer.stop();
            System.out.println(timer.getElapsedTime());
        }
    }

    private static void ifTestSwitch() {
        /*  Test Switch with a (String) */
        System.out.println("\nTesting Switch with a (String)");
        for (int i = 0; i < ITER; i++) {
            timer.clear();
            timer.start();
            for (int j = 0; j < COUNT; j++) {
                st.doSwitch("reverse");
            }
            timer.stop();
            System.out.println(timer.getElapsedTime());
        }
    }

    private static void ifTestTop() {
        /*  Test If-Then where match is at top */
        System.out.println("\nTesting if-then (Top)");
        for (int i = 0; i < ITER; i++) {
            timer.clear();
            timer.start();
            for (int j = 0; j < COUNT; j++) {
                st.doIf("quit");
            }
            timer.stop();
            System.out.println(timer.getElapsedTime());
        }
    }


    /**
     *  switch structure that accepts Strings.
     *  @param s the search String
     */
    private void doSwitch(String s) {
        switch (getIdForString(s)) {
            case QUIT:
                //System.out.print ("QUIT=" + QUIT);
                break;
            case DIR:
                //System.out.print ("DIR=" + DIR);
                break;
            case RUN:
                //System.out.print ("RUN=" + RUN);
                break;
            case STOP:
                //System.out.print ("STOP=" + STOP);
                break;
            case DEBUG:
                //System.out.print ("DEBUG=" + DEBUG);
                break;
            case FWD:
                //System.out.print ("FWD=" + FWD);
                break;
            case REV:
                //System.out.print ("REV=" + REV);
                break;
        }
    }

    /**
     *  Standard switch structure that accepts ints.
     *  @param i the search int
     */
    private void doSwitch2(int i) {
        switch (i) {
            case QUIT:
                //System.out.print ("QUIT=" + QUIT);
                break;
            case DIR:
                //System.out.print ("DIR=" + DIR);
                break;
            case RUN:
                //System.out.print ("RUN=" + RUN);
                break;
            case STOP:
                //System.out.print ("STOP=" + STOP);
                break;
            case DEBUG:
                //System.out.print ("DEBUG=" + DEBUG);
                break;
            case FWD:
                //System.out.print ("FWD=" + FWD);
                break;
            case REV:
                //System.out.print ("REV=" + REV);
                break;
        }
    }

    /**
     *  <fast> switch structure that accepts Strings.
     *  Accesses the hashmap locally instead of calling on object.
     *  @param s the search String
     */
    private void fastSwitch(String s) {
        switch (((Integer) hashMap.get(s)).intValue()) {
            case QUIT:
                //System.out.print ("QUIT=" + QUIT);
                break;
            case DIR:
                //System.out.print ("DIR=" + DIR);
                break;
            case RUN:
                //System.out.print ("RUN=" + RUN);
                break;
            case STOP:
                //System.out.print ("STOP=" + STOP);
                break;
            case DEBUG:
                //System.out.print ("DEBUG=" + DEBUG);
                break;
            case FWD:
                //System.out.print ("FWD=" + FWD);
                break;
            case REV:
                //System.out.print ("REV=" + REV);
                break;
        }
    }

    /**
     *  if-then structure operating on a String.
     *  @param s the search String
     */
    private void doIf(String s) {
        if (s.equals("quit")) {
            //System.out.print ("QUIT=" + QUIT);
        } else if (s.equals("dir")) {
            //System.out.print ("DIR=" + DIR);
        } else if (s.equals("run")) {
            //System.out.print ("RUN=" + RUN);
        } else if (s.equals("stop")) {
            //System.out.print ("STOP=" + STOP);
        } else if (s.equals("debug")) {
            //System.out.print ("DEBUG=" + DEBUG);
        } else if (s.equals("forward")) {
            //System.out.print ("FWD=" + FWD);
        } else if (s.equals("reverse")) {
            //System.out.print ("REV=" + REV);
        }
    }

    public String getStringForId(int id) {
        return (String) hashMap.get(new Integer(id));
    }

    public int getIdForString(String s) {
        Integer i = (Integer) hashMap.get(s);
        return i.intValue();
    }

    /**
     *  Adds a new key-value pair to the hashmap.
     *  Used for testing using local hashmap.
     */
    private void addKeyValue(String s, final int id) {
        hashMap.put(s, new Integer(id));
    }

    private void initTypes() {
        add(TYPENOTFOUND, "TYPENOTFOUND");
        add(UUENCODED, "UUENCODED");
        add(BTOAD, "BTOAD");
        add(PBM, "PBM");
        add(PGM, "PGM");
        add(PPM, "PPM");
        add(PBM_RAWBITS, "PBM_RAWBITS");
        add(PGM_RAWBITS, "PGM_RAWBITS");
        add(PPM_RAWBITS, "PPM_RAWBITS");
        add(MGR_BITMAP, "MGR_BITMAP");
        add(GIF87a, "GIF87a");
        add(GIF89a, "GIF89a");
        add(IFF_ILBM, "IFF_ILBM");
        add(SUNRASTER, "SUNRASTER");
        add(SGI_IMAGE, "SGI_IMAGE");
        add(CMU_WINDOW_MANAGER_BITMAP, "CMU_WINDOW_MANAGER_BITMAP");
        add(SUN, "SUN");
        add(TIFF_BIG_ENDIAN, "TIFF_BIG_ENDIAN");
        add(TIFF_LITTLE_ENDIAN, "TIFF_LITTLE_ENDIAN");
        add(FLI, "FLI");
        add(MPEG, "MPEG");
        add(SUN_NEXT_AUDIO, "SUN_NEXT_AUDIO");
        add(STANDARD_MIDI, "STANDARD_MIDI");
        add(MICROSOFT_RIFF, "MICROSOFT_RIFF");
        add(BZIP, "BZIP");
        add(IFF_DATA, "IFF_DATA");
        add(NIFF_IMAGE, "NIFF_IMAGE");
        add(PC_BITMAP, "PC_BITMAP");
        add(PDF_DOCUMENT, "PDF_DOCUMENT");
        add(POSTSCRIPT_DOCUMENT, "POSTSCRIPT_DOCUMENT");
        add(SILICON_GRAPHICS_MOVIE, "SILICON_GRAPHICS_MOVIE");
        add(APPLE_QUICKTIME_MOVIE, "APPLE_QUICKTIME_MOVIE");
        add(ZIP_ARCHIVE, "ZIP_ARCHIVE");
        add(UNIX_COMPRESS, "UNIX_COMPRESS");
        add(GZIP, "GZIP");
        add(HUFFMAN, "HUFFMAN");
        add(PNG_IMAGE, "PNG_IMAGE");
        add(JPEG, "JPEG");
        add(JPG, "JPG");
        add(PSHOP8, "Photo Shop...8 bits per pel");
        add(ZIP, "Zip file...Wavelet encoded image sequence?");
    }

    public int ifTest(String s) {

        if (s.startsWith("TYPENOTFOUND"))
            return TYPENOTFOUND;
        if (s.startsWith("UUENCODED"))
            return UUENCODED;
        if (s.startsWith("BTOAD"))
            return BTOAD;
        if (s.startsWith("PBM"))
            return PBM;
        if (s.startsWith("PGM"))
            return PGM;
        if (s.startsWith("PPM"))
            return PPM;
        if (s.startsWith("PBM_RAWBITS"))
            return PBM_RAWBITS;
        if (s.startsWith("PGM_RAWBITS"))
            return PGM_RAWBITS;
        if (s.startsWith("PPM_RAWBITS"))
            return PPM_RAWBITS;
        if (s.startsWith("MGR_BITMAP"))
            return MGR_BITMAP;
        if (s.startsWith("GIF87a"))
            return GIF87a;
        if (s.startsWith("GIF89a"))
            return GIF89a;
        if (s.startsWith("IFF_ILBM"))
            return IFF_ILBM;
        if (s.startsWith("SUNRASTER"))
            return SUNRASTER;
        if (s.startsWith("SGI_IMAGE"))
            return SGI_IMAGE;
        if (s.startsWith("CMU_WINDOW_MANAGER_BITMAP"))
            return CMU_WINDOW_MANAGER_BITMAP;
        if (s.startsWith("SUN"))
            return SUN;
        if (s.startsWith("TIFF_BIG_ENDIAN"))
            return TIFF_BIG_ENDIAN;
        if (s.startsWith("TIFF_LITTLE_ENDIAN"))
            return TIFF_LITTLE_ENDIAN;
        if (s.startsWith("FLI"))
            return FLI;
        if (s.startsWith("MPEG"))
            return MPEG;
        if (s.startsWith("SUN_NEXT_AUDIO"))
            return SUN_NEXT_AUDIO;
        if (s.startsWith("STANDARD_MIDI"))
            return STANDARD_MIDI;
        if (s.startsWith("MICROSOFT_RIFF"))
            return MICROSOFT_RIFF;
        if (s.startsWith("BZIP"))
            return BZIP;
        if (s.startsWith("IFF_DATA"))
            return IFF_DATA;
        if (s.startsWith("NIFF_IMAGE"))
            return NIFF_IMAGE;
        if (s.startsWith("PC_BITMAP"))
            return PC_BITMAP;
        if (s.startsWith("PDF_DOCUMENT"))
            return PDF_DOCUMENT;
        if (s.startsWith("POSTSCRIPT_DOCUMENT"))
            return POSTSCRIPT_DOCUMENT;
        if (s.startsWith("SILICON_GRAPHICS_MOVIE"))
            return SILICON_GRAPHICS_MOVIE;
        if (s.startsWith("APPLE_QUICKTIME_MOVIE"))
            return APPLE_QUICKTIME_MOVIE;
        if (s.startsWith("ZIP_ARCHIVE"))
            return ZIP_ARCHIVE;
        if (s.startsWith("UNIX_COMPRESS"))
            return UNIX_COMPRESS;
        if (s.startsWith("GZIP"))
            return GZIP;
        if (s.startsWith("HUFFMAN"))
            return HUFFMAN;
        if (s.startsWith("PNG_IMAGE"))
            return PNG_IMAGE;
        if (s.startsWith("JPEG"))
            return JPEG;
        if (s.startsWith("JPG"))
            return JPG;
        if (s.startsWith("Photo Shop...8 bits per pel"))
            return PSHOP8;
        if (s.startsWith("Zip file...Wavelet encoded image sequence?"))
            return ZIP;
        else
            return -1;
    }

    protected void add(int i, String s) {
        hashMap.put(s, new Integer(i));
    }

    public static final int TYPENOTFOUND = 0;
    public static final int UUENCODED = 1;
    public static final int BTOAD = 2;
    public static final int PBM = 3;
    public static final int PGM = 4;
    public static final int PPM = 5;
    public static final int PBM_RAWBITS = 6;
    public static final int PGM_RAWBITS = 7;
    public static final int PPM_RAWBITS = 8;
    public static final int MGR_BITMAP = 9;
    public static final int GIF87a = 10;
    public static final int GIF89a = 11;
    public static final int IFF_ILBM = 12;
    public static final int SUNRASTER = 13;
    public static final int SGI_IMAGE = 14;
    public static final int CMU_WINDOW_MANAGER_BITMAP = 15;
    public static final int SUN = 16;
    public static final int TIFF_BIG_ENDIAN = 17;
    public static final int TIFF_LITTLE_ENDIAN = 18;
    public static final int FLI = 19;
    public static final int MPEG = 20;
    public static final int SUN_NEXT_AUDIO = 21;
    public static final int STANDARD_MIDI = 22;
    public static final int MICROSOFT_RIFF = 23;
    public static final int BZIP = 24;
    public static final int IFF_DATA = 25;
    public static final int NIFF_IMAGE = 26;
    public static final int PC_BITMAP = 27;
    public static final int PDF_DOCUMENT = 28;
    public static final int POSTSCRIPT_DOCUMENT = 29;
    public static final int SILICON_GRAPHICS_MOVIE = 30;
    public static final int APPLE_QUICKTIME_MOVIE = 31;
    public static final int ZIP_ARCHIVE = 32;
    public static final int UNIX_COMPRESS = 33;
    public static final int GZIP = 34;
    public static final int HUFFMAN = 35;
    public static final int PNG_IMAGE = 38;
    public static final int JPEG = 39;
    public static final int JPG = 40;
    public static final int PSHOP8 = 41;
    public static final int ZIP = 42;

}//SwitchTest

