package sound;

import gui.ClosableJFrame;
import gui.In;
import gui.run.RunJob;
import math.transforms.DFT;
import sound.ulaw.UlawCodec;

import java.awt.*;

public class Oscillator {
    private double audioData[];
    private double waveTable[];
    private int sampleRate = 8000;
    private double frequency;
    private double lambda;
    private double samplesPerCycle;

    // private double delta_freq;

    private final static double twopi = Math.PI * 2;
    private double frequencyOfModulation;
    private double modulationIndex;

    public Oscillator(double frequency_, int length) {
        frequency = frequency_;
        audioData = new double[length];

        //the period of the wave form is
        lambda = 1 / frequency;

        //The number of samples per period is
        samplesPerCycle = sampleRate * lambda;

        //delta_freq = 1 / samplesPerCycle;
        waveTable =
                new double[(int) Math.round(samplesPerCycle)];
        System.out.println("Wavetable size=" + waveTable.length);
    }
    /**
     * Plays the system beep. I only put this here
     * because I did not know where else to put it.
     * After all, it uses the AWT tool kit, and how
     * does that make any sense?
     */
    public static void beep() {
        final Toolkit tk = Toolkit.getDefaultToolkit();
        tk.beep();
    }

    public static void main(String arg[]) {
        beep();
    }

    public static void testPlayTone() {
        while (true)
            playTone(220,In.getInt("enter duration"));
    }

    private static void playTone(int f, int dur) {
        Oscillator osc = new Oscillator(f, dur);
        UlawCodec ulc = new UlawCodec(osc.getSineWave());
        ulc.play();
    }

    public static void asynchronousToneTest() {
        RunJob rj1 = new RunJob(.25, true, 8) {
            public void run() {
                playTone(440);
            }
        };
        RunJob rj2 = new RunJob(2, true, 2) {
            public void run() {
                playTone(230);
            }
        };
        rj1.start();
        rj2.start();
    }

    public static void testPlayAndDisplayTone() {
        playAndDisplayTone(440);
        playAndDisplayTone(220);
    }

    public static void playAndDisplayTone(int frequency) {
        Oscillator osc = new Oscillator(frequency, 8000);
        double d [] = osc.getSineWave();
        UlawCodec ulc = new UlawCodec(d);
        ulc.displayInternalData();
        System.out.println("freq="+ulc.getFrequency());
    }

    public static void playTone(int frequency) {
        Oscillator osc = new Oscillator(frequency, 8000);
        double d [] = osc.getSineWave();
        UlawCodec ulc = new UlawCodec(d);
        ulc.play();
    }

    public double actualFrequency() {
        return (double) sampleRate / (double) waveTable.length;
    }

    private double[] AudioDataFromTable() {
        int k = 0;
        for (int i = 0; i < audioData.length; i++) {
            audioData[i] = waveTable[k];
            k++;
            if (k >= waveTable.length)
                k = 0;
        }
        System.out.println("\nlambda=" + lambda +
                "\nfrequency = " + frequency +
                "\nwaveTable.length = " + waveTable.length +
                "\nsampleRate = " + sampleRate +
                "\naudioData.length = " + audioData.length +
                "\nactual frequency = " + actualFrequency());
        return audioData;
    }

    public double[] getSineWave() {
        for (int i = 0; i < waveTable.length; i++)
            waveTable[i] =
                    0.98 * Math.sin(twopi * i / waveTable.length);
        return AudioDataFromTable();
    }

    public double[] getSquareWave() {
        getSawWave();
        for (int i = 0; i < waveTable.length; i++)
            if (waveTable[i] > 0)
                waveTable[i] = .98;
            else
                waveTable[i] = -.98;
        return AudioDataFromTable();
    }

    public double[] getSawWave() {
        double v = -0.99;
        double dv = 2.0 / (double) waveTable.length;
        for (int i = 0; i < waveTable.length; i++) {
            waveTable[i] = v;
            v += dv;
        }
        System.out.println("Sawwave ends at:" + (v - dv));
        return AudioDataFromTable();
    }

    public double[] getTriangleWave() {
        int sign;
        double v = 0;
        double dv = 3.d / (double) waveTable.length;
        for (int i = 0; i < waveTable.length; i++) {
            waveTable[i] = v;
            sign =
                    sign(Math.cos(twopi * i / waveTable.length));
            v = v + sign * dv;
        }
        return AudioDataFromTable();
    }

    public int sign(double d) {
        if (d < 0) return -1;
        return 1;
    }

    public double getDuration() {
        return
                ((double) audioData.length /
                (double) sampleRate);
    }

    public int getSampleRate() {
        return sampleRate;
    }

    public double getFrequency() {
        return frequency;
    }

    public void setModulationIndex(double I) {
        modulationIndex = I;
    }

    public void setModulationFrequency(double fm) {
        frequencyOfModulation = fm;
    }

    public double[] getFM() {
        double r1 = twopi * frequency / sampleRate;
        double r2 = twopi * frequencyOfModulation / sampleRate;
        for (int n = 0; n < audioData.length; n++)
            audioData[n] =
                    Math.sin(r1 * n +
                    +modulationIndex * Math.sin(r2 * n));
        return audioData;
    }

    public double[] getAM() {
        double r1 = twopi * frequency / sampleRate;
        double r2 = twopi * frequencyOfModulation / sampleRate;
        for (int n = 0; n < audioData.length; n++)
            audioData[n] =
                    Math.cos(r1 * n) * Math.cos(r2 * n);
        return audioData;
    }

    public static void testAudioDft() {
        Oscillator o = new Oscillator(440, 64);
        double d[] = o.getSineWave();
        DFT f = new DFT(d.length);
        f.dft(d);
        double psd[] = f.getPowerSpectralDensity();
        OscopePanel op = new OscopePanel(psd);
        ClosableJFrame cjf = new ClosableJFrame("psd");
        Container c = cjf.getContentPane();
        c.setLayout(new BorderLayout());
        c.add(op, BorderLayout.CENTER);
        cjf.setSize(400, 400);
        cjf.show();
    }
}