/*
 * The org.opensourcephysics.numerics package contains numerical methods
 * for the book Simulations in Physics.
 * Copyright (c) 2004  H. Gould, J. Tobochnik, and W. Christian.
 */
package math.numerics;

/**
 * Euler implements an Euler method ODE solver.
 *
 * The Euler method is unstable for many systems.  It is included as an  example of
 * how to use the ODE and ODESolver interface.
 *
 * @author       Wolfgang Christian
 * @version 1.0
 */
public class Euler extends Object implements ODESolver {

   private double stepSize = 0.1; // parameter increment such as delta time
   private int  numEqn   = 0;     // number of equations
   private double[] rate;         // array that stores the rate
   private ODE ode;               // object that computes rate

  /**
   * Constructs the Euler ODESolver for a system of ordinary differential equations.
   *
   * @param _ode the system of differential equations.
   */
   public Euler(ODE _ode) {
      ode = _ode;
      initialize(0.1);
   }

  /**
   * Steps (advance) the differential equations by the stepSize
   *
   * The ODESolver invokes the ODE's getRate method to obtain the initial state of the system.
   * The ODESolver then advances the solution and copies the new state into the
   * state array at the end of the solution step.
   *
   * @return the step size
   */
   public double step() {
      double[] state = ode.getState();
      ode.getRate(state, rate);
      for (int i = 0; i < numEqn; i++) {
         state[i] = state[i] + stepSize * rate[i];
      }
      return stepSize;
   }

  /**
   * Sets the step size
   *
   * The step size remains fixed in this algorithm
   *
   * @param _stepSize
   */
   public void setStepSize(double _stepSize) {
      stepSize = _stepSize;
   }

  /**
   * Initializes the ODE solver
   *
   * The rate array is allocated.  The number of differential equations is
   * determined by invoking getState().length on the ODE.
   *
   * @param _stepSize
   */
   public void initialize(double _stepSize) {
      stepSize = _stepSize;
      double state[] = ode.getState();
      if (state == null) {  // state vector not defined
         numEqn = 0;
      } else {
         numEqn = state.length;
      }
      rate = new double[numEqn];
   }

  /**
   * Gets the step size
   *
   * The stepsize is constant in this algorithm
   *
   * @return the step size
   */
   public double getStepSize() {
      return stepSize;
   }
}

