/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */
package j3d.polyViewer;


import futils.Futil;
import gui.ClosableJFrame;
import gui.layouts.DialogLayout;
import gui.run.RunMenu;
import gui.run.RunMenuItem;
import gui.run.RunSpinnerSlider;
import j2d.ImageUtils;
import j3d.polyViewer.vec3.*;
import j3d.utils.PolyhedralShape3d;
import j3d.utils.RunTexCoordGenerationPanel;
import j3d.utils.TriangleGeometryUtils;
import j3d.utils.Utils;
import utils.ResourceManager;

import javax.media.j3d.*;
import javax.swing.*;
import javax.vecmath.Color3f;
import java.awt.*;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.io.File;


public abstract class RunShapePanel
        extends JTabbedPane implements Runnable,
        MouseMotionListener,
        MouseListener {
    Vec3FunctionInterface shapesFunctionInterface[] = {
        new Torus(),
        new Mobius(),
        new Klein(),
        new Gauss(),
        new DerivativeOfGauss(),
        new LaplacianOfGauss(),
        new MessedUp(),
        new GaborCos(),
        new GaborSin(),
        //new ParabolicCyn(),
        new Cone(),
        new Ellipsoid(),
        new ParabolicCylinder(),
        //new RealElipticalCylinder(),
        new HyperbolicParaboloid(),
        new EllipticParaboloid(),
        new ConicalSpiral(),
        new CylindricalSpiral(),
        new Cornucopia(),
        new FrenelsElasticitySurface(),
        new ZonalHarmonics(),
        new SpaceStation(),
        new Tubular()

    };
    // todo add a means of interactivly adding functions at run-time.
    // perhaps java-assist?
    Vec3FunctionInterface currentShape = shapesFunctionInterface[0];
    // # divisions around equator
    private SpinnerNumberModel n1 = new SpinnerNumberModel(3,
            3,
            200,
            1);
    // # divisions through hole
    private SpinnerNumberModel n2 = new SpinnerNumberModel(2,
            2,
            200,
            1);
    // mid-radius   (0 to 1)
    private SpinnerNumberModel r1 = new SpinnerNumberModel(22,
            0,
            1000,
            1);
    // small radius (0 to 1)
    private SpinnerNumberModel r2 = new SpinnerNumberModel(1,
            1,
            1000,
            1);
    private TexCoordGeneration texCoordGen = new TexCoordGeneration(TexCoordGeneration.OBJECT_LINEAR,
            TexCoordGeneration.TEXTURE_COORDINATE_2);
    private Texture imageTexture = null;
    private Appearance appearance = getAppearance();
    private Vec3[][] geometry = getGeometry();


    public RunShapePanel() {
        setToolTipText("A Shape Panel!");

        addTab("dimensions",
                null,
                getDimensionPanel(),
                "dimensions");
        addTab("TextureCoordGenerator",
                null, getTexCoordGenerationPanel(),
                "TextureCoordGenerator");
    }

    private Component getTexCoordGenerationPanel() {
        RunTexCoordGenerationPanel runTexCoordGenerationPanel
                = new RunTexCoordGenerationPanel() {
                    public void run() {
                        texCoordGen = super.getValue();
                        appearance = getAppearance();
                        RunShapePanel.this.run();
                    }
                };
        return runTexCoordGenerationPanel;
    }

    private Component getDimensionPanel() {
        JPanel jp = new JPanel();

        jp.setLayout(new DialogLayout());
        jp.add(new JLabel("n1"));

        jp.add(new RunSpinnerSlider(n1) {
            public void run() {
                n1.setValue(new Integer(getValue()));
                geometry = getGeometry();
                RunShapePanel.this.run();
            }
        });
        jp.add(new JLabel("n2"));

        jp.add(new RunSpinnerSlider(n2) {
            public void run() {
                n2.setValue(new Integer(getValue()));
                geometry = getGeometry();
                RunShapePanel.this.run();
            }
        });
        jp.add(new JLabel("r1"));

        jp.add(new RunSpinnerSlider(r1) {
            public void run() {
                r1.setValue(new Integer(getValue()));
                geometry = getGeometry();
                RunShapePanel.this.run();
            }
        });
        jp.add(new JLabel("r2"));

        jp.add(new RunSpinnerSlider(r2) {
            public void run() {
                r2.setValue(new Integer(getValue()));
                geometry = getGeometry();
                RunShapePanel.this.run();
            }
        });
        return jp;
    }


    private static final int getValue(SpinnerNumberModel nm) {
        return ((Integer) nm.getValue()).intValue();
    }

    public Shape3D getShape3D() {

        return new PolyhedralShape3d(geometry, appearance);
    }

    public Appearance getAppearance() {
        if (imageTexture == null)
            return getUntexturedAppearance();
        return getTexturedAppearance();
    }

    private Appearance getTexturedAppearance() {
        Appearance app = new Appearance();
        app.setTexture(imageTexture);
        setCapabilityBits(app);
        app.setTexCoordGeneration(texCoordGen);
        return app;
    }

    private Texture getImageTexture(String imageFile) {
        ResourceManager rm = ResourceManager.getResourceManager();
        Image img = ImageUtils.getImage(rm.getImageFile(imageFile));
        //Image img = null;
        //try {
        //    img = ImageUtils.getImage(CameraUtils.getImageFromCamera());
        //} catch (Exception e) {
        //    img = ImageUtils.getImage(rm.getImageFile(imageFile));
        //}
        Texture imageTexture = Utils.getTexture(img);
        return imageTexture;
    }
    // want is
    // CameraUtils.getImageFromCamera

    private void setCapabilityBits(Appearance app) {
        app.setCapability(Appearance.ALLOW_POLYGON_ATTRIBUTES_READ);
        app.setCapability(Appearance.ALLOW_POLYGON_ATTRIBUTES_WRITE);
        app.setCapability(Appearance.ALLOW_RENDERING_ATTRIBUTES_READ);
    }

    private Appearance getUntexturedAppearance() {
        Appearance app = new Appearance();
        setCapabilityBits(app);
        Color3f objColor = new Color3f(0.5f, 0.5f, 0.9f);
        Color3f black = new Color3f(0.0f, 0.0f, 0.0f);
        Color3f white = new Color3f(1.0f, 1.0f, 1.0f);
        app.setMaterial(new Material(objColor,
                black,
                objColor,
                white,
                100.0f));

        return app;
    }

    private Vec3[][] getGeometry() {
        return getGeometry(currentShape);
    }

    public Vec3[][] getGeometry(Vec3FunctionInterface tp3) {
        TriangleGeometryUtils ptg = new TriangleGeometryUtils(tp3);
        Vec3[][] triangles = ptg.getTriangles(getValue(n1),
                getValue(n2),
                getValue(r1) / 100.,
                getValue(r2) / 100.);
        return triangles;
    }


    public static void main(String[] args) {
        ClosableJFrame cf = new ClosableJFrame();
        Container c = cf.getContentPane();
        c.setLayout(new FlowLayout());
        c.add(new RunShapePanel() {
            public void run() {
                getShape3D();
            }
        });

        cf.setSize(400, 400);
        cf.show();
    }


    public void showInFrame() {
        ClosableJFrame cf = new ClosableJFrame();
        Container c = cf.getContentPane();
        c.setLayout(new FlowLayout());
        c.add(this);

        cf.setSize(400, 400);
        cf.show();
    }

    public void mouseClicked(MouseEvent e) {
        //this.setCenterPoint(e.getX(), e.getY());
    }

    public void mouseEntered(MouseEvent e) {
    }

    public void mouseExited(MouseEvent e) {
    }

    public void mouseReleased(MouseEvent e) {

    }

    public void mousePressed(MouseEvent e) {

    }

    public void mouseDragged(MouseEvent e) {
        int m = e.getModifiers();
        //int xc = getValue(xcNm);
        //int yc = getValue(ycNm);

        //final int dx = (int) (e.getX() - xc);
        //final int dy = (int) (e.getY() - yc);

        if ((m & (Event.SHIFT_MASK)) != 0) {
            //setScale(dx, dy);
        } else if ((m & (Event.ALT_MASK)) != 0) {
            //setRotation(dx, dy);
        } else {
            //setTranslation(dx, dy);
        }
    }

    public void mouseMoved(MouseEvent e) {

    }

    public RunMenu getFileMenu() {
        RunMenu rm = new RunMenu("[File");
        rm.add(new RunMenuItem("open texture image...") {
            public void run() {
                File readFile = Futil.getReadFile("select a jpg or gif");
                if (readFile == null) return; // user canceled

                imageTexture = getImageTexture(
                        readFile.getName());
                appearance = getAppearance();
                RunShapePanel.this.run();
            }
        });
        rm.add(new RunMenuItem("exit{control q}") {
            public void run() {
                System.exit(0);
            }
        });
        return rm;
    }

    public RunMenu getShapeMenu() {
        RunMenu rm = new RunMenu("[shape");
        for (int i = 0; i < shapesFunctionInterface.length; i++) {
            final Vec3FunctionInterface v3fi = shapesFunctionInterface[i];
            String name = v3fi.getClass().getName();
            int periodIndex = name.lastIndexOf(".") + 1;
            String label = name.substring(periodIndex);
            rm.add(new RunMenuItem(label) {
                public void run() {
                    currentShape = v3fi;
                    geometry = getGeometry();
                    RunShapePanel.this.run();
                }
            });
        }
        return rm;
    }
}
