/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */
package j3d.examples.swing;

import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGImageEncoder;
import futils.Futil;

import javax.media.j3d.*;
import javax.vecmath.Vector3d;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.FileOutputStream;

/**
 * Copyright DocJava, inc. User: lyon
 * <p/>
 * Date: Mar 8, 2005
 * <p/>
 * Time: 7:17:26 PM
 */
public class ViewUtils {
    /**
     * an offscreen Canvas3D that is used to perform screen captures
     */
    protected Canvas3D offScreenCanvas3D = null;
    /**
     * the image that is attached to the offscreen Canvas3D and contains
     * the results of screen captures
     */
    protected ImageComponent2D imageComponent = null;
    /**
     * the width of the offscreen Canvas3D
     */
    private static final int offScreenWidth = 400;
    /**
     * the height of the offscreen Canvas3D
     */
    private static final int offScreenHeight = 400;

    /**
     * Create a Java 3D View and attach it to a ViewPlatform
     */
    protected View createView(ViewPlatform vp, Container c) {
        View view = new View();

        PhysicalBody pb = createPhysicalBody();
        PhysicalEnvironment pe = createPhysicalEnvironment();

        view.setPhysicalEnvironment(pe);
        view.setPhysicalBody(pb);

        if (vp != null)
            view.attachViewPlatform(vp);

        view.setBackClipDistance(getBackClipDistance());
        view.setFrontClipDistance(getFrontClipDistance());

        // create the visible canvas
        Canvas3D c3d = createCanvas3D(false);
        view.addCanvas3D(c3d);

        // create the off screen canvas
        view.addCanvas3D(createOffscreenCanvas3D());

        // add the visible canvas to a component
        c.add(c3d, BorderLayout.CENTER);

        return view;
    }

    /**
     * Create a Canvas3D.
     *
     * @param offscreen true to specify an offscreen canvas
     */
    protected Canvas3D createCanvas3D(boolean offscreen) {
        GraphicsConfigTemplate3D gc3D = new GraphicsConfigTemplate3D();
        gc3D.setSceneAntialiasing(GraphicsConfigTemplate.PREFERRED);
        GraphicsDevice gd[] = GraphicsEnvironment.getLocalGraphicsEnvironment()
                .getScreenDevices();

        Canvas3D c3d = new Canvas3D(gd[0].getBestConfiguration(gc3D),
                offscreen);
        c3d.setSize(500, 500);

        return c3d;
    }

    /**
     * Initialize an offscreen Canvas3D.
     */
    protected Canvas3D createOffscreenCanvas3D() {
        offScreenCanvas3D = createCanvas3D(true);
        offScreenCanvas3D.getScreen3D().setSize(offScreenWidth,
                offScreenHeight);
        offScreenCanvas3D.getScreen3D().setPhysicalScreenHeight(0.0254 / 90 * offScreenHeight);
        offScreenCanvas3D.getScreen3D().setPhysicalScreenWidth(0.0254 / 90 * offScreenWidth);

        RenderedImage renderedImage = new BufferedImage(offScreenWidth,
                offScreenHeight,
                BufferedImage.TYPE_3BYTE_BGR);
        imageComponent =
                new ImageComponent2D(ImageComponent.FORMAT_RGB8,
                        renderedImage);
        imageComponent.setCapability(ImageComponent2D.ALLOW_IMAGE_READ);
        offScreenCanvas3D.setOffScreenBuffer(imageComponent);

        return offScreenCanvas3D;
    }

    /**
     * Callback to get the scale factor for the View side of the
     * scenegraph
     */
    protected double getScale() {
        return 3;
    }

    /**
     * Get the TransformGroup for the View side of the scenegraph
     */
    public TransformGroup[] getViewTransformGroupArray() {
        TransformGroup[] tgArray = new TransformGroup[1];
        tgArray[0] = new TransformGroup();

        // move the camera BACK a little...
        // note that we have to invert the matrix as
        // we are moving the viewer
        Transform3D t3d = new Transform3D();
        t3d.setScale(getScale());
        t3d.setTranslation(new Vector3d(0.0, 0.0, -20.0));
        t3d.invert();
        tgArray[0].setTransform(t3d);

        return tgArray;
    }

    /**
     * Adds the View side of the scenegraph to the Locale
     */
    protected void addViewBranchGroup(Locale locale, BranchGroup bg) {
        locale.addBranchGraph(bg);
    }

    /**
     * Creates the PhysicalBody for the View
     */
    protected PhysicalBody createPhysicalBody() {
        return new PhysicalBody();
    }

    /**
     * Creates the PhysicalEnvironment for the View
     */
    protected PhysicalEnvironment createPhysicalEnvironment() {
        return new PhysicalEnvironment();
    }

    /**
     * Returns the distance to the rear clipping plane.
     */
    protected double getBackClipDistance() {
        return 100.0;
    }

    /**
     * Returns the distance to the near clipping plane.
     */
    protected double getFrontClipDistance() {
        return 1.0;
    }

    /**
     * Called to render the scene into the offscreen Canvas3D and save the
     * image (as a JPEG) to disk.
     */
    protected void saveImage() {
        offScreenCanvas3D.renderOffScreenBuffer();
        offScreenCanvas3D.waitForOffScreenRendering();
        System.out.println("Rendered to offscreen");

        try {
            FileOutputStream fileOut = new FileOutputStream(Futil.getWriteFile("select an output jpg file"));

            JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(fileOut);
            encoder.encode(imageComponent.getImage());

            fileOut.flush();
            fileOut.close();
        } catch (Exception e) {
            System.err.println("Failed to save image: " + e);
        }

        System.out.println("Saved image.");
    }

    /**
     * Returns the View Platform Activation Radius
     */
    protected static float getViewPlatformActivationRadius() {
        return 100;
    }

    /**
     * Creates the View Platform for the View
     */
    protected static ViewPlatform createViewPlatform() {
        ViewPlatform vp = new ViewPlatform();
        vp.setViewAttachPolicy(View.RELATIVE_TO_FIELD_OF_VIEW);
        vp.setActivationRadius(getViewPlatformActivationRadius());

        return vp;
    }

    /**
     * Creates the View side BranchGroup. The ViewPlatform is wired in
     * beneath the TransformGroups.
     */
    protected static BranchGroup createViewBranchGroup(TransformGroup[] tgArray,
                                                       ViewPlatform vp) {
        BranchGroup vpBranchGroup = new BranchGroup();

        if (tgArray != null && tgArray.length > 0) {
            Group parentGroup = vpBranchGroup;
            TransformGroup curTg = null;

            for (int n = 0; n < tgArray.length; n++) {
                curTg = tgArray[n];
                parentGroup.addChild(curTg);
                parentGroup = curTg;
            }

            tgArray[tgArray.length - 1].addChild(vp);
        } else
            vpBranchGroup.addChild(vp);

        return vpBranchGroup;
    }
}
