package j3d.examples.light;

/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */

import com.sun.j3d.utils.applet.MainFrame;
import com.sun.j3d.utils.behaviors.vp.OrbitBehavior;
import com.sun.j3d.utils.geometry.Sphere;
import com.sun.j3d.utils.universe.SimpleUniverse;
import com.sun.j3d.utils.universe.ViewingPlatform;
import gui.layouts.LeftAlignComponent;
import gui.run.*;
import j3d.utils.BackgroundTool;
import j3d.utils.Java3DExplorerConstants;
import j3d.utils.KeyPrintBehavior;
import j3d.utils.OffScreenCanvas3D;
import utils.ResourceManager;

import javax.media.j3d.*;
import javax.swing.*;
import javax.vecmath.AxisAngle4f;
import javax.vecmath.Color3f;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3f;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;


public class EnvironmentExplorer extends JApplet implements
        Java3DExplorerConstants {
    ResourceManager rm = ResourceManager.getResourceManager();

    // Scene graph items
    SimpleUniverse u;

    // Light items
    Group lightGroup;
    AmbientLight lightAmbient;
    DirectionalLight lightDirectional;
    PointLight lightPoint;
    SpotLight lightSpot;
    Point3f attenuation = new Point3f(1.0f, 0.0f, 0.0f);
    float spotSpreadAngle = 60; // degrees
    float spotConcentration = 5.0f;


    // Fog items
    Switch fogSwitch;
    IntChooser fogChooser;

    // Background items
    Switch bgSwitch;
    IntChooser bgChooser;

    // Sound items
    // Sound is busted. Check:
    // http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4908425
    //  - DL

    //Switch soundSwitch;
    //IntChooser soundChooser;
    ///BackgroundSound soundBackground;
    //PointSound soundPoint;

    // Display object
    Switch spheresSwitch;
    Switch gridSwitch;

    // image grabber
    boolean isApplication;
    Canvas3D canvas;
    OffScreenCanvas3D offScreenCanvas;
    View view;

    // GUI elements
    JTabbedPane tabbedPane;

    // Config items
    String codeBaseString;
    String outFileBase = "env";
    int outFileSeq = 0;
    static final float OFF_SCREEN_SCALE = 1.0f;
    int colorMode = USE_COLOR;

    // Temporaries that are reused
    Transform3D tmpTrans = new Transform3D();
    Vector3f tmpVector = new Vector3f();
    AxisAngle4f tmpAxisAngle = new AxisAngle4f();

    // configurable colors.  These get set based on the rendering
    // mode.  By default they use color. B&W is set up for print
    // file output: white background with B&W coloring.
    Color3f objColor;

    // geometric constants
    Point3f origin = new Point3f();


    /* Set up the lights.  This is a group which contains the
       ambient light and a switch for the other lights.
	directional   : white light pointing along Z axis
	point	      : white light near upper left corner of spheres
	spot	      : white light near upper left corner of spheres,
			      pointing towards center.
     */
    void setupLights() {

        lightGroup = new Group();

        // Set up the ambient light
        lightAmbient = new AmbientLight(darkGrey);
        lightAmbient.setInfluencingBounds(infiniteBounds);
        lightAmbient.setCapability(Light.ALLOW_STATE_WRITE);
        lightAmbient.setEnable(true);
        lightGroup.addChild(lightAmbient);

        // Set up the directional light
        Vector3f lightDirection = new Vector3f(0.65f, -0.65f, -0.40f);
        lightDirectional = new DirectionalLight(white, lightDirection);
        lightDirectional.setInfluencingBounds(infiniteBounds);
        lightDirectional.setEnable(true);
        lightDirectional.setCapability(Light.ALLOW_STATE_WRITE);
        lightGroup.addChild(lightDirectional);

        // Set up the point light
        Point3f lightPosition = new Point3f(-1.0f, 1.0f, 0.6f);
        lightPoint = new PointLight(white, lightPosition, attenuation);
        lightPoint.setInfluencingBounds(infiniteBounds);
        lightPoint.setEnable(false);
        lightPoint.setCapability(Light.ALLOW_STATE_WRITE);
        lightPoint.setCapability(PointLight.ALLOW_ATTENUATION_WRITE);
        lightGroup.addChild(lightPoint);

        // Set up the spot light
        // Point the light back at the origin
        lightSpot = new SpotLight(white, lightPosition, attenuation,
                lightDirection, (float) Math.toRadians(spotSpreadAngle),
                spotConcentration);
        lightSpot.setInfluencingBounds(infiniteBounds);
        lightSpot.setEnable(false);
        lightSpot.setCapability(Light.ALLOW_STATE_WRITE);
        lightSpot.setCapability(PointLight.ALLOW_ATTENUATION_WRITE);
        lightSpot.setCapability(SpotLight.ALLOW_CONCENTRATION_WRITE);
        lightSpot.setCapability(SpotLight.ALLOW_SPREAD_ANGLE_WRITE);
        lightGroup.addChild(lightSpot);
    }

    /* Setup the backgrounds.  The bg tool creates a Switch and a GUI
     * component for the backgrounds
     */
    void setupBackgrounds() {
        // initialize the background tool
        BackgroundTool bgTool = new BackgroundTool();
        bgSwitch = bgTool.getSwitch();
        bgChooser = bgTool.getChooser();
    }

    /* Setup the fog Switch and Chooser.  Child values are:
	CHILD_NONE: 	Don't use a fog
	0: 		The linear Fog node
	1: 		The exponential Fog node
    */
    void setupFogs() {
        fogSwitch = new Switch(Switch.CHILD_NONE);
        fogSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);

        // set up the linear fog
        LinearFog fogLinear = new LinearFog(skyBlue, 6.0f, 12.0f);
        fogLinear.setInfluencingBounds(infiniteBounds);
        fogSwitch.addChild(fogLinear);

        // set up the exponential fog
        ExponentialFog fogExp = new ExponentialFog(skyBlue, 0.3f);
        fogExp.setInfluencingBounds(infiniteBounds);
        fogSwitch.addChild(fogExp);

        // Create the chooser GUI
        String[] fogNames = {
            "None",
            "Linear",
            "Exponential",
        };
        int[] fogValues = {Switch.CHILD_NONE, 0, 1};

        fogChooser = new IntChooser("Fog:", fogNames,
                fogValues, 0);
        fogChooser.addIntListener(new IntListener() {
            public void intChanged(IntEvent event) {
                int value = event.getValue();
                fogSwitch.setWhichChild(value);
            }
        });
        fogChooser.setValue(Switch.CHILD_NONE);
    }


    /* Set up the sound switch.  The child values are:
	CHILD_NONE:	1No sound
	0: 		BackgroundSound
	1: 	  	PointSound
	2: 	  	ConeSound
     */
    /* sound is busted, check:
    http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4908425
    void setupSounds() {
        soundSwitch = new Switch(Switch.CHILD_NONE);
        soundSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);

        // Set up the sound media container
        java.net.URL soundURL = null;
        String soundFile = "techno_machine.au";
        try {
            soundURL = rm.getAudioURL(soundFile);
        } catch (java.net.MalformedURLException ex) {
            System.out.println(ex.getMessage());
            System.exit(1);
        }

        //System.out.println("soundURL = " + soundURL);
        MediaContainer soundMC = new MediaContainer(soundURL);

        // set up the Background Sound
        soundBackground = new BackgroundSound();
        soundBackground.setCapability(Sound.ALLOW_ENABLE_WRITE);
        soundBackground.setSoundData(soundMC);
        soundBackground.setSchedulingBounds(infiniteBounds);
        soundBackground.setEnable(false);
        soundBackground.setLoop(Sound.INFINITE_LOOPS);
        soundSwitch.addChild(soundBackground);

        // set up the point sound
        soundPoint = new PointSound();
        soundPoint.setCapability(Sound.ALLOW_ENABLE_WRITE);
        soundPoint.setSoundData(soundMC);
        soundPoint.setSchedulingBounds(infiniteBounds);
        soundPoint.setEnable(false);
        soundPoint.setLoop(Sound.INFINITE_LOOPS);
        soundPoint.setPosition(-5.0f, 5.0f, 0.0f);
        Point2f[] distGain = new Point2f[2];
        // set the attenuation to linearly decrease volume from max at
        // source to 0 at a distance of 15m
        distGain[0] = new Point2f(0.0f, 1.0f);
        distGain[1] = new Point2f(15.0f, 0.0f);
        soundPoint.setDistanceGain(distGain);
        soundSwitch.addChild(soundPoint);

        // Create the chooser GUI
        String[] soundNames = {
            "None",
            "Background",
            "Point",
        };

        soundChooser = new IntChooser("Sound:", soundNames);
        soundChooser.addIntListener(new IntListener() {
            public void intChanged(IntEvent event) {
                int value = event.getValue();
                // Should just be able to use setWhichChild on
                // soundSwitch, have to explictly enable/disable due to
                // bug.
                switch (value) {
                    case 0:
                        soundSwitch.setWhichChild(Switch.CHILD_NONE);
                        soundBackground.setEnable(false);
                        soundPoint.setEnable(false);
                        break;
                    case 1:
                        soundSwitch.setWhichChild(0);
                        soundBackground.setEnable(true);
                        soundPoint.setEnable(false);
                        break;
                    case 2:
                        soundSwitch.setWhichChild(1);
                        soundBackground.setEnable(false);
                        soundPoint.setEnable(true);
                        break;
                }
            }
        });
        soundChooser.setValue(Switch.CHILD_NONE);

    }
    */

    // sets up a grid of spheres
    void setupSpheres() {

        // create a Switch for the spheres, allow switch changes
        spheresSwitch = new Switch(Switch.CHILD_ALL);
        spheresSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);

        // Set up an appearance to make the Sphere with objColor ambient,
        // black emmissive, objColor diffuse and white specular coloring
        Material material = new Material(objColor, black, objColor, white, 32);
        Appearance appearance = new Appearance();
        appearance.setMaterial(material);

        // create a sphere and put it into a shared group
        Sphere sphere = new Sphere(0.5f, appearance);
        SharedGroup sphereSG = new SharedGroup();
        sphereSG.addChild(sphere);

        // create a grid of spheres in the z=0 plane
        // each has a TransformGroup to position the sphere which contains
        // a link to the shared group for the sphere
        for (int y = -2; y <= 2; y++) {
            for (int x = -2; x <= 2; x++) {
                TransformGroup tg = new TransformGroup();
                tmpVector.set(x * 1.2f, y * 1.2f, -0.1f);
                tmpTrans.set(tmpVector);
                tg.setTransform(tmpTrans);
                tg.addChild(new Link(sphereSG));
                spheresSwitch.addChild(tg);
            }
        }
    }

    // sets up a grid of squares
    void setupGrid() {

        // create a Switch for the spheres, allow switch changes
        gridSwitch = new Switch(Switch.CHILD_NONE);
        gridSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);

        // Set up an appearance to make the square3s with red ambient,
        // black emmissive, red diffuse and black specular coloring
        Material material = new Material(red, black, red, black, 64);
        Appearance appearance = new Appearance();
        appearance.setMaterial(material);

        // create a grid of quads
        int gridSize = 20;  // grid is gridSize quads along each side
        int numQuads = gridSize * gridSize;
        int numVerts = numQuads * 4; // 4 verts per quad
        // there will be 3 floats per coord and 4 coords per quad
        float[] coords = new float[3 * numVerts];
        // All the quads will use the same normal at each vertex, so
        // allocate an array to hold references to the same normal
        Vector3f[] normals = new Vector3f[numVerts];
        Vector3f vertNormal = new Vector3f(0.0f, 0.0f, 1.0f);
        float edgeLength = 5.0f; // length of each edge of the grid
        float gridGap = 0.03f;    // the gap between each quad
        // length of each quad is (total length - sum of the gaps) / gridSize
        float quadLength = (edgeLength - gridGap * (gridSize - 1)) / gridSize;

        // create a grid of quads in the z=0 plane
        // each has a TransformGroup to position the sphere which contains
        // a link to the shared group for the sphere
        float curX, curY;
        for (int y = 0; y < gridSize; y++) {
            curY = y * (quadLength + gridGap); // offset to lower left corner
            curY -= edgeLength / 2;	       // center on 0,0
            for (int x = 0; x < gridSize; x++) {
                // this is the offset into the vertex array for the first
                // vertex of the quad
                int vertexOffset = (y * gridSize + x) * 4;
                // this is the offset into the coord array for the first
                // vertex of the quad, where there are 3 floats per vertex
                int coordOffset = vertexOffset * 3;
                curX = x * (quadLength + gridGap); // offset to ll corner
                curX -= edgeLength / 2;	       // center on 0,0
                // lower left corner
                coords[coordOffset + 0] = curX;
                coords[coordOffset + 1] = curY;
                coords[coordOffset + 2] = 0.0f; // z
                // lower right corner
                coords[coordOffset + 3] = curX + quadLength;
                coords[coordOffset + 4] = curY;
                coords[coordOffset + 5] = 0.0f; // z
                // upper right corner
                coords[coordOffset + 6] = curX + quadLength;
                coords[coordOffset + 7] = curY + quadLength;
                coords[coordOffset + 8] = 0.0f; // z
                // upper left corner
                coords[coordOffset + 9] = curX;
                coords[coordOffset + 10] = curY + quadLength;
                coords[coordOffset + 11] = 0.0f; // z
                for (int i = 0; i < 4; i++) {
                    normals[vertexOffset + i] = vertNormal;
                }
            }
        }
        // now that we have the data, create the QuadArray
        QuadArray quads = new QuadArray(numVerts, QuadArray.COORDINATES | QuadArray.NORMALS);
        quads.setCoordinates(0, coords);
        quads.setNormals(0, normals);

        // create the shape
        Shape3D shape = new Shape3D(quads, appearance);

        // add it to the switch
        gridSwitch.addChild(shape);
    }


    BranchGroup createSceneGraph() {
        // Create the root of the branch graph
        BranchGroup objRoot = new BranchGroup();

        // Add the primitives to the scene
        setupSpheres();
        objRoot.addChild(spheresSwitch);
        setupGrid();
        objRoot.addChild(gridSwitch);
        objRoot.addChild(lightGroup);
        objRoot.addChild(bgSwitch);
        objRoot.addChild(fogSwitch);
        //objRoot.addChild(soundSwitch);

        KeyPrintBehavior key = new KeyPrintBehavior();
        key.setSchedulingBounds(infiniteBounds);
        objRoot.addChild(key);
        return objRoot;
    }


    public EnvironmentExplorer(boolean isApplication, boolean blackAndWhite) {
        if (blackAndWhite) {
            colorMode = USE_BLACK_AND_WHITE;
        }
        this.isApplication = isApplication;
    }

    public EnvironmentExplorer(boolean isApplication) {
        this(isApplication, false);
    }

    public EnvironmentExplorer() {
        this(false, false);
    }

    public void init() {
        // initialize the code base


        if (colorMode == USE_COLOR) {
            objColor = red;
        } else {
            objColor = white;
        }

        Container contentPane = getContentPane();

        contentPane.setLayout(new BorderLayout());

        GraphicsConfiguration config =
                SimpleUniverse.getPreferredConfiguration();


        canvas = new Canvas3D(config);

        u = new SimpleUniverse(canvas);

        if (isApplication) {
            offScreenCanvas = new OffScreenCanvas3D(config, true);
            // set the size of the off-screen canvas based on a scale
            // of the on-screen size
            Screen3D sOn = canvas.getScreen3D();
            Screen3D sOff = offScreenCanvas.getScreen3D();
            Dimension dim = sOn.getSize();
            dim.width *= OFF_SCREEN_SCALE;
            dim.height *= OFF_SCREEN_SCALE;
            sOff.setSize(dim);
            sOff.setPhysicalScreenWidth(sOn.getPhysicalScreenWidth() *
                    OFF_SCREEN_SCALE);
            sOff.setPhysicalScreenHeight(sOn.getPhysicalScreenHeight() *
                    OFF_SCREEN_SCALE);

// attach the offscreen canvas to the view
            u.getViewer().getView().addCanvas3D(offScreenCanvas);

        }
        contentPane.add("Center", canvas);

        // setup the env nodes and their GUI elements
        setupLights();
        setupBackgrounds();
        setupFogs();
        // sound is busted - DL

        // //setupSounds();

        // Create a simple scene and attach it to the virtual universe
        BranchGroup scene = createSceneGraph();

        // set up sound
        u.getViewer().createAudioDevice();

        // get the view
        view = u.getViewer().getView();

        // Get the viewing platform
        ViewingPlatform viewingPlatform = u.getViewingPlatform();

        // Move the viewing platform back to enclose the -4 -> 4 range
        double viewRadius = 4.0; 	// want to be able to see circle
        // of viewRadius size around origin
        // get the field of view
        double fov = u.getViewer().getView().getFieldOfView();

        // calc view distance to make circle view in fov
        float viewDistance = (float) (viewRadius / Math.tan(fov / 2.0));
        tmpVector.set(0.0f, 0.0f, viewDistance);// setup offset
        tmpTrans.set(tmpVector);	  	// set trans to translate
        // move the view platform
        viewingPlatform.getViewPlatformTransform().setTransform(tmpTrans);

        // add an orbit behavior to move the viewing platform
        OrbitBehavior orbit = new OrbitBehavior(canvas,
                OrbitBehavior.STOP_ZOOM);
        orbit.setSchedulingBounds(infiniteBounds);
        viewingPlatform.setViewPlatformBehavior(orbit);

        u.addBranchGraph(scene);

        contentPane.add("East", guiPanel());
    }

    // create a panel with a tabbed pane holding each of the edit panels
    JPanel guiPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        tabbedPane = new JTabbedPane();
        tabbedPane.addTab("Light", lightPanel());
        tabbedPane.addTab("Background", backgroundPanel());
        tabbedPane.addTab("Fog", fogPanel());
        //tabbedPane.addTab("Sound", soundPanel());
        panel.add("Center", tabbedPane);

        panel.add("South", configPanel());
        return panel;
    }


    Box lightPanel() {
        Box panel = new Box(BoxLayout.Y_AXIS);

        // add the ambient light checkbox to the panel
        JCheckBox ambientCheckBox = new JCheckBox("Ambient Light");
        ambientCheckBox.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                JCheckBox checkbox = (JCheckBox) e.getSource();
                lightAmbient.setEnable(checkbox.isSelected());
            }
        });
        ambientCheckBox.setSelected(true);
        panel.add(new LeftAlignComponent(ambientCheckBox));

        String[] lightTypeValues = {
            "None",
            "Directional",
            "Positional",
            "Spot"
        };
        IntChooser lightTypeChooser = new IntChooser("Light Type:",
                lightTypeValues);
        lightTypeChooser.addIntListener(new IntListener() {
            public void intChanged(IntEvent event) {
                int value = event.getValue();
                switch (value) {
                    case 0:
                        lightDirectional.setEnable(false);
                        lightPoint.setEnable(false);
                        lightSpot.setEnable(false);
                        break;
                    case 1:
                        lightDirectional.setEnable(true);
                        lightPoint.setEnable(false);
                        lightSpot.setEnable(false);
                        break;
                    case 2:
                        lightDirectional.setEnable(false);
                        lightPoint.setEnable(true);
                        lightSpot.setEnable(false);
                        break;
                    case 3:
                        lightDirectional.setEnable(false);
                        lightPoint.setEnable(false);
                        lightSpot.setEnable(true);
                        break;
                }
            }
        });
        lightTypeChooser.setValueByName("Directional");
        panel.add(lightTypeChooser);


        // Set up the sliders for the attenuation

        // top row
        panel.add(new LeftAlignComponent(new JLabel("Light attenuation:")));

        FloatLabelJSlider constantSlider = new FloatLabelJSlider("Constant ", 0.1f, 0.0f, 3.0f, attenuation.x);
        constantSlider.setMajorTickSpacing(1.0f);
        constantSlider.setPaintTicks(true);
        constantSlider.addFloatListener(new FloatListener() {
            public void floatChanged(FloatEvent e) {
                attenuation.x = e.getValue();
                lightPoint.setAttenuation(attenuation);
                lightSpot.setAttenuation(attenuation);
            }
        });
        panel.add(constantSlider);

        FloatLabelJSlider linearSlider = new FloatLabelJSlider("Linear   ", 0.1f, 0.0f, 3.0f, attenuation.y);
        linearSlider.setMajorTickSpacing(1.0f);
        linearSlider.setPaintTicks(true);
        linearSlider.addFloatListener(new FloatListener() {
            public void floatChanged(FloatEvent e) {
                attenuation.y = e.getValue();
                lightPoint.setAttenuation(attenuation);
                lightSpot.setAttenuation(attenuation);
            }
        });
        panel.add(linearSlider);

        FloatLabelJSlider quadradicSlider = new FloatLabelJSlider("Quadradic", 0.1f, 0.0f, 3.0f, attenuation.z);
        quadradicSlider.setMajorTickSpacing(1.0f);
        quadradicSlider.setPaintTicks(true);
        quadradicSlider.addFloatListener(new FloatListener() {
            public void floatChanged(FloatEvent e) {
                attenuation.z = e.getValue();
                lightPoint.setAttenuation(attenuation);
                lightSpot.setAttenuation(attenuation);
            }
        });
        panel.add(quadradicSlider);


        // Set up the sliders for the attenuation
        // top row
        panel.add(new LeftAlignComponent(new JLabel("Spot light:")));

        // spread angle is 0-180 degrees, no slider scaling
        FloatLabelJSlider spotSpreadSlider = new FloatLabelJSlider("Spread Angle ", 1.0f, 0.0f, 180.0f, spotSpreadAngle);
        spotSpreadSlider.addFloatListener(new FloatListener() {
            public void floatChanged(FloatEvent e) {
                spotSpreadAngle = e.getValue();
                lightSpot.setSpreadAngle((float) Math.toRadians(spotSpreadAngle));
            }
        });
        panel.add(spotSpreadSlider);

        // concentration angle is 0-128 degrees
        FloatLabelJSlider spotConcentrationSlider = new FloatLabelJSlider("Concentration", 1.0f, 0.0f, 128.0f, spotConcentration);
        spotConcentrationSlider.addFloatListener(new FloatListener() {
            public void floatChanged(FloatEvent e) {
                spotConcentration = e.getValue();
                lightSpot.setConcentration(spotConcentration);
            }
        });
        panel.add(spotConcentrationSlider);

        return panel;
    }

    JPanel backgroundPanel() {
        JPanel panel = new JPanel();
        panel.add(bgChooser);
        return panel;
    }

    JPanel fogPanel() {
        JPanel panel = new JPanel();
        panel.add(fogChooser);
        return panel;
    }


    //JPanel soundPanel() {
    //    JPanel panel = new JPanel();
    //    panel.add(soundChooser);
    //    return panel;
    // }

    JPanel configPanel() {
        JPanel panel = new JPanel();
        panel.setLayout(new GridLayout(1, 0));

        String[] dataTypeValues = {
            "Spheres",
            "Grid",
        };
        IntChooser dataTypeChooser = new IntChooser("Data:",
                dataTypeValues);
        dataTypeChooser.addIntListener(new IntListener() {
            public void intChanged(IntEvent event) {
                int value = event.getValue();
                switch (value) {
                    case 0:
                        spheresSwitch.setWhichChild(Switch.CHILD_ALL);
                        gridSwitch.setWhichChild(Switch.CHILD_NONE);
                        break;
                    case 1:
                        gridSwitch.setWhichChild(Switch.CHILD_ALL);
                        spheresSwitch.setWhichChild(Switch.CHILD_NONE);
                        break;
                }
            }
        });
        panel.add(dataTypeChooser);

        if (isApplication) {
            JButton snapButton = new JButton("Snap Image");
            snapButton.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    Point loc = canvas.getLocationOnScreen();
                    offScreenCanvas.setOffScreenLocation(loc);
                    Dimension dim = canvas.getSize();
                    dim.width *= OFF_SCREEN_SCALE;
                    dim.height *= OFF_SCREEN_SCALE;
                    nf.setMinimumIntegerDigits(3);
                    offScreenCanvas.snapImageFile(outFileBase + nf.format(outFileSeq++),
                            dim.width, dim.height);
                    nf.setMinimumIntegerDigits(0);
                }
            });
            panel.add(snapButton);
        }

        return panel;
    }

    public void destroy() {
        u.removeAllLocales();
    }

    // The following allows EnvironmentExplorer to be run as an application
    // as well as an applet
    //
    public static void main(String[] args) {
        boolean useBlackAndWhite = false;
        for (int i = 0; i < args.length; i++) {
            //System.out.println("args[" + i + "] = " + args[i]);
            if (args[i].equals("-b")) {
                System.out.println("Use Black And White");
                useBlackAndWhite = true;
            }
        }
        new MainFrame(new EnvironmentExplorer(true, useBlackAndWhite),
                950, 600);
    }
}
