package j3d.examples.color;

// WrapCheckers3D.java
// Andrew Davison, January 2003, dandrew@ratree.psu.ac.th

/* Illustrates how to create a simple world. The checkboard
   floor is created by CheckerFloor. The background, lighting,
   key controls, and initial user positioning is done here.

   Most of this will stay the same from one example to another.

   floatingSphere() shows how to create a coloured, shiny object
   which is affected by the lighting in the world.

   The scene graph display code is in WrapCheckers3D() and
   createSceneGraph().
*/


import com.sun.j3d.utils.behaviors.vp.OrbitBehavior;
import com.sun.j3d.utils.geometry.Sphere;
import com.sun.j3d.utils.universe.SimpleUniverse;
import com.sun.j3d.utils.universe.ViewingPlatform;
import j2d.ImageUtils;
import j3d.examples.checkers.CheckerFloor;

import javax.media.j3d.*;
import javax.swing.*;
import javax.vecmath.Color3f;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;
import java.awt.*;
import java.util.HashMap;
import java.util.Iterator;

// import com.tornadolabs.j3dtree.*;    // for displaying the scene graph


public class CheckerPanel extends JPanel {
    private static final int PWIDTH = 512;   // size of panel
    private static final int PHEIGHT = 512;

    private static final int BOUNDSIZE = 100;  // larger than world

    private static final Point3d USERPOSN = new Point3d(0, 5, 20);
    // initial user position

    private SimpleUniverse su;
    private BranchGroup sceneBG;
    private BoundingSphere bounds;
    private Image image = ImageUtils.getImage();
    private BranchGroup objSplice;
    private BranchGroup objBranch;

    public CheckerPanel() {
        initObjectSplice();
        setLayout(new BorderLayout());
        setOpaque(false);
        setPreferredSize(new Dimension(PWIDTH, PHEIGHT));

        GraphicsConfiguration config =
                SimpleUniverse.getPreferredConfiguration();
        Canvas3D canvas3D = new Canvas3D(config);
        add(BorderLayout.CENTER, canvas3D);
        canvas3D.setFocusable(true);
        canvas3D.requestFocus();

        su = new SimpleUniverse(canvas3D);


        createSceneGraph();
        initUserPosition();
        // set user's viewpoint
        orbitControls(canvas3D);
        // controls for moving the viewpoint

        su.addBranchGraph(sceneBG);

        // j3dTree.updateNodes( su );    // build the tree display window

    } // end of WrapCheckers3D()

    public Sphere getSphere(Image image) {
        Appearance appearance = new Appearance();
        Color3f color = new Color3f(.1f, .2f, .3f);
        Material material = new Material();
        material.setDiffuseColor(color);
        appearance.setMaterial(material);
        return new Sphere(1.0f, appearance);
    }

    private void createSceneGraph() {
        sceneBG = new BranchGroup();
        bounds = new BoundingSphere(new Point3d(0, 0, 0), BOUNDSIZE);

        lightScene();         // add the lights
        addBackground();      // add the sky
        sceneBG.addChild(new CheckerFloor().getBG());  // add the floor

        floatingSphere();     // add the floating sphere

        sceneBG.compile();   // fix the scene
    } // end of createSceneGraph()


    private void lightScene()
            /* One ambient light, 2 directional lights */ {
        Color3f white = new Color3f(1.0f, 1.0f, 1.0f);

        // Set up the ambient light
        AmbientLight ambientLightNode = new AmbientLight(white);
        ambientLightNode.setInfluencingBounds(bounds);
        sceneBG.addChild(ambientLightNode);

        // Set up the directional lights
        Vector3f light1Direction = new Vector3f(-1.0f, -1.0f, -1.0f);
        // left, down, backwards
        Vector3f light2Direction = new Vector3f(1.0f, -1.0f, 1.0f);
        // right, down, forwards

        DirectionalLight light1 =
                new DirectionalLight(white, light1Direction);
        light1.setInfluencingBounds(bounds);
        sceneBG.addChild(light1);

        DirectionalLight light2 =
                new DirectionalLight(white, light2Direction);
        light2.setInfluencingBounds(bounds);
        sceneBG.addChild(light2);
    }  // end of lightScene()


    private void addBackground() {
        Background back = new Background();
        back.setApplicationBounds(bounds);
        back.setColor(0.17f, 0.65f, 0.92f);    // sky colour
        sceneBG.addChild(back);
    }  // end of addBackground()


    private void orbitControls(Canvas3D c) {
        OrbitBehavior orbit =
                new OrbitBehavior(c, OrbitBehavior.REVERSE_ALL);
        orbit.setSchedulingBounds(bounds);

        ViewingPlatform vp = su.getViewingPlatform();
        vp.setViewPlatformBehavior(orbit);
    }


    private void initUserPosition()
            // Set the user's initial viewpoint using lookAt()
    {
        ViewingPlatform vp = su.getViewingPlatform();
        TransformGroup steerTG = vp.getViewPlatformTransform();

        Transform3D t3d = new Transform3D();
        steerTG.getTransform(t3d);

        // args are: viewer posn, where looking, up direction
        t3d.lookAt(USERPOSN, new Point3d(0, 0, 0), new Vector3d(0, 1, 0));
        t3d.invert();

        steerTG.setTransform(t3d);
    }  // end of initUserPosition()


    // ---------------------- floating sphere -----------------


    private void floatingSphere() {
        getAppearance();
        float red, green, blue;
        float x, y, z;


        ColorMapGenerator cmg = new ColorMapGenerator(image);
        cmg.processImage();
        HashMap colorMap = cmg.getColorMap();
        Iterator colorMapIndex = colorMap.keySet().iterator();
        int colorCount = 0;

        do {
            Object colorKey = colorMapIndex.next();    // Returns color as Object
            Color color = (Color) colorKey;


            // Calculate color of sphere
            red = color.getRed() / 255f;
            green = color.getGreen() / 255f;
            blue = color.getBlue() / 255f;
            Color3f color3f = new Color3f(red, green, blue);
            Appearance appearance = new Appearance();
            Material material = new Material();
            material.setDiffuseColor(color3f);
            material.setSpecularColor(color3f);
            appearance.setMaterial(material);

            // Calculate location of sphere based on color
            x = (red - 0.5f) * 20f;
            y = (green - 0.5f) * 20f;
            z = (blue - 0.5f) * 20f;

            // Calculate size of sphere based on pixelCount
            float radius = 0.5f;//(float)Math.sqrt(pixelCount);

            // Tally total number of colors
            colorCount++;
            System.out.println("Total number of colors = " + colorCount);

            Transform3D t3d = new Transform3D();
            t3d.set(new Vector3f(x, y, z));
            TransformGroup tg = new TransformGroup(t3d);
            tg.addChild(new Sphere(radius, appearance));
            sceneBG.addChild(tg);
        } while (colorMapIndex.hasNext());
    }  // end of floatingSphere()

    private Appearance getAppearance() {
        Material blueMat = getMaterial();

        // sets ambient, emissive, diffuse, specular, shininess
        blueMat.setLightingEnable(true);

        Appearance blueApp = new Appearance();
        blueApp.setMaterial(blueMat);
        return blueApp;
    }

    private Material getMaterial() {
        // Create the blue appearance node
        Color3f black = new Color3f(0.0f, 0.0f, 0.0f);
        Color3f blue = new Color3f(0.3f, 0.3f, 0.8f);
        Color3f specular = new Color3f(0.9f, 0.9f, 0.9f);

        Material blueMat = new Material(blue, black, blue, specular, 25.0f);
        return blueMat;
    }

    public void openImage() {
        image = ImageUtils.getImage();
        createSceneGraph();

    }

    private void initObjectSplice() {
        objSplice = new BranchGroup();
        objSplice.setCapability(BranchGroup.ALLOW_CHILDREN_EXTEND);
        objSplice.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);
    }

    public void updateShape() {

        BranchGroup newBG = getColorSphereBranchGroup();
        newBG.setCapability(BranchGroup.ALLOW_DETACH);
        replaceBranch(newBG);
    }

    private BranchGroup getColorSphereBranchGroup() {
        floatingSphere();
        return sceneBG;
    }

    private void replaceBranch(BranchGroup newBranch) {
        objBranch.detach();
        // remove old one
        objBranch = newBranch;

        objSplice.addChild(objBranch);
        // splice in new one
    }
} // end of WrapCheckers3D class
