package j3d.examples.appearance.texture.noise;

import javax.media.j3d.*;

/*
 * Water is a Surface subclass that demonstrates how
 * to change Java3D geometry at runtime.  This 
 * implementation uses Perlin noise to create a geometry 
 * that appears to be water with gentle waves.
 */

public class Water extends Surface implements GeometryUpdater {
    protected float waveSize;
    private float transparency = 0.0f;
    private float seaLevel = 0.0f;
    private float red, green, blue;

    public Water(float sideLength, int divisions, float waveSize) {
        this(sideLength,
                divisions,
                waveSize,
                0f,
                0f,
                0.11f,
                0.20f,
                0.29f);
    }

    public Water(float sideLength,
                 int divisions,
                 float waveSize,
                 float transparency,
                 float seaLevel,
                 float r,
                 float g,
                 float b) {
        this.waveSize = waveSize;
        this.transparency = transparency;
        this.seaLevel = seaLevel;
        this.divisions = divisions;
        red = r;
        green = g;
        blue = b;
        metersPerDivision = sideLength / divisions;
        setGeometry(super.createGeometry());
        setAppearance(createAppearance());
        setCapabilities();
    }

    public void nextFrame(int ticks) {
        /* Called during animation.  Because
         * this object is a GeometryUpdater, get
         * the geometry and pass this object to
         * the updateData method on the geometry.
         */
        this.ticks = ticks;
        IndexedTriangleStripArray tsa =
                (IndexedTriangleStripArray) getGeometry();
        tsa.updateData(this);
    }

    public void updateData(Geometry geometry) {
        // Go through the entire surface and recalculate
        // the y component of each vertex.
        IndexedTriangleStripArray tsa = (IndexedTriangleStripArray) geometry;
        float[] coordinates = tsa.getCoordRefFloat();
        for (int row = 0; row < (divisions + 1); row++) {
            int width = row * (divisions + 1);
            for (int col = 0; col < (divisions + 1); col++) {
                int ci = (col + width) * 3;
                float x = coordinates[ci + 0];
                float y = ticks;
                float z = coordinates[ci + 2];
                float nextValue = calculateHeight(x, y, z);
                // Change just the y value (the height)
                coordinates[ci + 1] = nextValue;
            }
        }
        generateNormals(tsa, coordinates);
    }


    protected float calculateHeight(double x, double y, double z) {
        /* Use fractional Brownian Motion approach to generate
         * the height.  This is essentially a degenerate fractal
         * since we are only using 2 octaves.  Three is considered
         * the minimum # of octaves for something to be called fractal.
         */

        // Transform from world space to noise space.
        double s = x / NOISE_ZOOM;
        double t = y / NOISE_ZOOM;
        double r = z / NOISE_ZOOM;
        float answer = seaLevel +
                (float) (waveSize * ImprovedNoise.fBm(s, t, r, 3, 2));
        return answer;
    }

    protected Appearance createAppearance() {
        Appearance appearance = new Appearance();
        PolygonAttributes polyAttrib = new PolygonAttributes();
        polyAttrib.setCullFace(PolygonAttributes.CULL_NONE);
        polyAttrib.setPolygonMode(PolygonAttributes.POLYGON_FILL);
        // Setting back face normal flip to true allows backfacing
        // polygons to be lit (normal is facing wrong way) but only
        // if the normal is flipped.
        polyAttrib.setBackFaceNormalFlip(true);
        appearance.setPolygonAttributes(polyAttrib);
        Material material = new Material();
        material.setAmbientColor(red, green, blue);
        material.setSpecularColor(0.5f, 0.5f, 0.5f);
        material.setDiffuseColor(0.3f, 0.3f, 0.3f);
        material.setShininess(180);
        appearance.setMaterial(material);

        TransparencyAttributes ta = new TransparencyAttributes();
        ta.setTransparency(transparency);
        ta.setTransparencyMode(TransparencyAttributes.NICEST);
        appearance.setTransparencyAttributes(ta);

        return appearance;
    }

    protected void setCapabilities() {
        super.setCapabilities();
        Geometry geometry = getGeometry();
        geometry.setCapability(GeometryArray.ALLOW_REF_DATA_READ);
        geometry.setCapability(GeometryArray.ALLOW_NORMAL_WRITE);
        geometry.setCapability(GeometryArray.ALLOW_REF_DATA_WRITE);
    }
}
