package j3d.examples.appearance.texture.noise;

import com.sun.j3d.utils.applet.MainFrame;
import com.sun.j3d.utils.behaviors.keyboard.KeyNavigatorBehavior;
import com.sun.j3d.utils.universe.SimpleUniverse;

import javax.media.j3d.*;
import javax.vecmath.Color3f;
import javax.vecmath.Vector3f;
import java.applet.Applet;
import java.awt.*;

/**
 * This is a more advanced example of noise and J3D features:
 * 1. Perlin Noise
 * 2. Multifractal terrain
 * 3. Transparency
 * 4. Animation through the use of a Behavior
 * <p/>
 * Uses Perlin Noise for a number of things:
 * 1. Terrain is generated with noise allowing the number
 * of divisions to be any number (instead of being limited
 * to a power of 2 minus 1).  This is encapsulated in the
 * RidgedFractalTerrain class.
 * 2. The color index selection perturbs the index so that
 * the layering effect is minimized.
 * 3. Water is generated
 */
public class TerrainAndWaterMain extends Applet implements IElapsedTimeListener {
    private static final float SIDE_LENGTH = 700.0f; // Meters
    private static final int DIVISIONS = 350;

    private RidgedFractalTerrain terrain;
    private Water water;
    private int ticks = 0;
    private boolean animateWater = false;

    public TerrainAndWaterMain(boolean animate) {
        animateWater = animate;
        initialize();
    }

    private void initialize() {
        setLayout(new BorderLayout());
        GraphicsConfiguration config =
                SimpleUniverse.getPreferredConfiguration();
        Canvas3D canvas3D = new Canvas3D(config);
        add("Center", canvas3D);
        SimpleUniverse simpleU = new SimpleUniverse(canvas3D);

        //Position the view
        TransformGroup viewingPlatformGroup =
                simpleU.getViewingPlatform().getViewPlatformTransform();
        Transform3D t3d = new Transform3D();
        t3d.rotY(-Math.PI / 4);
        t3d.setTranslation(new Vector3f(0, 100, 0));
        viewingPlatformGroup.setTransform(t3d);

        BranchGroup scene = createSceneGraph(simpleU);
        simpleU.addBranchGraph(scene);

        // Adjust the clipping planes
        canvas3D.getView().setBackClipDistance(3000.0d);
        canvas3D.getView().setFrontClipDistance(1d);
    }

    public void tick() {
        // Called when the timer wakes us up.
        water.nextFrame(++ticks);
    }

    public BranchGroup createSceneGraph(SimpleUniverse su) {
        BranchGroup objRoot = new BranchGroup();

        terrain = new RidgedFractalTerrain(SIDE_LENGTH, DIVISIONS);
        objRoot.addChild(terrain);

        // Add ambient light
        BoundingSphere bounds = new BoundingSphere();
        bounds.setRadius(10000);
        AmbientLight ambient = new AmbientLight();
        ambient.setColor(new Color3f(1f, 1f, 1f));
        ambient.setInfluencingBounds(bounds);
        objRoot.addChild(ambient);

        // Add a directional light
        DirectionalLight directional = new DirectionalLight();
        directional.setDirection(0f, -1f, 1f);
        directional.setColor(new Color3f(0.5f, 0.5f, 0.5f));
        directional.setInfluencingBounds(bounds);
        objRoot.addChild(directional);
        // Add a directional light
        DirectionalLight directional2 = new DirectionalLight();
        directional2.setDirection(0.4f, -0.6f, -1f);
        directional2.setColor(new Color3f(1f, 1f, 1f));
        directional2.setInfluencingBounds(bounds);
        objRoot.addChild(directional2);

        // Add a behavior to notify us when a certain
        // amount of time has passed.  This is used
        // to animate the water.
        if (animateWater) {
            water = new Water(SIDE_LENGTH, DIVISIONS / 3, 1.5f, 0.15f, 30f, 0.31f, 0.40f, 0.33f);
            ElapsedTimeBehavior timer = new ElapsedTimeBehavior(this, 100);
            timer.setSchedulingBounds(bounds);
            objRoot.addChild(timer);
        } else {
            water = new Water(SIDE_LENGTH, 3 * DIVISIONS / 2, 0.5f, 0.15f, 30f, 0.31f, 0.40f, 0.33f);
        }
        objRoot.addChild(water);

        Background background = new Background();
        background.setApplicationBounds(bounds);
        background.setColor(0.53f, 0.81f, 0.92f);
        objRoot.addChild(background);

        // Add a keyboard navigator to move around
        TransformGroup vpTrans =
                su.getViewingPlatform().getViewPlatformTransform();
        KeyNavigatorBehavior keyNavBeh =
                new KeyNavigatorBehavior(vpTrans);
        keyNavBeh.setSchedulingBounds(bounds);
        objRoot.addChild(keyNavBeh);

        // Optimize the scene graph
        objRoot.compile();
        return objRoot;
    }

    public static void main(String[] args) {
        // Depending on the options used,
        // you may need to use the -Xms256M -Xmx1000M
        // options when running this example.

        // The scene is reproducable with the same
        // seed so if you find one you like, keep
        // the seed.
        long seed = System.currentTimeMillis();
        //long seed = 1098059964617L;
        ImprovedNoise.reinitialize(seed);
        System.out.println("Seed: " + seed);
        Frame frame =
                new MainFrame(new TerrainAndWaterMain(false), 800, 400);
    }

}
