package j2d.gui;

import futils.StreamSniffer;
import gui.ClosableJFrame;
import ip.color.FloatImageBean;
import j2d.*;
import j2d.file.ExtensionFileFilter;
import j2d.file.PPMToolkit;
import j2d.hpp.HppFilterInterface;
import j2d.imageproc.ExponentialStretchProcessor;
import j2d.imageproc.FalseColorProcessor;
import j2d.imageproc.HistogramEQProcessor;
import j2d.imageproc.LinearMappingProcessor;
import utils.OsUtils;

import javax.media.CannotRealizeException;
import javax.media.NoPlayerException;
import javax.media.Player;
import javax.media.format.VideoFormat;
import javax.media.protocol.PushBufferDataSource;
import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.image.BufferedImage;
import java.awt.image.PixelGrabber;
import java.io.*;
import java.util.Observable;
import java.util.Observer;


//todo: add j2d.border.BorderPanel
//todo: add color quantization
//todo: add affine transforms.

public class Main extends ClosableJFrame
        implements ImageProcessListener, ImageBeanInterface {

    private MDIDesktopPane deskTop = new MDIDesktopPane();

    private FalseColorToolbox falseColorToolbox;
    private LinearMappingFrame linearMappingFrame;
    private HistogramEQToolbox histEqToolbox;
    ExponentialStretchToolbox estStretchControls;
    private VideoCaptureChildFrame vcf = null;


    // Image processors

    private FalseColorProcessor fcProcessor
            = new FalseColorProcessor();
    private LinearMappingProcessor lmProcessor
            = new LinearMappingProcessor();
    private HistogramEQProcessor heProcessor
            = new HistogramEQProcessor();
    private ExponentialStretchProcessor esProcessor
            = new ExponentialStretchProcessor();
    private Player player;

    public void closePlayer() {
        if (player != null)
            CameraUtils.close(player);
    }

    /**
     * Construct a new ImageTool
     */
    public Main() {
        Container c;

        c = getContentPane();
        c.setLayout(new BorderLayout());


        c.add(new JScrollPane(deskTop));

        setTitle("Image Tool");
        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

        // Set up the menus

        JMenuBar mb = new JMenuBar();
        setJMenuBar(mb);
        FileMenu.addFileMenu(this, mb);
        MainMenus.addProcessMenu(this, mb);
        WindowMenu windowMenu = new WindowMenu(deskTop);
        mb.add(windowMenu);
        windowMenu.setMnemonic('W');
    }

    public void addPanel(JPanel cp, String title) {
        if (OsUtils.isMacOs()) {
            addPanelMac(cp, title);
            return;
            // macs can't handle sliders on internal frames, I dunno why.
            // DocJava.
        }
        addPanelWindows(title, cp);

    }

    private void addPanelWindows(String title, JPanel cp) {
        JInternalFrame jif = new JInternalFrame(title);
        jif.setClosable(true);
        jif.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
        Container c = jif.getContentPane();
        //c.setLayout(new FlowLayout());
        c.add(cp);
        jif.setSize(200, 200);

        deskTop.add(jif);
        jif.setVisible(true);
        jif.setResizable(true);
        jif.setRequestFocusEnabled(true);
        jif.pack();
    }

    public void addPanelMac(JPanel cp, String title) {
        ClosableJFrame jif = new ClosableJFrame(title);
        //jif.setClosable(true);
        //jif.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
        Container c = jif.getContentPane();
        c.setLayout(new FlowLayout());
        c.add(cp);
        jif.setSize(200, 200);

        // deskTop.add(jif);
        jif.setVisible(true);
        jif.setResizable(true);
        //jif.setRequestFocusEnabled(true);
        jif.repaint();

    }

    public void process(FloatImageProcessorInterface fip) {
        FloatImageBean fib = getFloatImage();
        fib = fip.process(fib);
        setImage(fib.getImage());
        setFloatImage(fib);
    }

    public FloatImageBean getFloatImage() {
        InternalImageFrame icf = getChildFrame();
        return icf.getFib();

    }

    public void process(ImageProcessorInterface ip) {
        setImage(ip.process(getImage()));
    }

    public void process(HppFilterInterface hppfilter) {
        Image img = getImage();
        ShortImageBean sib = new ShortImageBean(img);

        sib.process(hppfilter);
        setImage(sib.getImage());
    }

    /**
     * Get the base image before it is processed.
     *
     * @return unprocessed base image.
     */
    public Image getImage() {
        InternalImageFrame icf = getChildFrame();
        return icf.getBaseImage();
    }

    /**
     * This is the result of an image processing operation
     *
     * @return a processed image
     */
    public Image getProcessedImage() {
        InternalImageFrame icf = getChildFrame();
        return icf.getDisplayedImage();
    }

    public void update(ImageProcessorInterface ip) {
        if (ip == null) {
            revertImage();
            return;
        }
        setImage(ip.process(getImage()));
    }

    public void setImage(Image img) {
        InternalImageFrame icf = getChildFrame();
        icf.setImage(img);
    }

    public void setFloatImage(FloatImageBean img) {
        InternalImageFrame icf = getChildFrame();
        icf.setFib(img);
    }

    private InternalImageFrame getChildFrame() {
        InternalImageFrame icf = (InternalImageFrame)
                deskTop.getTopmostFrame(InternalImageFrame.class);
        return icf;
    }

    public static String stripSuffix(String s) {
        int i = s.indexOf('.');
        if (i == -1) return s;
        return s.substring(0, i);
    }

    public void printHexImage() {

        String title = "foo";
        try {
            InternalImageFrame icf =
                    (InternalImageFrame) deskTop.getTopmostFrame(InternalImageFrame.class);
            if (icf == null) return;
            title = icf.getTitle();
        } catch (Exception e) {
            return;
        }
        Image img = getImage();
        if (img == null) System.out.println("img == null");
        ImageBean ib = new ImageBean();
        ib.setImage(img);
        int h = ib.getImageHeight();
        int w = ib.getImageWidth();

        String varName = stripSuffix(title);
        StringBuffer sb = new StringBuffer("   public static Icon get" +
                varName +
                "Icon() {\n" +
                "        return new ImageIcon(getImage(" +
                varName +
                ", " +
                h +
                ", " +
                w +
                "), \"" +
                varName +
                "\"" +
                ");\n" +
                "    }" +
                "\n\nprivate static int " +
                varName +
                "[] = {\n");
        Image img1 = ib.getImage();

        int pixels[] = new int[w * h];
        PixelGrabber pg = new PixelGrabber(img1,
                0,
                0,
                w,
                h,
                pixels,
                0,
                w);
        try {
            pg.grabPixels();
        } catch (InterruptedException e1) {
        }
        String newline = new String("\n");
        int i = 0;
        for (int y1 = 0; y1 < h; y1++) {
            for (int x1 = 0; x1 < w; x1++) {
                i = x1 + y1 * w;

                sb.append("0x" + Integer.toHexString(pixels[i]) + ", ");
            }
            sb.append(newline);

        }
        sb.append("};");
        String hexImage = sb.toString();
        System.out.println(hexImage);


    }

    /**
     * Helper function for closing the ImageTool. Could put other
     * termination code here.
     */
    public void shutdown() {
        dispose();
    }

    public void captureVideo() {

        CameraConfigToolbox cctb = new CameraConfigToolbox();
        cctb.show();


        PushBufferDataSource pushBufferDataSource = cctb.getPushBufferDataSource();
        VideoFormat selectedFormat = cctb.getSelectedFormat();
        vcf =
                new VideoCaptureChildFrame(pushBufferDataSource,
                        selectedFormat);
        vcf.setSize(160, 140);
        vcf.show();
        deskTop.add(vcf);

    }

    public void usbSnap() {
        if (vcf != null && vcf.isVisible()) {
            vcf.setVisible(false);
            vcf.stopVideo();
        }
        ;
        try {
            // Create capture device

            if (player == null)
                player = CameraUtils.getPlayer();
            BufferedImage imageFromCamera = CameraUtils.getImageFromCamera(player);

            BufferedImage bi = imageFromCamera;
            Image imgProcessee = ImageUtils.getImage(bi);

            InternalImageFrame icfNewFrameInternal =
                    new InternalImageFrame(imgProcessee,
                            "screen shot");
            deskTop.add(icfNewFrameInternal);
        } catch (CannotRealizeException e) {
            e.printStackTrace();

        } catch (IOException e) {
            e.printStackTrace();

        } catch (NoPlayerException e) {
            e.printStackTrace();

        } catch (InterruptedException e) {
            e.printStackTrace();

        }

    }

    /**
     * This is called when the user requests a new file to be opened.
     */
    public void captureScreen() {

        try {
            BufferedImage bi = ImageUtils.captureWholeScreen();
            Image imgProcessee = ImageUtils.getImage(bi);

            InternalImageFrame icfNewFrameInternal =
                    new InternalImageFrame(imgProcessee,
                            "screen shot");
            deskTop.add(icfNewFrameInternal);
        } catch (AWTException e) {
            e.printStackTrace();

        }

    }

    /**
     * This is called when the user requests a new file to be opened.
     */
    public void openImage() {
        File imageFile =
                futils.Futil.getReadFile("select an image file");
        Image imageToBeProcessed = getImage(imageFile);
        InternalImageFrame icfNewFrameInternal =
                new InternalImageFrame(imageToBeProcessed,
                        imageFile.getName());
        deskTop.add(icfNewFrameInternal);
    }

    public Image getImage(File f) {
        if (f == null) return null;
        if (!f.exists()) return null;

        InputStream is = null;
        try {
            is = new FileInputStream(f);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }

        StreamSniffer ss = new StreamSniffer(is);
        int id = ss.classifyStream();
        switch (id) {
            case StreamSniffer.PPM:
                return getPPMImage(f);
            case StreamSniffer.PPM_RAWBITS:
                return getPPMImage(f);
            case StreamSniffer.GIF87a:
                return getGifOrJpg(f);
            case StreamSniffer.GIF89a:
                return getGifOrJpg(f);
            case StreamSniffer.JPEG:
                return getGifOrJpg(f);
            default:
                {
                    System.out.println("Can not open " +
                            ss +
                            " as image");
                    return null;
                }

        }
    }

    public void update(Object o) {
        if (o == null) revertImage();
    }

    private Image getGifOrJpg(File imageFile) {
        return Toolkit.getDefaultToolkit()
                .getImage(imageFile.getAbsolutePath());
    }


    private Image getPPMImage(File imageFile) {
        try {
            Image imgProcessee;
            imgProcessee =
                    PPMToolkit.getImage(imageFile,
                            false);
            return imgProcessee;
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * This is called when the user picks the save menu option.
     */
    public void saveImage() {
        InternalImageFrame icf;
        int intSuccess;
        File imageFile;
        try {
            icf =
                    (InternalImageFrame) deskTop.getTopmostFrame(InternalImageFrame.class);
            if (icf == null) return;
        } catch (Exception e) {
            return;
        }

        JFileChooser jfc = new JFileChooser();
        ExtensionFileFilter eff1 =
                new ExtensionFileFilter("ppm",
                        "Portable Pixelmap image file");
        jfc.addChoosableFileFilter(eff1);
        String extension = "ppm.gz";
        String description = "Zipped Pixelmap image file";
        ExtensionFileFilter eff2 =
                new ExtensionFileFilter(extension,
                        description);
        jfc.addChoosableFileFilter(eff2);
        jfc.setFileFilter(eff1);
        jfc.setAcceptAllFileFilterUsed(false);

        intSuccess = jfc.showSaveDialog(this);

        if (intSuccess ==
                JFileChooser.APPROVE_OPTION) {
            imageFile = jfc.getSelectedFile();

            ExtensionFileFilter eff = (ExtensionFileFilter) jfc.getFileFilter();

            try {
                if (eff == eff1) {
                    imageFile =
                            fixExtension(imageFile,
                                    "ppm");
                    savePpm(icf, imageFile);
                } else if (eff == eff2) {
                    imageFile =
                            fixExtension(imageFile,
                                    "ppm.gz");
                    savePpmGz(icf, imageFile);
                }
                icf.setTitle(imageFile.getName());
            } catch (Exception e) {
                JOptionPane.showMessageDialog(this, "File write error: \n"
                        + e.toString());
            }
        }
    }

    private void savePpm(InternalImageFrame icf, File imageFile)
            throws IOException {
        PPMToolkit.saveImage(icf.getDisplayedImage(),
                imageFile,
                false);
    }

    private void savePpmGz(InternalImageFrame icf, File imageFile)
            throws IOException {
        PPMToolkit.saveImage(icf.getDisplayedImage(),
                imageFile,
                true);
    }

    /**
     * Fix the file extension
     */
    private File fixExtension(File file,
                              String extension) {
        String path = file.getAbsolutePath();

        if (!path.endsWith("." + extension)) {
            path += "." + extension;
        }
        return new File(path);
    }

    /**
     * Set the image in the active frame to the base image.
     */
    public void revertImage() {
        InternalImageFrame icf;

        try {
            icf =
                    (InternalImageFrame)
                    deskTop.getTopmostFrame(InternalImageFrame.class);
            icf.revert();
        } catch (Exception e) {
            // do nothing
        }
    }

    /**
     * Helper function to show LinearMappingFrame
     */
    public void launchMappingControls() {
        linearMappingFrame =
                new LinearMappingFrame();
        linearMappingFrame.show();
        linearMappingFrame.getSliderBank()
                .addObserver(new Observer() {
                    public void update(Observable x,
                                       Object y) {
                        adjustLinearMapping(false);
                    }
                });
        linearMappingFrame.getButton()
                .addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent e) {
                        adjustLinearMapping(true);
                    }
                });
        deskTop.add(linearMappingFrame);
    }

    /**
     * Helper function to show FalseColorToolbox
     */
    public void launchColorControls() {
        falseColorToolbox =
                new FalseColorToolbox();
        falseColorToolbox.show();
        falseColorToolbox.getSliderBank()
                .addObserver(new Observer() {
                    public void update(Observable x,
                                       Object y) {
                        adjustColorization();
                    }
                });
        deskTop.add(falseColorToolbox);
    }

    /**
     * Helper function to show HistogramEQToolbox
     */
    public void launchHistogramControls() {
        histEqToolbox = new HistogramEQToolbox();
        histEqToolbox.show();
        histEqToolbox.getButton()
                .addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent e) {
                        adjustHistogram();
                    }
                });
        histEqToolbox.getSlider().addObserver(new Observer() {
            public void update(Observable x,
                               Object y) {
                adjustHistogram();
            }
        });

        deskTop.add(histEqToolbox);
    }

    /**
     * Helper function to show ExponentialStretchToolbox
     */
    public void launchStretchControls() {
        estStretchControls =
                new ExponentialStretchToolbox();
        estStretchControls.show();
        estStretchControls.getSlider()
                .addObserver(new Observer() {
                    public void update(Observable x,
                                       Object y) {
                        adjustExponentialStretch();
                    }
                });
        deskTop.add(estStretchControls);
    }


    /**
     * This is called each time a slider in the FalseColorToolbox is moved.
     * This is where the real image processing happens.
     */
    private void adjustColorization() {
        float coeff[] = new float[3];
        InternalImageFrame icf;

        try {
            coeff =
                    falseColorToolbox.getSliderBank()
                    .getValues();

            icf =
                    (InternalImageFrame)
                    deskTop.getTopmostFrame(InternalImageFrame.class);
            fcProcessor.setBaseImage(icf.getBaseImage());
            fcProcessor.setColorization(coeff[0],
                    coeff[1],
                    coeff[2]);
            fcProcessor.processImage();
            icf.setImage(fcProcessor.getProcessedImage());
        } catch (Exception e) {
            // for now, do nothing.
        }
    }

    /**
     * This is called each time a slider on the LinearMappingFrame is moved
     * (btn = false) or the button on the LinearMappingFrame is pressed
     * (btn = true).  This is where the real image processing happens.
     */
    private void adjustLinearMapping(boolean btn) {
        float coeff[] = new float[2];
        InternalImageFrame icf;

        try {
            icf =
                    (InternalImageFrame) deskTop.getTopmostFrame(InternalImageFrame.class);
            lmProcessor.setBaseImage(icf.getBaseImage());
            if (btn) {
// compute parameters from the image, then update the toolbox sliders
                coeff =
                        lmProcessor.setOptimalParameters();
                linearMappingFrame.getSliderBank()
                        .setValues(coeff);
            } else {
// get the parameters from the sliders, then update the image
                coeff =
                        linearMappingFrame.getSliderBank()
                        .getValues();
                lmProcessor.setParameters(coeff[0], coeff[1]);
            }
            lmProcessor.processImage();
            icf.setImage(lmProcessor.getProcessedImage());
        } catch (Exception e) {
            // for now, do nothing.
        }
    }

    /**
     * This is called when the slider on the HistogramEQToolbox is moved,
     * or when its button is pressed.
     */
    private void adjustHistogram() {
        boolean exponential;
        float alpha;
        InternalImageFrame icf;

        try {
            icf =
                    (InternalImageFrame) deskTop.getTopmostFrame(InternalImageFrame.class);
            heProcessor.setBaseImage(icf.getBaseImage());

            exponential =
                    histEqToolbox.getExponential();
            heProcessor.setExponential(exponential);
            if (exponential) {
                alpha =
                        histEqToolbox.getSlider()
                        .getValue();
                heProcessor.setAlpha(alpha);
            }

            heProcessor.processImage();
            icf.setImage(heProcessor.getProcessedImage());
        } catch (Exception e) {
            // for now, do nothing.
        }
    }

    /**
     * This is called each time a slider in the ExponentialStretchToolbox
     * is moved. This is where the real image processing happens.
     */
    private void adjustExponentialStretch() {
        float coeff;
        InternalImageFrame icf;
        try {
            coeff =
                    estStretchControls.getSlider()
                    .getValue();

            icf =
                    (InternalImageFrame) deskTop.getTopmostFrame(InternalImageFrame.class);
            esProcessor.setBaseImage(icf.getBaseImage());
            esProcessor.setPower(coeff);
            esProcessor.processImage();
            icf.setImage(esProcessor.getProcessedImage());
        } catch (Exception e) {
            // for now, do nothing.
        }
    }

    /**
     * Entry point of the ImageTool application.
     */
    public static void main(String args[]) {
        Main fc = new Main();
        fc.setSize(500, 400);
        fc.show();
    }

    public void setBaseImage(Image image) {
        InternalImageFrame icf = getChildFrame();
        icf.setImage(image);
    }

    public void apply() {
        InternalImageFrame icf = getChildFrame();
        icf.apply();

    }
}