package j2d;

import ip.transforms.Kernels;
import math.Random;

import java.awt.*;
import java.io.*;

import utils.UByte;
import j2d.hpp.InvertFilter;
import j2d.hpp.HppFilterInterface;

/**
 * GPL code by DocJava, Inc. User: lyon Date: Mar 5, 2003 Time: 5:44:03 PM
 */
public class ShortImageBean implements Serializable {
    private short r[][];
    private short g[][];
    private short b[][];
    private Clipper clipper = new Clipper();

    public ShortImageBean() {
    }

    public void swapGreenAndBlue() {
        short temp[][] = getG();
        setG(getB());
        setB(temp);
    }

    public float getTotalNoisePower(ShortImageBean sib) {
        float sum = 0;
        float error = 0;
        for (int x = 0; x < getWidth(); x++)
            for (int y = 0; y < getHeight(); y++) {
                final int dr = r[x][y] - sib.getR()[x][y];
                final int dg = g[x][y] - sib.getG()[x][y];
                final int db = b[x][y] - sib.getB()[x][y];
                error = Math.abs(dr) + Math.abs(dg) + Math.abs(db);
                sum = sum + error;
            }
        return sum;
    }

    public float getTotalSignalPower() {
        float sum = 0;
        float signal = 0;
        for (int x = 0; x < getWidth(); x++)
            for (int y = 0; y < getHeight(); y++) {
                signal = r[x][y] +
                        g[x][y] +
                        b[x][y];
                sum = sum + signal * signal;
            }
        return sum;
    }

    public ShortImageBean getImageFlowBean() {
        float ns[][] = Kernels.getRobinson1();
        float ew[][] = Kernels.getRobinson3();
        Image maxImage = getImage();
        ShortImageBean nsb =
                new ShortImageBean(ImageUtils.convolution(maxImage, ns));
        ShortImageBean ewb =
                new ShortImageBean(ImageUtils.convolution(maxImage, ew));
        short imaginaryPart[][] = nsb.getR();
        short realPart[][] = ewb.getR();
        int w = realPart.length;
        int h = realPart[0].length;
        ShortImageBean ifb = new ShortImageBean(w, h);
        short rIfb[][] = ifb.getR();
        int length = 1;
        for (int x = 0; x < w; x++)
            for (int y = 0; y < h; y++) {
                double dy = imaginaryPart[x][y];
                double dx = realPart[x][y];
                // y = mx+b,b=y-m*x
                //double b = y - x* dy/dx ;
                //double x1=x+5;
                //double y1=dy/dx * (x+5) + b;
                dy = dy / dx;
                if ((x % length) != 0) continue;
                if ((y % length) != 0) continue;
                if (x + length >= w) continue;
                if (y + length * dy >= h) continue;
                // use DDA to draw a short line segment
                for (int i = 0; i < length; i++)
                    rIfb[x + i][(int) (y + i * dy)] = 255;
            }
        ifb.copyRedToGreenAndBlue();
        return ifb;
    }

    /**
     * average averages the input short image bean with the given short
     * image bean <code> ShortImageBean sib, sib2; sib.average(sib2); This
     * is like: a = avg(a,b); </code>
     *
     * @param sib
     */
    public void average(ShortImageBean sib) {
        for (int x = 0; x < getWidth(); x++)
            for (int y = 0; y < getHeight(); y++) {
                r[x][y] =
                        (short) ((r[x][y] + sib.getR()[x][y]) / 2);
                g[x][y] =
                        (short) ((g[x][y] + sib.getG()[x][y]) / 2);
                b[x][y] =
                        (short) ((b[x][y] + sib.getB()[x][y]) / 2);
            }
    }

    public ShortImageBean(int w, int h) {
        r = new short[w][h];
        g = new short[w][h];
        b = new short[w][h];
    }

    public static ShortImageBean
            getShortImageBean(ShortImageBean sib) {
        int w = sib.getWidth();
        int h = sib.getHeight();
        ShortImageBean nsib = new ShortImageBean(w, h);
        short _r[][] = nsib.getR();
        short _g[][] = nsib.getG();
        short _b[][] = nsib.getB();
        for (int x = 0; x < w; x++)
            for (int y = 0; y < h; y++) {
                _r[x][y] = sib.getR()[x][y];
                _g[x][y] = sib.getG()[x][y];
                _b[x][y] = sib.getB()[x][y];
            }
        return nsib;
    }

    public Clipper getClipper() {
        return clipper;
    }

    public int getWidth() {
        return r.length;
    }

    public int getHeight() {
        return r[0].length;
    }

    public ShortImageBean(Image img) {
        setImage(img);
    }

    public Image getImage() {
        return ImageUtils.getImage(r, g, b);
    }

    public void setImage(Image img) {
        final Frame f = new Frame();
        ImageUtils.waitForImage(f, img);
        int w = img.getWidth(f);
        int h = img.getHeight(f);
        if (w == -1) return;
        short[][] r1 = new short[w][h];
        r = r1;
        short[][] g1 = new short[w][h];
        g = g1;
        short[][] b1 = new short[w][h];
        b = b1;
        ImageUtils.pelsToShort(r, g, b,
                ImageUtils.getPels(img,
                        w,
                        h),
                w, h);
    }

    /**
     * convert the image to gray scale by taking the average of the red,
     * green and blue colors. It is silly, but fast.
     */
    public void gray() {
        for (int x = 0; x < getWidth(); x++)
            for (int y = 0; y < getHeight(); y++) {
                r[x][y] = (short)
                        ((r[x][y] + g[x][y] + b[x][y]) / 3);
                g[x][y] = r[x][y];
                b[x][y] = r[x][y];
            }
    }

    /*
   public void colorToRed2() {
       for (int x = 0; x < getImageWidth(); x++)
           for (int y = 0; y < getImageHeight(); y++)
               getR()[x][y] = (short)
                       ((getR()[x][y] + getG()[x][y] + getB()[x][y]) / 3);
   }
   */
    /**
     * Copy all colors to red plane, using an average operation. <code>
     * ShortImageBean.colorToRed(sib); r = (r+g+b)/3;
     * <p/>
     * </code>
     *
     * @param sib
     */
    public static void colorToRed(ShortImageBean sib) {
        for (int x = 0; x < sib.getWidth(); x++)
            for (int y = 0; y < sib.getHeight(); y++)
                sib.r[x][y] = (short)
                        ((sib.r[x][y] +
                        sib.g[x][y] +
                        sib.b[x][y]) / 3);
    }

    public static void clip(ShortImageBean sib) {
        int w = sib.getWidth();
        int h = sib.getHeight();
        for (int x = 0; x < w; x++)
            for (int y = 0; y < h; y++) {
                if (sib.r[x][y] > 255) sib.r[x][y] = 255;
                if (sib.g[x][y] > 255) sib.g[x][y] = 255;
                if (sib.b[x][y] > 255) sib.b[x][y] = 255;
                if (sib.r[x][y] < 0) sib.r[x][y] = 0;
                if (sib.g[x][y] < 0) sib.g[x][y] = 0;
                if (sib.b[x][y] < 0) sib.b[x][y] = 0;
            }
    }

    public void saltAndPepper(int n) {
        for (int i = 0; i < n; i++) {
            int rx = Random.rand(0, getWidth() - 1);
            int ry = Random.rand(0, getHeight() - 1);
            r[rx][ry] = 255;
            g[rx][ry] = 255;
            b[rx][ry] = 255;
            rx = Random.rand(0, getWidth() - 1);
            ry = Random.rand(0, getHeight() - 1);
            r[rx][ry] = 0;
            g[rx][ry] = 0;
            b[rx][ry] = 0;
        }
    }

    public void add(int n) {
        for (int x = 0; x < getWidth(); x++)
            for (int y = 0; y < getHeight(); y++) {
                r[x][y] = (short) (r[x][y] + n);
                g[x][y] = (short) (g[x][y] + n);
                b[x][y] = (short) (b[x][y] + n);
            }
    }

    public void negate() {
        InvertFilter invFilter = new InvertFilter();
        process(invFilter);
    }

    public void process(HppFilterInterface hppFilter) {
        for (int x = 0; x < getWidth(); x++)
            for (int y = 0; y < getHeight(); y++) {

                r[x][y] = hppFilter.getR(r[x][y]);
                g[x][y] = hppFilter.getG(g[x][y]);
                b[x][y] = hppFilter.getB(b[x][y]);
            }
        clip(this);
    }

    public void negate2() {
        for (int x = 0; x < getWidth(); x++)
            for (int y = 0; y < getHeight(); y++) {
                r[x][y] = (short) (255 - r[x][y]);
                g[x][y] = (short) (255 - g[x][y]);
                b[x][y] = (short) (255 - b[x][y]);
            }
        clip(this);
    }

    public void copyRedToGreenAndBlue() {
        g = new short[getWidth()][getHeight()];
        b = new short[getWidth()][getHeight()];
        for (int x = 0; x < getWidth(); x++)
            for (int y = 0; y < getHeight(); y++) {
                g[x][y] = r[x][y];
                b[x][y] = r[x][y];
            }
    }

    public static void subtract(ShortImageBean sibA, ShortImageBean sibB) {
        for (int i = 0; i < sibB.getWidth(); i++)
            for (int j = 0; j < sibB.getHeight(); j++) {
                sibA.r[i][j] = (short) (sibA.r[i][j]
                        - sibB.r[i][j]);
                sibA.g[i][j] = (short) (sibA.g[i][j]
                        - sibB.g[i][j]);
                sibA.b[i][j] = (short) (sibA.b[i][j]
                        - sibB.b[i][j]);
            }
    }

    public int getAverage(int x, int y) {
        return (r[x][y] + g[x][y] + b[x][y]) / 3;
    }

    public short[][] getR() {
        return r;
    }

    public void setR(short[][] r) {
        this.r = r;
    }

    public short[][] getG() {
        return g;
    }

    public void setG(short[][] g) {
        this.g = g;
    }

    public short[][] getB() {
        return b;
    }

    public void setB(short[][] b) {
        this.b = b;
    }

    public int[] getPels() {
        final int w = getWidth();
        final int h = getHeight();
        int pels[] = new int[w * h];

        for (int x = 0; x < w; x++) {

            for (int y = 0; y < h; y++) {
                pels[x + y * w] =
                        0xff000000
                        |
                        (getR()[x][y] << 16)
                        |
                        (getG()[x][y] << 8)
                        | getB()[x][y];
            }
        }
        return pels;
    }

    public double getSNRinDb(ShortImageBean sib) {
        if (sib == null) {
            System.out.println("sib=null!");
            return 0;
        }
// Math.log is a natural log
// but want a common log.
// To Convert to a base 10 log divide by the ln(10).
        final float totalSignalPower = getTotalSignalPower();
        final float totalNoisePower = getTotalNoisePower(sib);
        return 10 * Math.log(totalSignalPower / totalNoisePower) / Math.log(10);
    }

    public static void linearCut(short a[][], int numberOfBitsToCut) {
        int mask = 255 << numberOfBitsToCut;
        int w = a.length;
        int h = a[0].length;
        for (int x = 0; x < w; x++)
            for (int y = 0; y < h; y++)
                a[x][y] = (short) (a[x][y] & mask);
    }

    public void scale(int scale) {
        final int width = getWidth();
        int nw = width * scale;
        final int height = getHeight();
        int nh = height * scale;
        short sr[][] = new short[nw][nh];
        short sg[][] = new short[nw][nh];
        short sb[][] = new short[nw][nh];
        for (int y = 0; y < height; y++)
            for (int j = 0; j < nw; j++)
                for (int x = 0; x < width; x++)
                    for (int i = 0; i < nh; i++) {
                        sr[i][j] = getR()[x][y];
                        sg[i][j] = getG()[x][y];
                        sb[i][j] = getB()[x][y];
                    }
        setR(sr);
        setG(sg);
        setB(sb);
    }


    public static ShortImageBean readPpm(File fn)
            throws IOException {
        InputStream in = new FileInputStream(fn);
        char c1, c2;

        c1 = (char) readByte(in);
        c2 = (char) readByte(in);

        if (c1 != 'P') {
            throw new IOException("not a PPM file");
        }
        if (c2 != '6') {
            throw new IOException("not a PPM file");
        }
        final int w = readInt(in);
        final int h = readInt(in);

// Read maximum value of each color, and ignore it.
// In PPM_RAW we know r,g,b use full range (0-255).
        readInt(in);
        ShortImageBean sib = new ShortImageBean(w, h);
        short r[][] = sib.getR();
        short g[][] = sib.getG();
        short b[][] = sib.getB();
        byte buf[] = new byte[w * h * 3];
        int offset = 0;
        int count = buf.length;
        in.read(buf, offset, count);
        int j = 0;
        for (int col = 0; col < h; col++)
            for (int row = 0; row < w; row++) {
                r[row][col] = UByte.us(buf[j++]);
                g[row][col] = UByte.us(buf[j++]);
                b[row][col] = UByte.us(buf[j++]);
            }

        return sib;
    }

    public static int readInt(InputStream in)
            throws IOException {
        char c;
        int i;

        c = readNonwhiteChar(in);
        if ((c < '0') || (c > '9')) {
            throw new IOException("Invalid integer when reading PPM image file.");
        }

        i = 0;
        do {
            i = i * 10 + c - '0';
            c = readChar(in);
        } while ((c >= '0') && (c <= '9'));

        return (i);
    }

    public static int readByte(InputStream in)
            throws IOException {
        int b = in.read();

// if end of file
        if (b == -1) {
            throw new EOFException();
        }
        return b;
    }

    public static char readNonwhiteChar(InputStream in)
            throws IOException {
        char c;

        do {
            c = readChar(in);
        } while ((c == ' ') || (c == '\t') || (c == '\n') || (c == '\r'));

        return c;
    }

    public static char readChar(InputStream in)
            throws IOException {
        char c;

        c = (char) readByte(in);
        if (c == '#') {
            do {
                c = (char) readByte(in);
            } while ((c != '\n') && (c != '\r'));
        }

        return (c);
    }

    public Color getColor(int x, int y) {
        return new Color(r[x][y],g[x][y],b[x][y]);
    }
}