/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */
package gui.run;

// Fig. 25.33: SliderFieldPanel.java
// A subclass of JPanel containing a JSlider and a JTextField


import gui.ClosableJFrame;
import gui.layouts.DialogLayout;

import javax.swing.*;
import java.awt.*;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.geom.AffineTransform;

/**
 * Only works for integers. Shows a textfield and a slider, as well as a
 * label. Shortcut mnemonics are used in the label to focus onto the
 * textfield.
 *
 * @see RunSlider
 */
public abstract class RunAffinePanel
        extends JTabbedPane implements Runnable,
        MouseMotionListener,
        MouseListener {

    private AffineTransform at = new AffineTransform();
    private SpinnerNumberModel rotationNm = new SpinnerNumberModel(0,
            -180,
            180,
            1);
    private SpinnerNumberModel xcNm = new SpinnerNumberModel(0,
            0,
            1024,
            1);
    private SpinnerNumberModel ycNm = new SpinnerNumberModel(0,
            0,
            1024,
            1);
    private SpinnerNumberModel xtNm = new SpinnerNumberModel(0,
            0,
            1024,
            1);
    private SpinnerNumberModel ytNm = new SpinnerNumberModel(0,
            0,
            1024,
            1);
    private SpinnerNumberModel xsNm = new SpinnerNumberModel(100,
            0,
            200,
            1);
    private SpinnerNumberModel ysNm = new SpinnerNumberModel(100,
            0,
            200,
            1);
    private SpinnerNumberModel xshNm = new SpinnerNumberModel(1,
            -200,
            200,
            0);
    private SpinnerNumberModel yshNm = new SpinnerNumberModel(1,
            -200,
            200,
            1);
    private int operationOrderCode = 0;

    public RunAffinePanel() {
        setToolTipText("A Fine Panel!");
        addTab("Rotate",
                null,
                getRotationPanel(),
                "Rotate");
        addTab("Shear",
                null,
                getShearPanel(),
                "Shear");
        addTab("Scale",
                null,
                getScalePanel(),
                "Scale");
        addTab("CenterPoint",
                null,
                getCenterPointPanel(),
                "CenterPoint");
        addTab("Order",
                null,
                getOrderPanel(),
                "Order");
        addTab("translate",
                null,
                getTranslatePanel(),
                "translate");
    }

    private JPanel getOrderPanel() {
        String items[] = {
            "scale shear rotate",
            "scale rotate shear",
            "shear scale rotate",
            "shear rotate scale",
            "rotate shear scale",
            "rotate scale shear"
        };
        JPanel jp = new JPanel();
        jp.add(new JLabel("select order of operations"),
                BorderLayout.NORTH);
        jp.add(new RunComboBox(items) {
            public void run() {
                operationOrderCode = getValue();
            }
        },
                BorderLayout.CENTER);
        return
                jp;
    }

    public void setCenterPoint(int xc, int yc) {
        xcNm.setValue(new Integer(xc));
        ycNm.setValue(new Integer(yc));
    }

    public void rmoveCenterPoint(int dx, int dy) {

        Integer x = (Integer)xcNm.getValue();
        Integer y = (Integer)ycNm.getValue();
        int xc = x.intValue();
        int yc = y.intValue();
        xcNm.setValue(new Integer(xc + dx));
        ycNm.setValue(new Integer(yc + dy));
    }

    public void setTranslation(int x, int y) {
        xtNm.setValue(new Integer(x));
        ytNm.setValue(new Integer(y));
    }

    public void setScale(int x, int y) {

        xsNm.setValue(new Integer(x));
        ysNm.setValue(new Integer(y));
    }

    public void setRotation(int x, int y) {
        double r = Math.atan2(y, x) * 180 / Math.PI;
        rotationNm.setValue(new Integer((int) r));
    }

    private JPanel getCenterPointPanel() {
        JPanel jp = new JPanel();

        jp.setLayout(new DialogLayout());
        jp.add(new JLabel("xc"));

        jp.add(new RunSpinnerSlider(xcNm) {
            public void run() {
                xcNm.setValue(new Integer(getValue()));
                RunAffinePanel.this.run();
            }
        });
        jp.add(new JLabel("yc"));

        jp.add(new RunSpinnerSlider(ycNm) {
            public void run() {
                ycNm.setValue(new Integer(getValue()));
                RunAffinePanel.this.run();
            }
        });

        return jp;
    }

    private JPanel getTranslatePanel() {
        JPanel jp = new JPanel();

        jp.setLayout(new DialogLayout());
        jp.add(new JLabel("xt"));

        jp.add(new RunSpinnerSlider(xtNm) {
            public void run() {
                xtNm.setValue(new Integer(getValue()));
                RunAffinePanel.this.run();
            }
        });
        jp.add(new JLabel("yt"));

        jp.add(new RunSpinnerSlider(ytNm) {
            public void run() {
                ytNm.setValue(new Integer(getValue()));
                RunAffinePanel.this.run();
            }
        });

        return jp;
    }

    private JPanel getScalePanel() {
        JPanel jp = new JPanel();

        jp.setLayout(new DialogLayout());
        jp.add(new JLabel("scx"));

        jp.add(new RunSpinnerSlider(xsNm) {
            public void run() {
                xsNm.setValue(new Integer(getValue()));
                RunAffinePanel.this.run();
            }
        });
        jp.add(new JLabel("scy"));

        jp.add(new RunSpinnerSlider(ysNm) {
            public void run() {
                ysNm.setValue(new Integer(getValue()));
                RunAffinePanel.this.run();
            }
        });

        return jp;
    }

    private JPanel getShearPanel() {
        JPanel jp = new JPanel();

        jp.setLayout(new DialogLayout());
        jp.add(new JLabel("shx"));
        jp.add(new RunSpinnerSlider(xshNm) {
            public void run() {
                xshNm.setValue(new Integer(getValue()));
                RunAffinePanel.this.run();
            }
        });
        jp.add(new JLabel("shy"));

        jp.add(new RunSpinnerSlider(yshNm) {
            public void run() {
                yshNm.setValue(new Integer(getValue()));
                RunAffinePanel.this.run();
            }
        });
        return jp;
    }

    private JPanel getRotationPanel() {
        JPanel jp = new JPanel();
        jp.setLayout(new DialogLayout());
        jp.add(new JLabel("rotation"));
        jp.add(new RunSpinnerSlider(rotationNm) {
            public void run() {
                rotationNm.setValue(new Integer(getValue()));
                RunAffinePanel.this.run();
            }
        });
        return jp;
    }

    private static final int getValue(SpinnerNumberModel nm) {
        return ((Integer) nm.getValue()).intValue();
    }

    public AffineTransform getValue() {
        /*
        String items[] = {
        "scaleShearRotate",0
        "scaleRotateShear",1
        "shearScaleRotate",2
        "shearRotateScale",3
        "rotateShearScale",4
        "rotateScaleShear" 5
        };
        */

        switch (operationOrderCode) {

            case 0:
                return scaleShearRotate();
            case 1:
                return scaleRotateShear();
            case 2:
                return shearScaleRotate();
            case 3:
                return shearRotateScale();
            case 4:
                return rotateShearScale();
            case 5:
                return rotateScaleShear();
        }
        return scaleShearRotate();
    }

    private void rotate(AffineTransform at) {
        final int r = getValue(rotationNm);
        at.rotate(r * Math.PI / 180.0);
    }

    private void scale(AffineTransform at) {
        at.scale(getValue(xsNm) / 100.0, getValue(ysNm) / 100.0);
    }

    private void setToTranslation(AffineTransform at) {
        at.setToTranslation(getValue(xcNm), getValue(ycNm));
    }

    private void translateBack(AffineTransform at) {
        at.translate(-getValue(xcNm), -getValue(ycNm));
        at.translate(getValue(xtNm), getValue(ytNm));
    }

    private void shear(AffineTransform at) {
        at.shear(getValue(xshNm) / 100.0, getValue(yshNm) / 100.0);
    }

    private AffineTransform shearScaleRotate() {
        setToTranslation(at);
        shear(at);
        scale(at);
        rotate(at);
        translateBack(at);
        return at;
    }

    private AffineTransform shearRotateScale() {
        setToTranslation(at);
        rotate(at);
        shear(at);
        scale(at);
        translateBack(at);
        return at;
    }

    private AffineTransform rotateShearScale() {
        setToTranslation(at);
        rotate(at);
        shear(at);
        scale(at);
        translateBack(at);
        return at;
    }

    private AffineTransform rotateScaleShear() {
        setToTranslation(at);
        rotate(at);
        scale(at);
        shear(at);
        translateBack(at);
        return at;
    }

    private AffineTransform scaleRotateShear() {
        setToTranslation(at);
        scale(at);
        rotate(at);
        shear(at);
        translateBack(at);
        return at;
    }

    private AffineTransform scaleShearRotate() {
        setToTranslation(at);
        scale(at);
        shear(at);
        rotate(at);
        translateBack(at);
        return at;
    }


    public static void main(String[] args) {
        ClosableJFrame cf = new ClosableJFrame();
        Container c = cf.getContentPane();
        c.setLayout(new FlowLayout());
        c.add(new RunAffinePanel() {
            public void run() {
                AffineTransform at = getValue();
                System.out.println(at);
            }
        });

        cf.setSize(400, 400);
        cf.show();
    }

    public void mouseClicked(MouseEvent e) {
        this.setCenterPoint(e.getX(), e.getY());
    }

    public void mouseEntered(MouseEvent e) {
    }

    public void mouseExited(MouseEvent e) {
    }

    public void mouseReleased(MouseEvent e) {

    }

    public void mousePressed(MouseEvent e) {

    }

    public void mouseDragged(MouseEvent e) {
        int m = e.getModifiers();
        int xc = getValue(xcNm);
        int yc = getValue(ycNm);

        final int dx = (int) (e.getX() - xc);
        final int dy = (int) (e.getY() - yc);

        if ((m & (Event.SHIFT_MASK)) != 0) {
            setScale(dx, dy);
        } else if ((m & (Event.ALT_MASK)) != 0) {
            setRotation(dx, dy);
        } else {
            setTranslation(dx, dy);
        }
    }

    public void mouseMoved(MouseEvent e) {

    }

}
