package futils;

import utils.Base64;
import utils.StringUtils;
import utils.CompactJava;
import xml.Utils;

import java.io.File;
import java.io.Serializable;
import java.util.Vector;

/**
 * DocJava, Inc.
 * http://www.docjava.com
 * Programmer: dlyon
 * Date: Sep 28, 2004
 * Time: 2:25:21 PM
 * @author D. Lyon
 */
public class DirCache implements Serializable {
    private Vector fileVector = new Vector();
    private File rootDirectory;

    public FileStore[] getFileStoreArray() {
        FileStore fs [] = new FileStore[fileVector.size()];
        fileVector.copyInto(fs);
        return fs;
    }

    public DirCache(File rootDirectory, File fa[]) {
        this.rootDirectory = rootDirectory;
        for (int i = 0; i < fa.length; i++) {
            File f = fa[i];
            if (f.isDirectory()) continue;
            fileVector.addElement(new FileStore(f));
        }
    }

    public static void main(String[] args) {
        //encodeTest();
        decodeTest();
    }

    private static void encodeTest() {
        DirList dl = new DirList("");
        File startDir = dl.getStartDir();
        DirCache dc = new DirCache(startDir, dl.getFiles());
        System.out.println(CompactJava.getCompactJava(dc, startDir.getName()));
        System.out.println("DirCache dc = (DirCache) o;");
    }

    /**
     * Save the <code>DirCache</code> to a directory selected by the user
     *
     */
    public void save() {
        File f = Futil.getReadDirFileJTree("please select an output directory");
        save(f);
    }

    /**
     * Save the <code>DirCache</code>to the <code>newRootDirectory</code>
     * @param newRootDirectory  directory used to replace old root directory
     */
    public void save(File newRootDirectory) {
        FileStore fs[] = getFileStoreArray();
        for (int i = 0; i < fs.length; i++) {
            String newFileName =
                    StringUtils.replaceFirstInstance(
                            fs[i].getFileName(),
                            rootDirectory.getAbsolutePath(),
                            newRootDirectory.getAbsolutePath());
            byte b[] = fs[i].getFileData();
            Futil.writeBytes(new File(newFileName),b);
        }
    }
    /**
     * Use the base-64 encoded gzip compressed string to
     * create a dir cache instance and save it to a file.
     */
    private static void decodeTest() {
        String addressName = "address";
        String address =
                "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";
        Object o = Base64.decodeToObject(address);
        DirCache dc = (DirCache) o;
        dc.save();
    }
}
