package classUtils.resolver.gui;

import classUtils.resolver.interfaces.Auditor;
import classUtils.resolver.interfaces.WalkObserver;
import classUtils.resolver.model.AuditMaster;
import classUtils.resolver.model.AuditableJarFile;
import classUtils.resolver.util.WalkObserverImpl;

import javax.swing.*;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeSelectionModel;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.awt.*;

/**
 * @author Douglas Tillman, VTE Consulting, 12/2002
 * @version 2.0 http://home.attbi.com/~doug.tillman/index.html Builds and
 *          displays Gui fo .jar file auditing utility. The utility scans
 *          the directory provided as a command line argument and displays
 *          all of the .jar files in the top of a split pane GUI and in the
 *          bottom pane all of the duplicates are listed.
 */
public class JarPanel extends JPanel {

    private String lineStyle = "Angled";
    private File root[];

    public JarPanel(File[] rootDirPath) throws IOException {
        //root directory for start of recursive inventory of .jar files
        root = rootDirPath;

        //Create the nodes.
        DefaultMutableTreeNode top =
                new DefaultMutableTreeNode("Jar File Inventory");
        //Create same for duplicates
        DefaultMutableTreeNode duptop =
                new DefaultMutableTreeNode("Duplicate Jar Files");
        createNodes(top, duptop);

        //Create a tree that allows one selection at a time.
        final JTree tree = new JTree(top);

        tree.getSelectionModel().setSelectionMode(
                TreeSelectionModel.SINGLE_TREE_SELECTION);

        tree.putClientProperty("JTree.lineStyle", lineStyle);
        add(tree);
        setLayout(new GridLayout(1,0));

    }

    /**
     * Creates the Tree and Duplicates Tree nodes
     *
     * @param top that serves as top level node of
     *                               master inventory tree
     * @param duptop that serves as top level node of
     *                               duplicates tree
     */
    private void createNodes(DefaultMutableTreeNode top,
                             DefaultMutableTreeNode duptop)
            throws IOException {

        //declare nodes used to populate JTree

        DefaultMutableTreeNode jar = null;
        DefaultMutableTreeNode jarFilePath = null;
        DefaultMutableTreeNode classFile = null;

        //objects for walking directory and callback functionality as .jar
        //files are encountered.
        Auditor auditMaster = new AuditMaster();
        WalkObserver woImpl = new WalkObserverImpl(auditMaster);
        for (int i = 0; i < root.length; i++)
            walk(root[i], woImpl, 0);

        //loop through collection of values from TreeMap of found .jar's
        Collection jarnames = auditMaster.getMasterJarList().values();
        Iterator iter = jarnames.iterator();
        while (iter.hasNext()) {
            AuditableJarFile jf = (AuditableJarFile) iter.next();

            jar = new DefaultMutableTreeNode(jf.getFile().getName());
            top.add(jar);

            jarFilePath = new DefaultMutableTreeNode(
                    jf.getFile().getAbsolutePath());

            jar.add(jarFilePath);

            ArrayList a = jf.getExplodedContents();
            for (int x = 0; x < a.size(); x++) {
                classFile = new DefaultMutableTreeNode((String) a.get(x));
                jarFilePath.add(classFile);
            }
        }


        DefaultMutableTreeNode dupJar = null;
        DefaultMutableTreeNode dupJarFilePath = null;
        DefaultMutableTreeNode dupClassFile = null;

        //loop through duplicates
        //loop through collection of values from TreeMap of duplicates
        Collection dupjarnames = auditMaster.getDuplicateJarList().values();
        Iterator dupIter = dupjarnames.iterator();
        while (dupIter.hasNext()) {
            AuditableJarFile dupjf = (AuditableJarFile) dupIter.next();

            dupJar =
                    new DefaultMutableTreeNode(dupjf.getFile().getName());
            duptop.add(dupJar);

            dupJarFilePath = new DefaultMutableTreeNode(
                    dupjf.getFile().getAbsolutePath());

            dupJar.add(dupJarFilePath);

            ArrayList a = dupjf.getExplodedContents();
            for (int x = 0; x < a.size(); x++) {
                dupClassFile =
                        new DefaultMutableTreeNode((String) a.get(x));
                dupJarFilePath.add(dupClassFile);
            }
        }
    }

    /**
     * Walks a directory tree, calling the WalkObserver at each file.
     *
     * @param rootDirectory Root of directory tree to walk
     * @param o             The directory WalkObserver object for
     *                      callbacks
     * @param indentLevel   The current level of recursion
     */
    private void walk(File rootDirectory,
                      WalkObserver o,
                      int indentLevel) throws IOException {

        // Get list of files in directory

        File files[] = getFileArray(rootDirectory);

        // Traverse files
        for (int i = 0; i < files.length; i++) {
            // Call observer callback with file and root dir path
            final File f = files[i];
            o.at(f, indentLevel);

            // If the file is a directory, recurse into it
            if (f.isDirectory())
                walk(f, o, indentLevel + 1);

        }
    }

    private File[] getFileArray(File rootDirectory) throws IOException {
        if (rootDirectory.isFile())
            return new File[]{rootDirectory.getCanonicalFile()};
        return rootDirectory.listFiles();
    }

}
