/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */
package addbk.print.labels;

import javax.swing.table.AbstractTableModel;
import javax.swing.event.TableColumnModelListener;
import javax.swing.event.TableColumnModelEvent;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.*;
import java.text.SimpleDateFormat;
import java.text.NumberFormat;
import java.util.Vector;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Calendar;
import java.sql.*;

public class LabelTableData extends AbstractTableModel
        implements TableColumnModelListener {

    static final public ColumnData m_columns[] = {
        new ColumnData("Symbol", 100, JLabel.LEFT),
        new ColumnData("Name", 160, JLabel.LEFT),
        new ColumnData("Last", 100, JLabel.RIGHT),
        new ColumnData("Open", 100, JLabel.RIGHT),
        new ColumnData("Change", 100, JLabel.RIGHT),
        new ColumnData("Change %", 100, JLabel.RIGHT),
        new ColumnData("Volume", 100, JLabel.RIGHT)
    };

    public static ImageIcon COLUMN_UP = new ImageIcon("SortUp.gif");
    public static ImageIcon COLUMN_DOWN = new ImageIcon("SortDown.gif");

    protected SimpleDateFormat m_frm;
    protected NumberFormat m_volumeFormat;
    protected Vector m_vector;
    protected Date m_date;
    protected int m_columnsCount = m_columns.length;

    public int m_sortCol = 0;
    public boolean m_sortAsc = true;

    public LabelTableData() {
        m_frm = new SimpleDateFormat("MM/dd/yyyy");
        m_volumeFormat = NumberFormat.getInstance();
        m_volumeFormat.setGroupingUsed(true);
        m_volumeFormat.setMaximumFractionDigits(0);

        m_vector = new Vector();
        setDefaultData();
    }

    public void setDefaultData() {
        try {
            m_date = m_frm.parse("12/18/2004");
        } catch (java.text.ParseException ex) {
            m_date = null;
        }

        m_vector.removeAllElements();
        m_vector.addElement(new LabelData("ORCL", "Oracle Corp.",
                23.6875, 25.375, -1.6875, -6.42, 24976600));
        m_vector.addElement(new LabelData("EGGS", "Egghead.com",
                17.25, 17.4375, -0.1875, -1.43, 2146400));
        m_vector.addElement(new LabelData("T", "AT&T",
                65.1875, 66, -0.8125, -0.10, 554000));
        m_vector.addElement(new LabelData("LU", "Lucent Technology",
                64.625, 59.9375, 4.6875, 9.65, 29856300));
        m_vector.addElement(new LabelData("FON",
                "Sprint",
                104.5625,
                106.375,
                -1.8125,
                -1.82,
                1135100));
        m_vector.addElement(new LabelData("ENML", "Enamelon Inc.",
                4.875, 5, -0.125, 0, 35900));
        m_vector.addElement(new LabelData("CPQ", "Compaq Computers",
                30.875, 31.25, -0.375, -2.18, 11853900));
        m_vector.addElement(new LabelData("MSFT", "Microsoft Corp.",
                94.0625, 95.1875, -1.125, -0.92, 19836900));
        m_vector.addElement(new LabelData("DELL", "Dell Computers",
                46.1875, 44.5, 1.6875, 6.24, 47310000));
        m_vector.addElement(new LabelData("SUNW", "Sun Microsystems",
                140.625, 130.9375, 10, 10.625, 17734600));
        m_vector.addElement(new LabelData("IBM", "Intl. Bus. Machines",
                183, 183.125, -0.125, -0.51, 4371400));
        m_vector.addElement(new LabelData("HWP", "Hewlett-Packard",
                70, 71.0625, -1.4375, -2.01, 2410700));
        m_vector.addElement(new LabelData("UIS", "Unisys Corp.",
                28.25, 29, -0.75, -2.59, 2576200));
        m_vector.addElement(new LabelData("SNE", "Sony Corp.",
                96.1875, 95.625, 1.125, 1.18, 330600));
        m_vector.addElement(new LabelData("NOVL", "Novell Inc.",
                24.0625, 24.375, -0.3125, -3.02, 6047900));
        m_vector.addElement(new LabelData("HIT", "Hitachi, Ltd.",
                78.5, 77.625, 0.875, 1.12, 49400));
    }

    public int getRowCount() {
        return m_vector == null ? 0 : m_vector.size();
    }

    public int getColumnCount() {
        return m_columnsCount;
    }

    public String getColumnName(int column) {
        return m_columns[column].m_title;
    }

    public Icon getColumnIcon(int column) {
        if (column == m_sortCol)
            return m_sortAsc ? COLUMN_UP : COLUMN_DOWN;
        return null;
    }

    public boolean isCellEditable(int nRow, int nCol) {
        return false;
    }

    public Object getValueAt(int nRow, int nCol) {
        if (nRow < 0 || nRow >= getRowCount())
            return "";
        LabelData row = (LabelData) m_vector.elementAt(nRow);
        switch (nCol) {
            case 0:
                return row.m_symbol;
            case 1:
                return row.m_name;
            case 2:
                return row.m_last;
            case 3:
                return row.m_open;
            case 6:
                return m_volumeFormat.format(row.m_volume);
        }
        return "";
    }

    public String getTitle() {
        if (m_date == null)
            return "Stock Quotes";
        return "Stock Quotes at " + m_frm.format(m_date);
    }


    static final String QUERY = "SELECT data.symbol, symbols.name, " +
            "data.last, data.open, data.change, data.changeproc, " +
            "data.volume FROM DATA INNER JOIN SYMBOLS " +
            "ON DATA.symbol = SYMBOLS.symbol WHERE " +
            "month(data.date1)=? AND day(data.date1)=?" +
            " AND year(data.date1)=?";


    public void retrieveData(Date date)
            throws SQLException, ClassNotFoundException {

        GregorianCalendar calendar = new GregorianCalendar();
        calendar.setTime(date);
        int month = calendar.get(Calendar.MONTH) + 1;
        int day = calendar.get(Calendar.DAY_OF_MONTH);
        int year = calendar.get(Calendar.YEAR);

        m_date = date;
        m_vector = new Vector();
        Connection conn = null;
        PreparedStatement pst = null;
        try {
            // Load the JDBC-ODBC bridge driver
            Class.forName("sun.jdbc.odbc.JdbcOdbcDriver");
            conn = DriverManager.getConnection("jdbc:odbc:Market", "admin", "");

            pst = conn.prepareStatement(QUERY);
            pst.setInt(1, month);
            pst.setInt(2, day);
            pst.setInt(3, year);
            ResultSet results = pst.executeQuery();

            while (results.next()) {
                String symbol = results.getString(1);
                String name = results.getString(2);
                double last = results.getDouble(3);
                double open = results.getDouble(4);
                double change = results.getDouble(5);
                double changePr = results.getDouble(6);
                long volume = results.getLong(7);
                m_vector.addElement(new LabelData(symbol,
                        name,
                        last,
                        open,
                        change,
                        changePr,
                        volume));
            }

        } finally {
            if (pst != null)
                pst.close();
            if (conn != null)
                conn.close();
        }
    }

    // TableColumnModelListener implementation

    public void columnAdded(TableColumnModelEvent e) {
        m_columnsCount++;
    }

    public void columnRemoved(TableColumnModelEvent e) {
        m_columnsCount--;
        if (m_sortCol >= m_columnsCount)
            m_sortCol = -1;
    }

    public void columnMarginChanged(ChangeEvent e) {
    }

    public void columnMoved(TableColumnModelEvent e) {
    }

    public void columnSelectionChanged(ListSelectionEvent e) {
    }
}
