package sound;

import javax.sound.sampled.*;

/**
 * Created by IntelliJ IDEA.
 * User: Douglas Lyon
 * Date: Dec 20, 2004
 * Time: 6:06:44 PM
 * Copyright DocJava, Inc.
 */
public class SoundUtils {
    public static void main(String[] args) {
        Port p = getMicrophonePort();
        Line.Info inf = p.getLineInfo();
        System.out.println(inf);
        p.addLineListener(getLineListener());
    }

    /**
     * Converts a byte array into an array of samples (a bouble).
     * Each consecutive bytes in the byte array are converted into a double,
     * and becomes the next element in the double array. The size of the
     * returned array is (length/bytesPerDouble).
     * Currently, only 1 byte (8-bit) or 2 bytes (16-bit)
     * samples are supported.
     *
     * @param byteArray      a byte array
     * @param offset         which byte to start from
     * @param length         how many bytes to convert
     * @param bytesPerSample the number of bytes per sample
     * @param signedData     whether the data is signed
     * @return a double array, or <code>null</code> if byteArray is of zero
     *         length
     * @throws java.lang.ArrayIndexOutOfBoundsException
     *
     */
    public static final double[] bytesToSamples(byte[] byteArray,
                                                int offset,
                                                int length,
                                                int bytesPerSample,
                                                boolean signedData)
            throws ArrayIndexOutOfBoundsException {

        if (0 < length && (offset + length) <= byteArray.length) {
            int doubleLength = length / bytesPerSample;
            double[] doubleArray = new double[doubleLength];

            if (bytesPerSample == 2) {
                if (!signedData) {
                    for (int i = offset, j = 0; j < doubleLength; j++) {
                        int temp = (int) byteArray[i++];
                        temp = (temp << 8);
                        temp |= (int) (0x000000FF & byteArray[i++]);
                        doubleArray[j] = (double) temp;
                    }
                } else {
                    for (int i = offset, j = 0; j < doubleLength; j++) {
                        short temp = (short) (byteArray[i++] << 8);
                        temp |= (short) (0x00FF & byteArray[i++]);
                        doubleArray[j] = (double) temp;
                    }
                }
            } else if (bytesPerSample == 1) {
                for (int i = offset; i < doubleLength; i++) {
                    doubleArray[i] = (double) byteArray[i];
                }
            } else {
                throw new Error
                        ("Unsupported bytes per sample: " + bytesPerSample);
            }

            return doubleArray;

        } else {
            throw new ArrayIndexOutOfBoundsException
                    ("offset: " + offset + ", length: " + length
                    + ", array length: " + byteArray.length);
        }
    }


    /**
     * Converts a little-endian byte array into an array of samples (double).
     * Each consecutive bytes of a float are converted into a double, and
     * becomes the next element in the double array. The number of bytes
     * in the double is specified as an argument. The size of
     * the returned array is (data.length/bytesPerSample).
     *
     * @param data           a byte array
     * @param offset         which byte to start from
     * @param length         how many bytes to convert
     * @param bytesPerSample the number of bytes per sample
     * @param signed         whether the data is signed
     * @return a double array, or <code>null</code> if byteArray is of zero
     *         length
     * @throws java.lang.ArrayIndexOutOfBoundsException
     *
     */
    public static final double[] littleEndianBytesToSamples(byte[] data,
                                                            int offset,
                                                            int length,
                                                            int bytesPerSample,
                                                            boolean signed)
            throws ArrayIndexOutOfBoundsException {

        if (0 < length && (offset + length) <= data.length) {
            double[] doubleArray = new double[length / bytesPerSample];

            if (bytesPerSample == 2) {
                if (signed) {
                    for (int i = offset, j = 0; i < length; j++) {
                        short temp = (short) ((0x000000FF & data[i++]) |
                                (data[i++] << 8));
                        doubleArray[j] = (double) temp;
                    }
                } else {
                    for (int i = offset, j = 0; i < length; j++) {
                        int temp = (int) ((0x000000FF & data[i++]) |
                                (data[i++] << 8));
                        doubleArray[j] = (double) temp;
                    }
                }
            } else if (bytesPerSample == 1) {
                for (int i = 0; i < doubleArray.length; i++) {
                    doubleArray[i] = data[i];
                }
            } else {
                throw new Error
                        ("Unsupported bytesPerSample: " + bytesPerSample);
            }

            return doubleArray;

        } else {
            throw new ArrayIndexOutOfBoundsException
                    ("offset: " + offset + ", length: " + length
                    + ", array length: " + data.length);
        }
    }

    private static LineListener getLineListener() {
        return new LineListener() {
            public void update(LineEvent e) {
                System.out.println("event:" + e);
            }
        };
    }

    public static Port getMicrophonePort() {
        if (AudioSystem.isLineSupported(Port.Info.MICROPHONE)) {
            try {
                return (Port) AudioSystem.getLine(Port.Info.MICROPHONE);
            } catch (LineUnavailableException e) {
                e.printStackTrace();

            }
        }
        return null;
    }

    public static void listMixersAndExit() {
        System.out.println("Available Mixers:");
        Mixer.Info[] aInfos = AudioSystem.getMixerInfo();
        for (int i = 0; i < aInfos.length; i++) {
            System.out.println(aInfos[i].getName());
        }
        if (aInfos.length == 0) {
            System.out.println("[No mixers available]");
        }
        System.exit(0);
    }

    public static void listSupportedTargetTypes() {
        String strMessage = "Supported target types:";
        AudioFileFormat.Type[] aTypes = AudioSystem.getAudioFileTypes();
        for (int i = 0; i < aTypes.length; i++) {
            strMessage += " " + aTypes[i].getExtension();
        }
        System.out.println(strMessage);
    }
}
