package net.server.servlets;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ResourceBundle;

/**
 * The FormProcessorServlet class handles the processing of
 * the HTML pages which prompt the user for input for the Form
 * C student evaluation form.
 *
 * @author Robert Lysik
 * @version 1.00
 */

public class FormProcessorServlet extends HttpServlet {

    ResourceBundle rb = ResourceBundle.getBundle("LocalStrings");

    private CourseDataFileReader reader; // Information read in from
    // CSV file

    public FormProcessorServlet() {
        reader = new CourseDataFileReader();
    }

    /**
     * This method handles the coordination of HTML page
     * display and form processing using parameters
     * extracted from the request object which is passed in
     * as a parameter. The value of the status parameter determines
     * which HTML page to display next.
     *
     * The initial state, where status is null, involves display of
     * a user vealidation form which prompts the user for their
     * user id and password.
     *
     * If the user is authorized, they are then prompted for the
     * course and section for which they wish to enter data for Form C.
     *
     * The Form C page is then displayed allowing the user to enter the
     * student assessment values for each category and student in the class.
     *
     * A confirmation page is then displayed to the user, following which, the
     * data is either written to file, or written to a database.
     *
     */
    public void doGet(HttpServletRequest request,
                      HttpServletResponse response)
            throws IOException, ServletException {
        response.setContentType("text/html");
        PrintWriter out = response.getWriter();
        String pageStatus = request.getParameter("status");

        // First entry into doGet method, display user authorization form.
        if (pageStatus == null) {
            UserAuthorizationPage page = new UserAuthorizationPage();

            out.println(page.getHtml());
        }

        // Once the course has been selected, the user may then select
        // the section number for the course
        else if (pageStatus.equals("course_id_selected")) {
            String courseId = request.getParameter("course");

            SelectSectionPage page =
                    new SelectSectionPage(courseId,
                            reader.getSectionIds(courseId));

            out.println(page.getHtml());
        }

        // Using the course number and section number obtained during the
        // preceding steps, display the Form C student evaluation form.
        else if (pageStatus.equals("section_id_selected")) {
            String courseId = request.getParameter("course");
            String sectionId = request.getParameter("section");

            ReviewForm page = new ReviewForm(courseId,
                    sectionId,
                    reader.getCourse(courseId, sectionId),
                    reader.getCourseStudents(courseId, sectionId));

            out.println(page.getHtml());
        }

        // Re-display the information submitted from Form C
        else if (pageStatus.equals("evaluation_complete")) {
            ConfirmationPage page = new ConfirmationPage(request);

            out.println(page.getHtml());
        }

        // Finally, save the data generated during the form C evaluation
        else if (pageStatus.equals("confirmed")) {
            SqlSynthesizer synthesizer = new SqlSynthesizer(request);

            synthesizer.save("c:\\statements.sql");
        }
    }

    /**
     * This method handles the coordination of HTML page
     * display and form processing using parameters
     * extracted from the request object which is passed in
     * as a parameter. The value of the status parameter determines
     * which HTML page to display next. The doPost method is used
     * for processing of the users id an password information
     *
     * The initial state, where status is null, involves display of
     * a user validation form which prompts the user for their
     * user id and password.
     *
     * If the user is authorized, they are then prompted for the
     * course and section for which they wish to enter data for Form C.
     *
     */
    public void doPost(HttpServletRequest request,
                       HttpServletResponse response)
            throws IOException, ServletException {
        response.setContentType("text/html");
        PrintWriter out = response.getWriter();
        String pageStatus = request.getParameter("status");

        if (pageStatus.equals("authorizing")) {
            String userId = request.getParameter("txtUserId");
            String password = request.getParameter("txtPassword");
            boolean userIsAuthorized = false;

            Authorization userAuthorization = new Authorization();
            userIsAuthorized = userAuthorization.verify(userId, password);

            if (!userIsAuthorized)
                out.println("<html><body>You are not authorized for access.</body></html>");

            // If user is authorized to continue, display the select course
            // page
            else {
                SelectCoursePage page =
                        new SelectCoursePage(reader.getCourseIds(),
                                reader.getCourses());

                out.println(page.getHtml());
            }
        }
    }
}



