/*
 * The org.opensourcephysics.numerics package contains numerical methods
 * for the book Simulations in Physics.
 * Copyright (c) 2003  H. Gould, J. Tobochnik, and W. Christian.
 */
package math.numerics;

/**
 * Verlet method ODE solver.
 *
 * The Verlet algorithm is a third order algorithm that uses the acceleration to estimate the
 * final position. Note that the velocity plays no part in the integration of the equations.
 *
 * x(n+1) = 2*x(n) - x(n-1) + a(n)*dt*dt
 * v(n+1) = (x(n+1) - x(n-1))/(2 dt) + a(n)*dt
 *
 * CAUTION! You MUST call the initialize if the state array is changed.
 * The Verlet algorithm is not self-starting.  The current state and a prior state
 * must both be known to advance the solution.  Since the prior state is not known
 * for the initial conditions, a prior state is estimated when the
 * initialize method is invoked.
 *
 * CAUTION! This implementation assumes that the state vector has 2*N + 1 variables.
 * These variables alternate between position and velocity with the last variable being time.
 * That is, the  state vector is ordered as follows:
 *
 * x1, d x1/dt, x2, d x2/dt, x3, d x3/dt ..... xN, d xN/dt, t
 *
 * @author       Wolfgang Christian
 * @version 1.0
 */
public class Verlet implements ODESolver {

  private double   stepSize = 0.1;
  private int      numEqn   = 0;
  private double[] rate;          // rate with orginal positions
  private double[] priorState;    // rate with new positions
  private double[] currentState;  // current state
  private ODE      ode;

  /**
   * Constructs the VelocityVerlet ODESolver for a system of ordinary  differential equations.
   *
   * @param _ode the system of differential equations.
   */
  public Verlet(ODE _ode) {
    ode = _ode;
    initialize(stepSize);
  }

  /**
   * Initializes the ODE solver.
   *
   * Two temporary state arrays and one rate array are allocated.
   * The number of differential equations is determined by invoking getState().length on the ODE.
   *
   * @param _stepSize
   */
  public void initialize(double _stepSize) {
    stepSize = _stepSize;
    double state[] = ode.getState();
    if(state == null) {  // state vector not defined.
      numEqn = 0;
    } else {
      numEqn = state.length;
    }
    numEqn       = state.length;
    rate         = new double[numEqn];
    priorState   = new double[numEqn];
    currentState = new double[numEqn];
    estimatePreviousState(state);
  }

  /**
   * Estimate's the previous state.
   *
   * The Verlet method is not self-starting.
   */
  protected void estimatePreviousState(double[] state){
    ode.getRate(state, rate);  // get the rate at the start
    double[] midstate = new double[numEqn];
    for(int i = 0; i < numEqn; i++) {
      // estimate the state at the midpoint
      midstate[i] = state[i] - stepSize * rate[i] / 2;
    }
    ode.getRate(midstate, rate);  // get the rate at the midpoint
    for(int i = 0; i < numEqn; i++) {
      priorState[i] = state[i] - stepSize * rate[i];
    }
  }

  /**
   * Steps (advances) the differential equations by the stepSize.
   *
   * The ODESolver invokes the ODE's getRate method to obtain the initial state of the system.
   * The ODESolver then advances the solution and copies the new state into the
   * state array at the end of the solution step.
   *
   * @return the step size
   */
  public double step() {
    // state[]: x1, d x1/dt, x2, d x2/dt .... xN, d xN/dt, t
    double[] state = ode.getState();
    System.arraycopy(state, 0, currentState, 0, numEqn);  // save the current state as the prior state
    if(state.length != numEqn) {
      initialize(stepSize);
    }
    ode.getRate(state, rate);  // get the initial rate
    double dt2 = stepSize * stepSize;  // the step size squared
    // increment the positions
    for(int i = 0; i < numEqn - 1; i += 2) {  // even numbers are positions
      state[i] = 2 * state[i] - priorState[i] + dt2 * rate[i + 1];
    }
    // increment the velocities
    for(int i = 1; i < numEqn - 1; i += 2) {  // odd numbers are velocities
      state[i] = (state[i - 1] - priorState[i - 1]) / 2.0 / stepSize;
    }
    state[numEqn - 1] += rate[numEqn - 1] * stepSize;  // increment the independent parameter
    System.arraycopy(currentState, 0, priorState, 0, numEqn);  // save the current state as the prior state
    return stepSize;
  }

  /**
   * Sets the step size.
   *
   * The step size remains fixed in this algorithm.
   * The prior state array is reinitialized.
   *
   * @param _stepSize
   */
  public void setStepSize(double _stepSize) {
    stepSize=_stepSize;
    estimatePreviousState( ode.getState() );
  }

  /**
   * Gets the step size.
   *
   * The stepsize is constant in this algorithm.
   *
   * @return the step size
   */
  public double getStepSize() {
    return stepSize;
  }
}
