package j3d.examples.swing;

import gui.ClosableJFrame;
import gui.run.RunMenu;
import gui.run.RunMenuBar;
import gui.run.RunMenuItem;

import javax.media.j3d.*;
import javax.swing.*;
import javax.vecmath.Color3f;
import javax.vecmath.Point3d;
import java.awt.*;

/**
 * This example displays a Swing based application with a JMenuBar that
 * allows you to switch between displaying a Cube and a Sphere. You can
 * also perform a simple screen capture and start/stop rotation using menu
 * commands.
 */
public class Main {

    private final BranchUtils branchUtils = new BranchUtils();
    private final ViewUtils viewUtils = new ViewUtils();


    /**
     * Constructor. Set the layout algorithm for the panel and initialize
     * the Java 3D rendering system and view side scenegraph.
     */
    public Main(Container c) {

        Locale locale = new Locale(new VirtualUniverse());

        BranchGroup sceneBranchGroup = branchUtils.createSceneBranchGroup();
        sceneBranchGroup.addChild(createBackground());

        ViewPlatform vp = ViewUtils.createViewPlatform();

        locale.addBranchGraph(sceneBranchGroup);
        viewUtils.addViewBranchGroup(locale,
                ViewUtils.createViewBranchGroup(
                viewUtils.getViewTransformGroupArray(), vp));

        viewUtils.createView(vp, c);
    }

    /**
     * Create a Background for the Canvas3D.
     */
    protected Background createBackground() {
        Background back = new Background(new Color3f(0.9f, 0.9f, 0.9f));
        back.setApplicationBounds(createApplicationBounds());
        return back;
    }

    /**
     * Create a Bounds object for the scene.
     */
    protected Bounds createApplicationBounds() {
        return new BoundingSphere(new Point3d(0.0, 0.0, 0.0), 100.0);
    }


    public JMenuBar getMenuBar() {
        RunMenuBar mb = new RunMenuBar();
        mb.add(getFileMenu());
        mb.add(getViewMenu());
        mb.add(getRotateMenu());
        return mb;
    }

    private RunMenu getRotateMenu() {
        RunMenu rm;
        rm = new RunMenu("[Rotate");
        rm.add(new RunMenuItem("[on"){
            public void run(){
                branchUtils.rotator.setEnable(true);
            }
        });
        rm.add(new RunMenuItem("o[ff") {
            public void run() {
                branchUtils.rotator.setEnable(false);
            }
        });
        return rm;
    }

    private RunMenu getViewMenu() {
        RunMenu rm;
        rm = new RunMenu("[View");
        rm.add(new RunMenuItem("[cube{control c}"){
            public void run(){
                branchUtils.removeShape("Sphere");
                branchUtils.sceneBranchGroup.addChild(
                        branchUtils.createCube());
            }
        });
        rm.add(new RunMenuItem("[sphere{control s}") {
            public void run() {
                branchUtils.removeShape("Cube");
                branchUtils.sceneBranchGroup.addChild(
                        branchUtils.createSphere());
            }
        });
        return rm;
    }

    private RunMenu getFileMenu() {
        RunMenu rm =  new RunMenu("[File");
        rm.add(new RunMenuItem("save") {
            public void run() {
                viewUtils.saveImage();
            }
        });
        rm.add(new RunMenuItem("e[xit{control x}") {
            public void run() {
                System.exit(0);
            }
        });
        return rm;
    }


    public static void main(String[] args) {
        ClosableJFrame cf = new ClosableJFrame();
        JPanel jp = new JPanel();
        jp.setLayout(new BorderLayout());
        Main st = new Main(jp);
        cf.setJMenuBar(st.getMenuBar());

        cf.getContentPane().add(jp);
        cf.setSize(550, 550);

        cf.setVisible(true);
    }
}
