/*
 * Copyright (c) 2005 DocJava, Inc. All Rights Reserved.
 */
package j3d.examples.swing;

import javax.media.j3d.*;
import javax.vecmath.Point3d;
import javax.vecmath.Color3f;
import javax.vecmath.Vector3f;

/**
 * Copyright DocJava, inc. User: lyon
 * <p/>
 * Date: Mar 8, 2005
 * <p/>
 * Time: 7:12:58 PM
 */
public class BranchUtils {
    /**
     * the is root of the scenegraph which contains the cube and sphere
     * objects
     */
    protected BranchGroup sceneBranchGroup = null;
    /**
     * a rotation interpolator that automatically rotates the cube and
     * sphere
     */
    protected RotationInterpolator rotator = null;

    /**
     * Create the scene side of the scenegraph
     */
    protected BranchGroup createSceneBranchGroup() {
        // create the root of the scene side scenegraph
        BranchGroup objRoot = new BranchGroup();

        // create a TransformGroup to rotate the objects in the scene
        // set the capability bits on the TransformGroup so that it
        // can be modified at runtime
        TransformGroup objTrans = new TransformGroup();
        objTrans.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        objTrans.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);

        // create a spherical bounding volume
        BoundingSphere bounds = new BoundingSphere(
                new Point3d(0.0, 0.0, 0.0), 100.0);

        // create a 4x4 transformation matrix
        Transform3D yAxis = new Transform3D();

        // create an Alpha interpolator to automatically generate
        // modifications to the rotation component of the transformation matrix
        Alpha rotationAlpha = new Alpha(-1, Alpha.INCREASING_ENABLE,
                0, 0,
                4000, 0, 0,
                0, 0, 0);

        // create a RotationInterpolator behavior to effect the TransformGroup
        rotator =
                new RotationInterpolator(rotationAlpha,
                        objTrans,
                        yAxis,
                        0.0f,
                        (float) Math.PI * 2.0f);

        // set the scheduling bounds on the behavior
        rotator.setSchedulingBounds(bounds);

        // add the behavior to the scenegraph
        objTrans.addChild(rotator);

        // create the BranchGroup which contains the objects
        // we add/remove to and from the scenegraph
        sceneBranchGroup = new BranchGroup();

        // allow the BranchGroup to have children added at runtime
        sceneBranchGroup.setCapability(Group.ALLOW_CHILDREN_EXTEND);
        sceneBranchGroup.setCapability(Group.ALLOW_CHILDREN_READ);
        sceneBranchGroup.setCapability(Group.ALLOW_CHILDREN_WRITE);

        // add both the cube and the sphere to the scenegraph
        sceneBranchGroup.addChild(createCube());
        sceneBranchGroup.addChild(createSphere());

        // create the colors for the lights
        Color3f lColor1 = new Color3f(0.7f, 0.7f, 0.7f);
        Vector3f lDir1 = new Vector3f(-1.0f, -1.0f, -1.0f);
        Color3f alColor = new Color3f(0.2f, 0.2f, 0.2f);

        // create the ambient light
        AmbientLight aLgt = new AmbientLight(alColor);
        aLgt.setInfluencingBounds(bounds);

        // create the directional light
        DirectionalLight lgt1 = new DirectionalLight(lColor1, lDir1);
        lgt1.setInfluencingBounds(bounds);

        // add the lights to the scenegraph
        objRoot.addChild(aLgt);
        objRoot.addChild(lgt1);

        // wire the scenegraph together
        objTrans.addChild(sceneBranchGroup);
        objRoot.addChild(objTrans);

        // return the root of the scene side of the scenegraph
        return objRoot;
    }

    /**
     * Create a BranchGroup that contains a Cube. The user data for the
     * BranchGroup is set so the BranchGroup can be identified.
     */
    protected BranchGroup createCube() {
        BranchGroup bg = new BranchGroup();
        bg.setCapability(BranchGroup.ALLOW_DETACH);
        bg.addChild(new com.sun.j3d.utils.geometry.ColorCube());
        bg.setUserData("Cube");
        return bg;
    }

    /**
     * Create a BranchGroup that contains a Sphere. The user data for the
     * BranchGroup is set so the BranchGroup can be identified.
     */
    protected BranchGroup createSphere() {
        BranchGroup bg = new BranchGroup();
        bg.setCapability(BranchGroup.ALLOW_DETACH);

        Appearance app = new Appearance();
        Color3f objColor = new Color3f(1.0f, 0.7f, 0.8f);
        Color3f black = new Color3f(0.0f, 0.0f, 0.0f);
        app.setMaterial(
                new Material(objColor, black, objColor, black, 80.0f));

        bg.addChild(new com.sun.j3d.utils.geometry.Sphere(1, app));
        bg.setUserData("Sphere");
        return bg;
    }

    /**
     * Removes a BranchGroup from the scene based on user data
     *
     * @param name the user data to look for
     */
    protected void removeShape(String name) {
        try {
            java.util.Enumeration
                    enum1 = sceneBranchGroup.getAllChildren();
            int index = 0;

            while (
                    enum1.hasMoreElements() != false) {
                SceneGraphObject sgObject = (SceneGraphObject)
                        enum1.nextElement();
                Object userData = sgObject.getUserData();

                if (userData instanceof String &&
                        ((String) userData).compareTo(name) == 0) {
                    System.out.println(
                            "Removing: " + sgObject.getUserData());
                    sceneBranchGroup.removeChild(index);
                }

                index++;
            }
        } catch (Exception e) {
            // the scenegraph may not have yet been synchronized...
        }
    }
}
