package gui.run;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import java.awt.*;
import java.util.Dictionary;
import java.util.Enumeration;

public abstract class RunSlider
        extends JSlider implements
        ChangeListener, Runnable {
    private JLabel valueLabel = new JLabel(getValue() + "");
    private String identifier = "";

    public RunSlider(int orientation,
                     int min, int max, int defaultValue) {
        super(orientation, min, max, defaultValue);
        setPaintTicks(true);
        int majorTickSpacking = (max - min) / 4;
        setMajorTickSpacing(majorTickSpacking);
        setMinorTickSpacing(majorTickSpacking / 5);
        setPaintLabels(true);
        addChangeListener(this);
    }

    /**
     * Set all the labels to the same font.
     *
     * @param f
     */
    public void setTickLabelFonts(Font f) {
        Dictionary d = getLabelTable();
        Enumeration e = d.elements();
        while (e.hasMoreElements()) {
            Object o = e.nextElement();
            if (o instanceof JLabel) {
                JLabel jl = (JLabel) o;
                jl.setFont(f);
            }
        }
    }


    public RunSlider() {
        this(javax.swing.JSlider.HORIZONTAL, 0, 100, 50);
    }

    /**
     * Creates a slider using the specified orientation with the
     * range 0 to 100 and an initial value of 50.
     */
    public RunSlider(int orientation) {
        this(orientation, 0, 100, 50);
    }

    /**
     * Creates a horizontal slider using the specified min and max
     * with an initial value equal to the average of the min plus max.
     */
    public RunSlider(int min, int max) {
        this(javax.swing.JSlider.HORIZONTAL, min, max, (max + min) / 2);
    }

    /**
     * Creates a horizontal slider using the specified min, max and value.
     */
    public RunSlider(int min, int max, double defaultValue) {
        this(javax.swing.JSlider.HORIZONTAL, min, max, (int) defaultValue);
    }

    /**
     * Creates a horizontal slider using the specified min, max and value.
     */
    public RunSlider(int min, int max, double defaultValue, int orientation) {
        this(orientation, min, max, (int) defaultValue);
    }

    public RunSlider(double min, double max, double defaultValue) {
        this(javax.swing.JSlider.HORIZONTAL,
                (int) (min * 100), (int) (max * 100),
                (int) (defaultValue * 100));
    }

    public void stateChanged(ChangeEvent ae) {
        valueLabel.setText(identifier + getValue() + "");
        SwingUtilities.invokeLater(this);
    }

    public JLabel getValueLabel() {
        return valueLabel;
    }

    public void setValueLabel(JLabel vl) {
        valueLabel = vl;
    }

    public void setIdentifier(String id) {
        identifier = id;
    }

    public static RunSlider getLabeledSlider(String s, RunSlider rs) {
        s = s + "=";
        JLabel valueLabel = new JLabel(s + rs.getValue());
        rs.setValueLabel(valueLabel);
        rs.setIdentifier(s);
        return rs;
    }

    public static void main(String args[]) {
        gui.ClosableJFrame cf = new gui.ClosableJFrame();
        cf.setSize(200, 200);
        java.awt.Container c = cf.getContentPane();
        c.setLayout(new java.awt.BorderLayout());
        c.add(new PlusMinusSlider(), BorderLayout.NORTH);
        c.add(new RunSlider(RunSlider.HORIZONTAL) {
            public void run() {
                System.out.println("value=" + getValue());
                this.setTickLabelFonts(new Font("times", Font.BOLD, getValue()));
            }
        }, BorderLayout.SOUTH);
        c.add(new RunSlider(RunSlider.VERTICAL) {
            public void run() {
                System.out.println("value=" + getValue());
            }
        }, BorderLayout.EAST);
        c.add(new RunSlider(RunSlider.VERTICAL) {
            public void run() {
                System.out.println("value=" + getValue());
            }
        }, BorderLayout.WEST);
        RunSlider centerSlider = new RunSlider(RunSlider.VERTICAL) {
            public void run() {
                System.out.println("value=" + getValue());
            }
        };
        centerSlider.setToolTipText("dont play in the highway without a car");
        c.add(centerSlider, BorderLayout.CENTER);
        cf.setVisible(true);
    }

    private static class PlusMinusSlider extends RunSlider {
        public PlusMinusSlider() {
            super(1, 100, 50);
            this.setTickLabelFonts(new Font("Times", Font.ITALIC, 10));
        }

        public void run() {
            System.out.println("value=" + getValue() / 100.0);
        }
    }
}