package classUtils.pack.util.xml.configuration;

import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * A base {@link Configurator Configurator}.
 * <p>
 * This class extends SAX's <tt>DefaultHandler</tt> and provides support
 * for:
 * <p>
 * <ul>
 * <li> collecting text (via the {@link #characters(char[], int, int) 
 *      characters()} method);
 * <li> ensuring that {@link #getConfiguredObject() getConfiguredObject()}
 *      throws an exception if parsing hasn't completed yet (via the
 *      {@link #checkCompleted() checkCompleted()} method).
 * </ul>
 * <p>
 * Subclasses which override {@link #endElement(java.lang.String, 
 * java.lang.String, java.lang.String) endElement()} should either invoke
 * this class' implementation or set to <b>true</b> the protected 
 * <b>{@link #completed completed}</b> field when the configured object
 * is complete.
 * 
 * @author Cristiano Sadun
 */
public abstract class BaseConfigurator extends DefaultHandler implements Configurator {

	/**
	 * The buffer collecting text received by {@link #characters(char[], int, int) 
	 * characters()}.
	 */
	protected StringBuffer buffer;
	
	/**
	 * Indicate that the object built from the XML is ready.
	 */
	protected boolean completed;

	/**
	 * Constructor for BaseConfigurator.
	 */
	public BaseConfigurator() {
		this.buffer = new StringBuffer();
		this.completed = false;
	}

	/**
	 * Set {@link #completed completed} to true.
	 * 
	 *
	 */
	public void endElement(
		String namespaceURI,
		String localName,
		String qName) {
		completed = true;
	}

	/**
	 * @see Configurator#getConfiguredObject()
	 */
	public final Object getConfiguredObject() {
		checkCompleted();
		return doGetConfiguredObject();
	}

	/**
	 * Throw an IllegalStateException if {@link #completed completed} is not <b>true</b>.
	 */
	protected void checkCompleted() {
		if (!completed)
			throw new IllegalStateException("The object(s) under configuration are not completed yet (possibly endElement has been overriden without invoking its super implementation)");
	}

	protected abstract Object doGetConfiguredObject();

	/**
	 * @see Configurator#characters(char[], int, int)
	 */
	public void characters(char[] ch, int start, int length)
		throws SAXException {
		buffer.append(ch, start, length);
	}

	/**
	 * Returns the completed.
	 * @return boolean
	 */
	public boolean isCompleted() {
		return completed;
	}

	/**
	 * @see Configurator#getText()
	 */
	public final String getText() {
		return buffer.toString();
	}

}
