package classUtils.pack.util.jdbc;

/**
 * An URL for the Jdbc-Odbc bridge. The <b>database</b> symbol is used for the data source.
 * Only the <b>user</b> and <b>password</b> are supported, but arbitrary ODBC attributes
 * can be passed at construction in a string whose format is 
 * <tt><i>attribute-name</i>=$(<i>attribute-name</i>);</tt>, for example:
 * <p>
 * <tt>JdbcOdbcURL url = new JdbcOdbcURL(&quot;ora123&quot;, &quot;Cachesize=$(cachesize);&quot;)</tt>
 * <p>
 * creates a Jdbc/Odbc URL for an oracle data source with a <tt>Cachesize</tt> attribute,
 * whose value can then be defined by:
 * <p>
 * <tt>url.setAttributeValue("cachesize", "300");</tt>
 * <p>.
 * 
 * @author Cristiano Sadun
 * 
 */
public class JdbcOdbcURL extends JdbcURL {

	/**
	 * Constructor for JdbcOdbcURL.
	 * 
	 * @param database the ODBC data source name
	 * @param uid the user id for connecting
	 * @param password the password for connecting
	 * @param additionalODBCattributes a semicolon-separated
	 *         of ODBC attributes in 
	 *         <tt><i>attribute-name</i>=$(;<i>attribute-name</i>);</tt>
	 *         format, different than <tt>UID</tt> and <tt>PWD</tt>
	 */
	public JdbcOdbcURL(String database, String uid, String pwd, String additionalODBCattributes) {
		super("jdbc:odbc:$<database>;UID=$<user>;PWD=$<password>"+
			   (additionalODBCattributes != null ? ";"+additionalODBCattributes : "")
		);
		if (database!=null) setDatabase(database);
		if (uid!=null) setUser(uid);
		if (pwd!=null) setPassword(pwd);
	}
	
	/**
	 * Constructor for JdbcOdbcURL which defines no uid/password.
	 * 
	 * @param database the ODBC data source name
	 * @param additionalODBCattributes a semicolon-separated
	 *         of ODBC attributes in 
	 *         <tt><i>attribute-name</i>=$(;<i>attribute-name</i>);</tt>
	 *         format, different than <tt>UID</tt> and <tt>PWD</tt>
	 */
	public JdbcOdbcURL(String database, String additionalODBCattributes) {
		this(database, "", "", additionalODBCattributes);
	}
	
	/**
	 * Constructor for JdbcOdbcURL requires post-construction definition of the
	 * data source, and which defines no uid/password.
	 * 
	 * @param additionalODBCattributes a semicolon-separated
	 *         of ODBC attributes in 
	 *         <tt><i>attribute-name</i>=$(;<i>attribute-name</i>);</tt>
	 *         format, different than <tt>UID</tt> and <tt>PWD</tt>
	 */
	public JdbcOdbcURL(String additionalODBCattributes) {
		this(null, null, null, additionalODBCattributes);
	}
	
	/**
	 * Constructor for JdbcOdbcURL requires post-construction definition of the
	 * data source, and which defines no uid/password.
	 */
	public JdbcOdbcURL() {
		this(null, null, null, null);
	}

	/**
	 * @see classUtils.pack.util.jdbc.JdbcURL#setDatabase(String)
	 */
	public void setDatabase(String database) {
		super.setDatabase(database);
	}

	/**
	 * @see classUtils.pack.util.jdbc.JdbcURL#setPassword(String)
	 */
	public void setPassword(String password) {
		super.setPassword(password);
	}

	/**
	 * @see classUtils.pack.util.jdbc.JdbcURL#setUser(String)
	 */
	public void setUser(String username) {
		super.setUser(username);
	}

	/**
	 * Set one of the additional attributes passed at construction.
	 */
	public void setAttributeValue(String attributeName, String value) {
		super.setSymbolValue(attributeName, value);
	}

}
